/*
	File:			Vec4.h

	Function:		Defines a length-4 vector.
					
	Author(s):		Andrew Willmott

 */

#ifndef __Vec4__
#define __Vec4__

#include "VL.h"


// --- Vec4 Class -------------------------------------------------------------


TMPLVec class Vec3;

TMPLVec class Vec4
{
public:

	// Constructors
	
	inline 		Vec4();
	inline 		Vec4(TReal x, TReal y, TReal z, TReal w);	// [x, y, z, w]
	inline 		Vec4(const TVec4 &v);			 		// Copy constructor
	inline 		Vec4(const TVec3 &v, Int w);			// Hom. 3D vector
	inline 		Vec4(ZeroOrOne zeroOrOne);	    				
	
	// Accessor functions
	
	inline Int			Elts() const { return(4); };
	
	inline TReal 		&operator () (Int i);		    // v(1) - Indexing
	inline const TReal	&operator () (Int i) const;		  

	inline TReal 		&operator [] (Int i);		    // v[1] - Has no index 
	inline const TReal	&operator [] (Int i) const;		//        check			  

	inline TReal		*Ref() const;					// Return pointer to data

	// Assignment operators
	
	inline TVec4		&operator =  (const TVec4 &a);	
	inline TVec4		&operator =  (ZeroOrOne k);	    				
	inline TVec4		&operator += (const TVec4 &a);
	inline TVec4		&operator -= (const TVec4 &a);
	inline TVec4		&operator *= (const TVec4 &a);
	inline TVec4		&operator *= (TReal s);
	inline TVec4		&operator /= (const TVec4 &a);
	inline TVec4		&operator /= (TReal s);
	
	// Comparison operators

	Bool				operator == (const TVec4 &a) const;	// v == a ?
	Bool				operator != (const TVec4 &a) const;	// v != a ?

	// Arithmetic operators
		
	inline TVec4		operator + (const TVec4 &a) const;	// v + a
	inline TVec4		operator - (const TVec4 &a) const;	// v - a
	inline TVec4		operator - () const;				// -v
	inline TVec4		operator * (const TVec4 &a) const;	// v * a (vx * ax, ...)
	inline TVec4		operator * (TReal s) const;			// v * s
	inline TVec4		operator / (const TVec4 &a) const;	// v / a (vx / ax, ...)
	inline TVec4		operator / (TReal s) const;			// v / s
	inline TReal		operator dot (const TVec4 &a) const;// v . a	

	// Initialisers
	
	inline void			MakeZero();							// Zero vector
	inline void			MakeUnit(Int i, TReal k = vl_one);	// kI[i]
	inline void			MakeBlock(TReal k = vl_one);		// All-k vector

	// Private...

protected:	
	
	TReal elt[4]; 
};


// --- Vec operators -------------------------------------------------

TMPLVec inline
	TVec4			operator * (TReal s, const TVec4 &v);   // Left mult. by s
TMPLVec TReal		len(const TVec4 &v);                    // || v ||
TMPLVec TReal		sqrlen(const TVec4 &v);                 // v . v
TMPLVec TVec4		norm(const TVec4 &v);                   // v / || v ||
TMPLVec TVec4		cross(const TVec4 &a, const TVec4 &b, const TVec4 &c);
                                                            // 4D cross prod.
TMPLVec TVec3		proj(const TVec4 &v);                   // hom. projection

TMPLVec ostream	&operator << (ostream &s, const TVec4 &v);
TMPLVec istream	&operator >> (istream &s, TVec4 &v);
	

// --- Inlines ----------------------------------------------------------------


#include "Mat3.h"

TMPLVec inline TReal &TVec4::operator () (Int i)
{
	CheckRange(i, 0, 4, "(Vec4::(i)) index out of range");
	
    return(elt[i]);
}

TMPLVec inline const TReal &TVec4::operator () (Int i) const
{
	CheckRange(i, 0, 4, "(Vec4::(i)) index out of range");
	
    return(elt[i]);
}

TMPLVec inline TReal &TVec4::operator [] (Int i)
{
    return(elt[i]);
}

TMPLVec inline const TReal &TVec4::operator [] (Int i) const
{
    return(elt[i]);
}


TMPLVec inline TVec4::Vec4()
{
}

TMPLVec inline TVec4::Vec4(TReal x, TReal y, TReal z, TReal w)
{
	elt[0] = x;
	elt[1] = y;
	elt[2] = z;
	elt[3] = w;
}

TMPLVec inline TVec4::Vec4(const TVec4 &v) 
{
	elt[0] = v[0];
	elt[1] = v[1];
	elt[2] = v[2];
	elt[3] = v[3];
}

TMPLVec inline TVec4::Vec4(const TVec3 &v, Int w)  	
{
	elt[0] = v[0];
	elt[1] = v[1];
	elt[2] = v[2];
	elt[3] = w;
}


TMPLVec inline TReal *TVec4::Ref() const
{
	return((TReal *) elt);
}

TMPLVec inline TVec4 &TVec4::operator = (const TVec4 &v)
{
	elt[0] = v[0];
	elt[1] = v[1];
	elt[2] = v[2];
	elt[3] = v[3];
	
	return(SELF);
}

TMPLVec inline TVec4 &TVec4::operator += (const TVec4 &v)
{
	elt[0] += v[0];
	elt[1] += v[1];
	elt[2] += v[2];
	elt[3] += v[3];
	
	return(SELF);
}

TMPLVec inline TVec4 &TVec4::operator -= (const TVec4 &v)
{
	elt[0] -= v[0];
	elt[1] -= v[1];
	elt[2] -= v[2];
	elt[3] -= v[3];
	
	return(SELF);
}

TMPLVec inline TVec4 &TVec4::operator *= (const TVec4 &v)
{
	elt[0] *= v[0];
	elt[1] *= v[1];
	elt[2] *= v[2];
	elt[3] *= v[3];
	
	return(SELF);
}

TMPLVec inline TVec4 &TVec4::operator *= (TReal s)
{
	elt[0] *= s;
	elt[1] *= s;
	elt[2] *= s;
	elt[3] *= s;
	
	return(SELF);
}

TMPLVec inline TVec4 &TVec4::operator /= (const TVec4 &v)
{
	elt[0] /= v[0];
	elt[1] /= v[1];
	elt[2] /= v[2];
	elt[3] /= v[3];
	
	return(SELF);
}

TMPLVec inline TVec4 &TVec4::operator /= (TReal s)
{
	elt[0] /= s;
	elt[1] /= s;
	elt[2] /= s;
	elt[3] /= s;
	
	return(SELF);
}


TMPLVec inline TVec4 TVec4::operator + (const TVec4 &a) const
{
	TVec4 result;
	
	result[0] = elt[0] + a[0];
	result[1] = elt[1] + a[1];
	result[2] = elt[2] + a[2];
	result[3] = elt[3] + a[3];
	
	return(result);
}

TMPLVec inline TVec4 TVec4::operator - (const TVec4 &a) const
{
	TVec4 result;
	
	result[0] = elt[0] - a[0];
	result[1] = elt[1] - a[1];
	result[2] = elt[2] - a[2];
	result[3] = elt[3] - a[3];
	
	return(result);
}

TMPLVec inline TVec4 TVec4::operator - () const
{
	TVec4 result;
	
	result[0] = -elt[0];
	result[1] = -elt[1];
	result[2] = -elt[2];
	result[3] = -elt[3];
	
	return(result);
}

TMPLVec inline TVec4 TVec4::operator * (const TVec4 &a) const			
{
	TVec4 result;

	result[0] = elt[0] * a[0];
	result[1] = elt[1] * a[1];
	result[2] = elt[2] * a[2];
	result[3] = elt[3] * a[3];
	
	return(result);
}

TMPLVec inline TVec4 TVec4::operator * (TReal s) const
{
	TVec4 result;
	
	result[0] = elt[0] * s;
	result[1] = elt[1] * s;
	result[2] = elt[2] * s;
	result[3] = elt[3] * s;
	
	return(result);
}

TMPLVec inline TVec4 TVec4::operator / (const TVec4 &a) const
{
	TVec4 result;
	
	result[0] = elt[0] / a[0];
	result[1] = elt[1] / a[1];
	result[2] = elt[2] / a[2];
	result[3] = elt[3] / a[3];
	
	return(result);
}

TMPLVec inline TVec4 TVec4::operator / (TReal s) const
{
	TVec4 result;
	
	result[0] = elt[0] / s;
	result[1] = elt[1] / s;
	result[2] = elt[2] / s;
	result[3] = elt[3] / s;
	
	return(result);
}

TMPLVec inline TReal TVec4::operator dot (const TVec4 &a) const
{
	return(elt[0] * a[0] + elt[1] * a[1] + elt[2] * a[2] + elt[3] * a[3]);
}

TMPLVec inline TVec4 operator * (TReal s, const TVec4 &v)
{
	return(v * s);
}

TMPLVec inline void TVec4::MakeZero()
{
	elt[0] = vl_zero; elt[1] = vl_zero; elt[2] = vl_zero; elt[3] = vl_zero;
}

TMPLVec inline void TVec4::MakeBlock(TReal k)
{
	elt[0] = k; elt[1] = k; elt[2] = k; elt[3] = k;
}

TMPLVec inline TVec4::Vec4(ZeroOrOne k)  	
{
	elt[0] = k;
	elt[1] = k;
	elt[2] = k;
	elt[3] = k;
}

TMPLVec	inline TVec4 &TVec4::operator = (ZeroOrOne k)
{
	elt[0] = k; elt[1] = k; elt[2] = k; elt[3] = k;
	
	return(SELF);
}

#ifdef VL_TEMPLATE
#include "Vec4.cc"
#endif

#endif
