/*
	File:			Vec3.h

	Function:		Defines a length-3 vector.
					
	Author(s):		Andrew Willmott

	Copyright:		Copyright (c) 1995-1996, Andrew Willmott
 */

#ifndef __Vec3__
#define __Vec3__

#include "VL.h"
 
 
// --- Vec3 Class -------------------------------------------------------------


TMPLVec class Vec2;

TMPLVec class Vec3
{
public:

	// Constructors
	
	inline				Vec3();
	inline				Vec3(TReal x, TReal y, TReal z);	// [x, y, z]
	inline				Vec3(const TVec3 &v);			    // Copy constructor
	inline				Vec3(const TVec2 &v, TReal w);		// Hom. 2D vector
	inline				Vec3(ZeroOrOne k);
	
	// Accessor functions
	
	inline Int			Elts() const { return(3); };
	
	inline TReal 		&operator () (Int i);		    // v(1) - Indexing
	inline const TReal	&operator () (Int i) const;		  

	inline TReal		&operator [] (Int i);		    // v[1] - No index
	inline const TReal	&operator [] (Int i) const;		//        check

	inline TReal		*Ref() const;					// Return pointer to data

	// Assignment operators
	
	inline TVec3	&operator =  (const TVec3 &a);			
	inline TVec3	&operator =  (ZeroOrOne k);	    				
	inline TVec3	&operator += (const TVec3 &a);
	inline TVec3	&operator -= (const TVec3 &a);
	inline TVec3	&operator *= (const TVec3 &a);
	inline TVec3	&operator *= (TReal s);
	inline TVec3	&operator /= (const TVec3 &a);
	inline TVec3	&operator /= (TReal s);
	
	// Comparison operators

	Bool			operator == (const TVec3 &a) const;	// v == a?
	Bool			operator != (const TVec3 &a) const;	// v != a?

	// Arithmetic operators
		
	inline TVec3	operator + (const TVec3 &a) const;		// v + a
	inline TVec3	operator - (const TVec3 &a) const;		// v - a
	inline TVec3	operator - () const;					// -v
	inline TVec3	operator * (const TVec3 &a) const;		// v * a (vx * ax, ...)
	inline TVec3	operator * (TReal s) const;				// v * s
	inline TVec3	operator / (const TVec3 &a) const;		// v / a (vx / ax, ...)
	inline TVec3	operator / (TReal s) const;				// v / s	
	inline TReal	operator dot (const TVec3 &a) const;	// v . a

	// Initialisers
	
	inline void		MakeZero();								// Zero vector
	inline void		MakeUnit(Int i, TReal k = vl_one);		// I[i]
	inline void		MakeBlock(TReal k = vl_one);			// All-k vector

	// Private...
	
protected:
	
	TReal elt[3]; 
};


// --- Vec operators -------------------------------------------------

TMPLVec inline TVec3 operator * (TReal s, const TVec3 &v);	// Left mult. by s
TMPLVec inline TReal len(const TVec3 &v);					// || v ||
TMPLVec inline TReal sqrlen(const TVec3 &v);				// v . v
TMPLVec inline TVec3 norm(const TVec3 &v);					// v / || v ||
TMPLVec inline void  normalise(TVec3 &v);					// v = norm(v)
TMPLVec inline TVec3 cross(const TVec3 &a, const TVec3 &b);	// a x b
TMPLVec inline TVec2 proj(const TVec3 &v);					// hom. projection

TMPLVec ostream	&operator << (ostream &s, const TVec3 &v);
TMPLVec istream	&operator >> (istream &s, TVec3 &v);


// --- Inlines ----------------------------------------------------------------


#include "Vec2.h"

TMPLVec inline TReal &TVec3::operator () (Int i)
{
	CheckRange(i, 0, 3, "(Vec3::(i)) index out of range");
	
    return(elt[i]);
}

TMPLVec inline const TReal &TVec3::operator () (Int i) const
{
	CheckRange(i, 0, 3, "(Vec3::(i)) index out of range");
	
    return(elt[i]);
}

TMPLVec inline TReal &TVec3::operator [] (Int i)
{
    return(elt[i]);
}

TMPLVec inline const TReal &TVec3::operator [] (Int i) const
{
    return(elt[i]);
}

TMPLVec inline TVec3::Vec3()
{
}

TMPLVec inline TVec3::Vec3(TReal x, TReal y, TReal z)
{
	elt[0] = x;
	elt[1] = y;
	elt[2] = z;
}

TMPLVec inline TVec3::Vec3(const TVec3 &v) 
{
	elt[0] = v[0];
	elt[1] = v[1];
	elt[2] = v[2];
}

TMPLVec inline TVec3::Vec3(const TVec2 &v, TReal w) 
{
	elt[0] = v[0];
	elt[1] = v[1];
	elt[2] = w;
}

TMPLVec inline TVec3::Vec3(ZeroOrOne k) 
{
	elt[0] = k; elt[1] = k; elt[2] = k;
}

TMPLVec inline TReal *TVec3::Ref() const
{
	return((TReal *) elt);
}

TMPLVec inline TVec3 &TVec3::operator = (const TVec3 &v)
{
	elt[0] = v[0];
	elt[1] = v[1];
	elt[2] = v[2];
	
	return(SELF);
}

TMPLVec	inline TVec3 &TVec3::operator = (ZeroOrOne k)
{
	elt[0] = k; elt[1] = k; elt[2] = k;
	
	return(SELF);
}

TMPLVec inline TVec3 &TVec3::operator += (const TVec3 &v)
{
	elt[0] += v[0];
	elt[1] += v[1];
	elt[2] += v[2];
	
	return(SELF);
}

TMPLVec inline TVec3 &TVec3::operator -= (const TVec3 &v)
{
	elt[0] -= v[0];
	elt[1] -= v[1];
	elt[2] -= v[2];
	
	return(SELF);
}

TMPLVec inline TVec3 &TVec3::operator *= (const TVec3 &a)
{
	elt[0] *= a[0];
	elt[1] *= a[1];
	elt[2] *= a[2];
	
	return(SELF);
}

TMPLVec inline TVec3 &TVec3::operator *= (TReal s)
{
	elt[0] *= s;
	elt[1] *= s;
	elt[2] *= s;
	
	return(SELF);
}

TMPLVec inline TVec3 &TVec3::operator /= (const TVec3 &a)
{
	elt[0] /= a[0];
	elt[1] /= a[1];
	elt[2] /= a[2];
	
	return(SELF);
}

TMPLVec inline TVec3 &TVec3::operator /= (TReal s)
{
	elt[0] /= s;
	elt[1] /= s;
	elt[2] /= s;
	
	return(SELF);
}

TMPLVec inline TVec3 TVec3::operator + (const TVec3 &a) const
{
	TVec3 result;
	
	result[0] = elt[0] + a[0];
	result[1] = elt[1] + a[1];
	result[2] = elt[2] + a[2];
	
	return(result);
}

TMPLVec inline TVec3 TVec3::operator - (const TVec3 &a) const
{
	TVec3 result;
	
	result[0] = elt[0] - a[0];
	result[1] = elt[1] - a[1];
	result[2] = elt[2] - a[2];
	
	return(result);
}

TMPLVec inline TVec3 TVec3::operator - () const
{
	TVec3 result;
	
	result[0] = -elt[0];
	result[1] = -elt[1];
	result[2] = -elt[2];
	
	return(result);
}

TMPLVec inline TVec3 TVec3::operator * (const TVec3 &a) const
{
	TVec3 result;
	
	result[0] = elt[0] * a[0];
	result[1] = elt[1] * a[1];
	result[2] = elt[2] * a[2];
	
	return(result);
}

TMPLVec inline TVec3 TVec3::operator * (TReal s) const
{
	TVec3 result;
	
	result[0] = elt[0] * s;
	result[1] = elt[1] * s;
	result[2] = elt[2] * s;
	
	return(result);
}

TMPLVec inline TVec3 TVec3::operator / (const TVec3 &a) const
{
	TVec3 result;
	
	result[0] = elt[0] / a[0];
	result[1] = elt[1] / a[1];
	result[2] = elt[2] / a[2];
	
	return(result);
}

TMPLVec inline TVec3 TVec3::operator / (TReal s) const
{
	TVec3 result;
	
	result[0] = elt[0] / s;
	result[1] = elt[1] / s;
	result[2] = elt[2] / s;
	
	return(result);
}

TMPLVec inline TReal TVec3::operator dot (const TVec3 &a) const
{
	return(elt[0] * a[0] + elt[1] * a[1] + elt[2] * a[2]);
}

TMPLVec inline TVec3 operator * (TReal s, const TVec3 &v)
{
	return(v * s);
}

TMPLVec inline void TVec3::MakeUnit(Int n, TReal k)
{
	if (n == 0)
	{ elt[0] = k; elt[1] = vl_zero; elt[2] = vl_zero; }
	else if (n == 1)
	{ elt[0] = vl_zero; elt[1] = k; elt[2] = vl_zero; }
	else if (n == 2)
	{ elt[0] = vl_zero; elt[1] = vl_zero; elt[2] = k; }
	else 
		Assert(false, "(Vec3::Unit) illegal unit vector");
}

TMPLVec inline void TVec3::MakeZero()
{
	elt[0] = vl_zero; elt[1] = vl_zero; elt[2] = vl_zero;
}

TMPLVec inline void TVec3::MakeBlock(TReal k)
{
	elt[0] = k; elt[1] = k; elt[2] = k;
}

TMPLVec inline Bool TVec3::operator == (const TVec3 &a) const
{
	return(elt[0] == a[0] && elt[1] == a[1] && elt[2] == a[2]);
}

TMPLVec inline Bool TVec3::operator != (const TVec3 &a) const
{
	return(elt[0] != a[0] || elt[1] != a[1] || elt[2] != a[2]);
}


TMPLVec inline TReal len(const TVec3 &v)
{
	return(sqrt(v dot v));
}

TMPLVec inline TReal sqrlen(const TVec3 &v)
{
	return(v dot v);
}

TMPLVec inline TVec3 norm(const TVec3 &v)	
{
	return(v / len(v));
}

TMPLVec inline void normalise(TVec3 &v)	
{
	v /= len(v);
}

TMPLVec inline TVec3 cross(const TVec3 &a, const TVec3 &b)
{
	TVec3 result;

	result[0] = a[1] * b[2] - a[2] * b[1];	
	result[1] = a[2] * b[0] - a[0] * b[2];	
	result[2] = a[0] * b[1] - a[1] * b[0];	

	return(result);
}

TMPLVec inline TVec2 proj(const TVec3 &v) 
{
	TVec2 result;
	
	Assert(v[2] != 0, "(Vec3/proj) last elt. is zero");
	
	result[0] = v[0] / v[2];
	result[1] = v[1] / v[2];
	
	return(result);
}


#ifdef VL_TEMPLATE
#include "Vec3.cc"
#endif

#endif
