/*
	File:			Vec.h

	Function:		Defines a generic resizeable vector.
					
	Author(s):		Andrew Willmott

	Copyright:		Copyright (c) 1995-1996, Andrew Willmott
 */

#ifndef __Vec__
#define __Vec__

#include "VL.h"
#include "Action.h"
#include "SGVec.h"
#include "Vec2.h"
#include "Vec3.h"
#include "Vec4.h"
#include <iostream.h>


// --- Vec Class --------------------------------------------------------------


TMPLVec class Vec
{
public:

	// Constructors
	
	Vec();								// Null vector: space allocated later
	Vec(Int n);										
	Vec(Int n, double elt0, ...);		// Vec(3, 1.1, 2.0, 3.4)						
	Vec(Int n, TReal data[]);			// Vector pointer...
	Vec(Int n, ZeroOrOne);				// Zero or all-ones vector	
	Vec(Int n, Axis a);					// Unit vector
	Vec(const TVec &v);			 		// Copy constructor
	Vec(const TSGVec &v);			 	
	Vec(const TVec2 &v);			 	
	Vec(const TVec3 &v);			 	
	Vec(const TVec4 &v);			 	
	
   ~Vec();			 					// Destructor
	
	// Accessor functions
		
	inline Int		Elts() const;

	inline TReal	&operator () (Int i);			// v(1) - Indexing
	inline TReal	operator () (Int i) const;		  

	inline TReal    &operator [] (Int i);			// v[1] - Has no index 
	inline TReal 	operator [] (Int i) const;		//        check			  

	inline TReal	*Ref() const;					// Return pointer to data

	// Assignment operators
	
	TVec	 		&operator =  (const TVec &v);	// v = a etc.
	TVec			&operator =  (const TSGVec &v);		
	TVec			&operator =  (ZeroOrOne k);
	TVec			&operator =  (Axis a);
	TVec 			&operator =  (const TVec2 &v);			
	TVec	 		&operator =  (const TVec3 &v);			
	TVec			&operator =  (const TVec4 &v);				
		
	TVec			&operator >> (Action<TReal> &a);// Apply action
	void			SetSize(Int n);					// resize the vector
	
	//	Vector initialisers
	
	void			MakeZero();
	void			MakeUnit(Int i, TReal k = vl_one);
	void			MakeBlock(TReal k);
	void			MakeBlock();
	
protected:
	TReal	*data; 
	UInt 	elts  : 31;
	UInt	isRef :  1;
};


// --- Vec In-Place operators -------------------------------------------------

TMPLVec TVec		&operator += (TVec &a, const TVec &b);
TMPLVec TVec		&operator -= (TVec &a, const TVec &b);
TMPLVec TVec		&operator *= (TVec &a, const TVec &b);
TMPLVec TVec		&operator *= (TVec &v, TReal s);
TMPLVec TVec		&operator /= (TVec &a, const TVec &b);
TMPLVec TVec		&operator /= (TVec &v, TReal s);

// --- Vec Comparison Operators -----------------------------------------------

TMPLVec Bool		operator == (const TVec &a, const TVec &b);
TMPLVec Bool		operator != (const TVec &a, const TVec &b);

// --- Vec Arithmetic Operators -----------------------------------------------

TMPLVec TVec		operator + (const TVec &a, const TVec &b);
TMPLVec TVec		operator - (const TVec &a, const TVec &b);
TMPLVec TVec		operator - (const TVec &v);
TMPLVec TVec		operator * (const TVec &a, const TVec &b);		
TMPLVec TVec		operator * (const TVec &v, TReal s);
TMPLVec TVec		operator / (const TVec &a, const TVec &b);
TMPLVec TVec		operator / (const TVec &v, TReal s);
TMPLMat TReal		operator dot (const TMVec &a, const TVec &b);
TMPLVec TVec		operator * (TReal s, const TVec &v);

TMPLVec TReal		len(const TVec &v);
TMPLVec TReal		sqrlen(const TVec &v);
TMPLVec TVec		norm(const TVec &v);

// --- Vec Input & Output -----------------------------------------------------

TMPLVec ostream		&operator << (ostream &s, const TVec &v);
TMPLVec istream		&operator >> (istream &s, TVec &v);

// --- Sub-matrix operators --------------------------------------------------

TMPLVec TVec		sub(const TVec &v, Int start, Int length); 		// Sub-vector	


// --- Vec inlines ------------------------------------------------------------


TMPLVec inline Int TVec::Elts() const
{
	return(elts);
}

TMPLVec inline TReal &TVec::operator () (Int i)
{
	CheckRange(i, 0, elts, "Vec::(i)");
	
    return(data[i]);
}

TMPLVec inline TReal TVec::operator () (Int i) const
{
	CheckRange(i, 0, elts, "Vec::(i)");
	
    return(data[i]);
}

TMPLVec inline TReal &TVec::operator [] (Int i)
{
	CheckRange(i, 0, elts, "Vec::[i]");
	
    return(data[i]);
}

TMPLVec inline TReal TVec::operator [] (Int i) const
{
	CheckRange(i, 0, elts, "Vec::[i]");

    return(data[i]);
}

TMPLVec inline TReal *TVec::Ref() const
{
	return(data);
}

TMPLVec inline TVec &TVec::operator = (ZeroOrOne k)
{
	MakeBlock(k);

	return(SELF);
}

TMPLVec inline TVec &TVec::operator = (Axis a)
{
	MakeUnit(a);

	return(SELF);
}

#ifdef VL_TEMPLATE
#include "Vec.cc"
#endif

#endif

