/*
	File:			SGMat.h

	Function:		Defines a scatter-gather matrix, i.e., a submatrix of another matrix.
					
	Author(s):		Andrew Willmott

	Copyright:		Copyright (c) 1995-1996, Andrew Willmott
 */

#ifndef __SGMat__
#define __SGMat__

#include "VL.h"
#include "Action.h"

// --- SGMat Class ------------------------------------------------------------

TMPLMat class Mat;
TMPLVec class Vec;

TMPLMat class SGMat
{
public:
	
	// Constructors
	
				SGMat(Int m, Int n, Int span, TMReal data[]);
				SGMat(const TSGMat &m);			
  	
	// Accessor functions
	
	Int			Rows() const { return rows; };
	Int			Cols() const { return cols; };

	inline TMVec   		operator () (Int i);		// Indexing by row
	inline const TMVec	operator () (Int i) const;		  

	inline TMVec		operator [] (Int i);		// Indexing by row; no checking
	inline const TMVec  operator [] (Int i) const;		  

	inline TMReal		&operator () (Int i, Int j);// Indexing by elt.
	inline TMReal		operator () (Int i, Int j) const;		  

	inline TMReal		&elt(Int i, Int j);	    	// Indexing by elt.; no checking
	inline TMReal		elt(Int i, Int j) const;

	// Assignment operators
	
	TSGMat			&operator = (const TSGMat &m);	// Assignment of a matrix
	TSGMat			&operator = (const TMat &m);	

	TSGMat 			&operator >> (Action<TMReal> &a);

protected:

	Int		rows;
	Int		cols;
	Int		span;
	TMReal	*data; 
};

// --- SGMat Inlines ----------------------------------------------------------


#include "Mat.h"

TMPLMat inline TMVec TSGMat::operator () (Int i)
{
	CheckRange(i, 0, Rows(), "(SGMat::(i)) index out of range");
	
    return(TMVec(cols, data + i * span));
}

TMPLMat inline const TMVec TSGMat::operator () (Int i) const
{
	CheckRange(i, 0, Rows(), "(SGMat::(i)) index out of range");
	
    return(TMVec(cols, data + i * span));
}

TMPLMat inline TMVec TSGMat::operator [] (Int i)
{
    return(TMVec(cols, data + i * span));
}

TMPLMat inline const TMVec TSGMat::operator [] (Int i) const
{
    return(TMVec(cols, data + i * span));
}

TMPLMat inline TMReal &TSGMat::operator () (Int i, Int j)
{
	CheckRange(i, 0, Rows(), "(SGMat::(i,j)) i index out of range");
	CheckRange(j, 0, Cols(), "(SGMat::(i,j)) j index out of range");
	
    return(data[i * span + j]);
}

TMPLMat inline TMReal TSGMat::operator () (Int i, Int j) const
{
	CheckRange(i, 0, Rows(), "(SGMat::(i,j)) i index out of range");
	CheckRange(j, 0, Cols(), "(SGMat::(i,j)) j index out of range");
	
    return(data[i * span + j]);
}

TMPLMat inline TMReal &TSGMat::elt(Int i, Int j)
{
    return(data[i * span + j]);
}

TMPLMat inline TMReal TSGMat::elt(Int i, Int j) const
{
    return(data[i * span + j]);
}


#ifdef VL_TEMPLATE
#include "SGMat.cc"
#endif

#endif

