/*
	File:			Mat4.h

	Function:		Defines a 4 x 4 matrix.
					
	Author(s):		Andrew Willmott

	Copyright:		Copyright (c) 1995-1996, Andrew Willmott
 */

#ifndef __Mat4__
#define __Mat4__

#include "VL.h"
#include "Vec4.h"


// --- Mat4 Class -------------------------------------------------------------


TMPLVec class Vec3;

TMPLMat class Mat4 
{
public:
	
	// Constructors
	
	inline		Mat4();
				Mat4(TMReal a, TMReal b, TMReal c, TMReal d,
					TMReal e, TMReal f, TMReal g, TMReal h,
					TMReal i, TMReal j, TMReal k, TMReal l,
					TMReal m, TMReal n, TMReal o, TMReal p);
				Mat4(const TMat4 &m);					// Copy constructor		
				Mat4(ZeroOrOne k);
					
	// Accessor functions
	
	inline Int			Rows() const { return(4); };
	inline Int			Cols() const { return(4); };
	
	inline TMVec4    	&operator () (Int i);			// m(1) - Indexing
	inline const TMVec4 &operator () (Int i) const;		  

	inline TMVec4    	&operator [] (Int i);		    // m[1] - Has no index 
	inline const TMVec4 &operator [] (Int i) const;		//		  check

	inline TMReal       &operator () (Int i, Int j);	// Indexing by elt.
	inline TMReal		operator () (Int i, Int j) const;		  

	inline TMReal		*Ref() const;					// Return pointer to data

	// Assignment operators
	
	TMat4	 			&operator =  (const TMat4 &m);	   
	inline TMat4	 	&operator =  (ZeroOrOne k);	   
	TMat4 				&operator += (const TMat4 &m);	
	TMat4 				&operator -= (const TMat4 &m);	    
	TMat4 				&operator *= (const TMat4 &m);	    
	TMat4 				&operator *= (TMReal s);	    			
	TMat4 				&operator /= (TMReal s);	    		

	// Comparison operators
	
	Bool				operator == (const TMat4 &m) const;	// M == N?
	Bool				operator != (const TMat4 &m) const;	// M != N?
	
	// Arithmetic operators
	
	TMat4				operator + (const TMat4 &m) const;	// M + N
	TMat4				operator - (const TMat4 &m) const;	// M - N
	TMat4				operator - () const;				// -M
	TMat4				operator * (const TMat4 &m) const;	// M * N
	TVec4				operator * (const TVec4 &v) const;	// M * v
	TMat4				operator * (TMReal s) const;		// M * s
	TMat4				operator / (TMReal s) const;		// M / s

	TMat4				operator ~ () const;				// trans(M)
	TMat4				operator ! () const;				// adj(M)
	
	// Mat-vector functions
	

	// Initialisers
	
	void		MakeZero();								// Zero matrix
	void		MakeUnit(TMReal k = vl_one);			// I
	void		MakeBlock(TMReal k = vl_one);			// all elts = k

	// Homogeneous Transforms
	
	void 		MakeHRot(const TVec3 &axis, Real theta);		
						// Rotate by theta radians about axis   
	void 		MakeHRot(const TVec4 &q);	// Rotate by quaternion   
	void		MakeHScale(const TVec3 &s);	// Scale by components of s
	void 		MakeHTrans(const TVec3 &t);	// Translation by t
	
	// Private...
	
protected:	

	TMVec4 row[4];	
};


// --- Matrix operators ------------------------------------------------------

TMPLMat TVec4        operator *  (const TVec4 &v, const TMat4 &m);// v * m
TMPLMat TVec4       &operator *= (TVec4 &a, const TMat4 &m);      // v *= m
TMPLMat inline TMat4 operator *  (TMReal s, const TMat4 &m);	  // s * m

TMPLMat inline TMat4	trans(const TMat4 &m);				// Transpose			
TMPLMat TMReal			trace(const TMat4 &m);				// Trace
TMPLMat inline TMat4	adj(const TMat4 &m);				// Adjoint
TMPLMat TMReal			det(const TMat4 &m);				// Determinant
TMPLMat TMat4			inv(const TMat4 &m);				// Inverse

TMPLMat ostream		&operator << (ostream &s, const TMat4 &m);
TMPLMat istream		&operator >> (istream &s, TMat4 &m);
	

// --- Inlines ----------------------------------------------------------------


TMPLMat inline TMat4::Mat4()
{
}


TMPLMat inline TMVec4 &TMat4::operator () (Int i)
{
	CheckRange(i, 0, 4, "(Mat4::(i)) index out of range");
	
    return(row[i]);
}

TMPLMat inline const TMVec4 &TMat4::operator () (Int i) const
{
	CheckRange(i, 0, 4, "(Mat4::(i)) index out of range");
	
    return(row[i]);
}

TMPLMat inline TMVec4 &TMat4::operator [] (Int i)
{
    return(row[i]);
}

TMPLMat inline const TMVec4 &TMat4::operator [] (Int i) const
{
    return(row[i]);
}

TMPLMat inline TMReal &TMat4::operator () (Int i, Int j)
{
	CheckRange(i, 0, 4, "(Mat4::(i, j)) index out of range");
	CheckRange(j, 0, 4, "(Mat4::(i, j)) index out of range");

	return(row[i][j]);
}

TMPLMat inline TMReal TMat4::operator () (Int i, Int j) const
{
	CheckRange(i, 0, 4, "(Mat4::(i, j)) index out of range");
	CheckRange(j, 0, 4, "(Mat4::(i, j)) index out of range");

	return(row[i][j]);
}

TMPLMat inline TMReal *TMat4::Ref() const
{
	return((TMReal *) row);
}

TMPLMat inline TMat4::Mat4(ZeroOrOne k)
{
	MakeUnit(k);
}

TMPLMat inline TMat4 &TMat4::operator = (ZeroOrOne k)
{
	MakeUnit(k);

	return(SELF);
}

TMPLMat inline TMat4 operator * (TMReal s, const TMat4 &m)
{
	return(m * s);
}

TMPLMat inline TMat4 trans(const TMat4 &m)
{
	return(~m);
}

TMPLMat inline TMat4 adj(const TMat4 &m)			
{
	return(!m);
}


#ifdef VL_TEMPLATE
#include "Mat4.cc"
#endif

#endif

