/*
	File:			Mat3.h

	Function:		Defines a 3 x 3 matrix.
					
	Author(s):		Andrew Willmott

	Copyright:		Copyright (c) 1995-1996, Andrew Willmott
 */

#ifndef __Mat3__
#define __Mat3__

#include "VL.h"
#include "Vec3.h"


// --- Mat3 Class -------------------------------------------------------------


TMPLVec class Vec4;

TMPLMat class Mat3 
{
public:
	
	// Constructors
	
	inline 			Mat3();
		 			Mat3(TMReal a, TMReal b, TMReal c,
					     TMReal d, TMReal e, TMReal f,
					     TMReal g, TMReal h, TMReal i);
					Mat3(const TMat3 &m);			// Copy constructor
					Mat3(ZeroOrOne k);
						
	// Accessor functions
	
	inline Int			Rows() const { return(3); };
	inline Int			Cols() const { return(3); };
	
	inline TMVec3		&operator () (Int i);			// m(1) - Indexing
	inline const TMVec3 &operator () (Int i) const;		  

	inline TMVec3    	&operator [] (Int i);			// m[1] - Has no index
	inline const TMVec3 &operator [] (Int i) const;		// 		  check

	inline TMReal       &operator () (Int i, Int j);		// Indexing by elt.
	inline TMReal		operator () (Int i, Int j) const;		  

	inline TMReal 		*Ref() const;					// Return pointer to data

	// Assignment operators
	
	TMat3 				&operator =  (const TMat3 &m);	    
	inline TMat3 		&operator =  (ZeroOrOne k);	    
	TMat3 				&operator += (const TMat3 &m);	    
	TMat3 				&operator -= (const TMat3 &m);	   
	TMat3 				&operator *= (const TMat3 &m);	   
	TMat3 				&operator *= (TMReal s);	    		
	TMat3 				&operator /= (TMReal s);	   
	
	// Comparison operators
	
	Bool				operator == (const TMat3 &m) const;	// M == N?
	Bool				operator != (const TMat3 &m) const;	// M != N?
	
	// Arithmetic operators
	
	TMat3				operator + (const TMat3 &m) const;	// M + N
	TMat3				operator - (const TMat3 &m) const;	// M - N
	TMat3				operator - () const;				// -M
	TMat3				operator * (const TMat3 &m) const;	// M * N
	inline TVec3		operator * (const TVec3 &v) const;	// M * v
	TMat3				operator * (TMReal s) const;		// M * s
	TMat3				operator / (TMReal s) const;		// M / s

	TMat3				operator ~ () const;				// trans(M)
	TMat3				operator ! () const;				// adj(M)
	
	// Matrix-vector functions
	
	// Initialisers
	
	void				MakeZero();							// Zero matrix
	void				MakeUnit(TMReal k = vl_one);			// I
	void				MakeBlock(TMReal k = vl_one);			// all elts = k

	// Vector Transforms
	
	void 				MakeRot(const TVec3 &axis, Real theta);		    
	void				MakeRot(const TVec4 &q);	// Rotate by quaternion 	
	void 				MakeScale(const TVec3 &s);

	// Homogeneous Transforms
	
	void				MakeHRot(Real theta);		// Rotate by theta rads    
	void				MakeHScale(const TVec2 &s);	// Scale by s
	void 				MakeHTrans(const TVec2 &t);	// Translation by t
	
	// Private...

protected:

	TMVec3 row[3];	
};


// --- Matrix operators ---------------------------------------------------


TMPLMat inline TVec3 &operator *= (TVec3 &v, const TMat3 &m);	// v *= m
TMPLMat inline TVec3  operator *  (const TVec3 &v, const TMat3 &m);// v * m
TMPLMat inline TMat3  operator *  (const TMReal s, const TMat3 &m);// s * m

TMPLMat inline TMat3	trans(const TMat3 &m);				// Transpose			
TMPLMat TMReal			trace(const TMat3 &m);				// Trace
TMPLMat inline TMat3	adj(const TMat3 &m);				// Adjoint
TMPLMat TMReal			det(const TMat3 &m);				// Determinant
TMPLMat TMat3			inv(const TMat3 &m);				// Inverse

TMPLMat ostream		&operator << (ostream &s, const TMat3 &m);
TMPLMat istream		&operator >> (istream &s, TMat3 &m);


// --- Inlines ----------------------------------------------------------------


TMPLMat inline TMat3::Mat3()
{
}

TMPLMat inline TMVec3 &TMat3::operator () (Int i)
{
	CheckRange(i, 0, 3, "(Mat3::(i)) index out of range");
	
    return(row[i]);
}

TMPLMat inline const TMVec3 &TMat3::operator () (Int i) const
{
	CheckRange(i, 0, 3, "(Mat3::(i)) index out of range");
	
    return(row[i]);
}

TMPLMat inline TMVec3 &TMat3::operator [] (Int i)
{
    return(row[i]);
}

TMPLMat inline const TMVec3 &TMat3::operator [] (Int i) const
{
    return(row[i]);
}

TMPLMat inline TMReal &TMat3::operator () (Int i, Int j)
{
	CheckRange(i, 0, 3, "(Mat3::(i, j)) index out of range");
	CheckRange(j, 0, 3, "(Mat3::(i, j)) index out of range");

	return(row[i][j]);
}

TMPLMat inline TMReal TMat3::operator () (Int i, Int j) const
{
	CheckRange(i, 0, 3, "(Mat3::(i, j)) index out of range");
	CheckRange(j, 0, 3, "(Mat3::(i, j)) index out of range");

	return(row[i][j]);
}

TMPLMat inline TMReal *TMat3::Ref() const
{
	return((TMReal *) row);
}

TMPLMat inline TMat3::Mat3(ZeroOrOne k)
{
	MakeUnit(k);
}

TMPLMat inline TMat3 &TMat3::operator = (ZeroOrOne k)
{
	MakeUnit(k);

	return(SELF);
}
	
TMPLMat inline TVec3 TMat3::operator * (const TVec3 &v) const
{
	TVec3 result;
	
	result[0] = v[0] * row[0][0] + v[1] * row[0][1] + v[2] * row[0][2];
	result[1] = v[0] * row[1][0] + v[1] * row[1][1] + v[2] * row[1][2];
	result[2] = v[0] * row[2][0] + v[1] * row[2][1] + v[2] * row[2][2];
	
	return(result);
}

TMPLMat inline TVec3 operator * (const TVec3 &v, const TMat3 &m)	
{
	TVec3 result;
	
	result[0] = v[0] * m[0][0] + v[1] * m[1][0] + v[2] * m[2][0];
	result[1] = v[0] * m[0][1] + v[1] * m[1][1] + v[2] * m[2][1];
	result[2] = v[0] * m[0][2] + v[1] * m[1][2] + v[2] * m[2][2];
	
	return(result);
}

TMPLMat inline TMat3 operator *  (const TMReal s, const TMat3 &m)
{
	return(m * s);
}

TMPLMat inline TVec3 &operator *= (TVec3 &v, const TMat3 &m)			
{	
	TReal t0, t1;
	
	t0   = v[0] * m[0][0] + v[1] * m[1][0] + v[2] * m[2][0];
	t1   = v[0] * m[0][1] + v[1] * m[1][1] + v[2] * m[2][1];
	v[2] = v[0] * m[0][2] + v[1] * m[1][2] + v[2] * m[2][2];
	v[0] = t0;
	v[1] = t1;

	return(v);
}

TMPLMat inline TMat3 trans(const TMat3 &m)
{
	return(~m);
}

TMPLMat inline TMat3 adj(const TMat3 &m)			
{
	return(!m);
}


// --- Matrix functions ---------------------------------


TMPLMat inline TMat3 Rot3(const TVec3 &axis, Real theta)
{
	TMat3 result; result.MakeRot(axis, theta); return(result);
}

TMPLMat inline TMat3 Rot3(const TVec4 &q)
{
	TMat3 result; result.MakeRot(q); return(result);
}

TMPLMat inline TMat3 Scale3(const TVec3 &s)
{
	TMat3 result; result.MakeScale(s); return(result);
}

TMPLMat inline TMat3 HRot3(Real theta)
{
	TMat3 result; result.MakeHRot(theta); return(result);
}

TMPLMat inline TMat3 HScale3(const TVec2 &s)
{
	TMat3 result; result.MakeHScale(s); return(result);
}

TMPLMat inline TMat3 HTrans3(const TVec2 &t)
{
	TMat3 result; result.MakeHTrans(t); return(result);
}


#ifdef VL_TEMPLATE
#include "Mat3.cc"
#endif

#endif

