/*
	File:			Basics.h

	Function:		Basic definitions for all files. Contains type definitions, assertion and
					debugging facilities, and miscellaneous useful template functions.
					
	Author(s):		Andrew Willmott

	Copyright:		Copyright (c) 1995-1996, Andrew Willmott
	
	Notes:			This header is affected by the following #defines:

					VL_CHECKING		- Include code for assertions, range errors and warnings.
					VL_DEBUG		- Enables misc. debugging statements.
					VL_FLOAT		- Use floats for real numbers. (Doubles are the default.)
*/

#ifndef __Basics__
#define __Basics__  

#include <iostream.h>
#include <math.h>


// --- Basic types --------------------------------


#ifndef VL_FLOAT
typedef double			Real;
#else
typedef float			Real;
#endif

typedef signed int		Int;
typedef signed short	Int16;
typedef signed long		Int32;

typedef unsigned int	UInt;
typedef unsigned short	UInt16;
typedef unsigned long	UInt32;

typedef unsigned long	Addr;		// Integral type that is the same size as a pointer.

typedef unsigned char	Byte;
typedef char			Char;

enum ZeroOrOne { vl_zero = 0, vl_0 = 0, vl_one = 1, vl_I = 1, vl_1 = 1 };

#define SELF (*this)	// A syntactic convenience.	


// --- Boolean type --------------------------------

#ifdef Bool
#undef Bool
#endif

class Bool 
{
public:
	
	Bool() : val(0) {};
	Bool(Byte b) { val = b; };

	operator Byte() { return val; };
	
private:
	Byte val;
};

ostream &operator << (ostream &s, Bool &b);
 
#ifndef HAS_TF
enum {false, true};
#endif

enum {vl_false, vl_true};


//	--- Mathematical constants -------------------------


#ifndef __MWERKS__
const Real vl_pi = M_PI;
const Real vl_halfPi = M_PI_2;
#else
const Real vl_pi = _PI;
const Real vl_halfPi = _PI / 2;
#endif


//	--- Dot product operator --------------------------------


#define dot		%


//	--- Assertions and Range checking ------------------------


#ifdef VL_CHECKING
#define Assert(b, e)				_Assert(b, e, __FILE__, __LINE__)
#define Expect(b, w)				_Expect(b, w, __FILE__, __LINE__)
#define CheckRange(a, l, u, r)		_CheckRange(a, l, u, r, __FILE__, __LINE__)
#else
#define Assert(b, e)
#define Expect(b, w)
#define CheckRange(a, l, u, r)
#endif

void _Assert(Int condition, Char *errorMessage, Char *file, Int line);
void _Expect(Int condition, Char *warningMessage, Char *file, Int line);
void _CheckRange(Int i, Int lowerBound, Int upperBound, char *rangeMessage, Char *file, Int line);
					// Check whether i is in the range [lowerBound, upperBound).


// --- Useful template functions ------------


template<class Value> inline Value Min(Value x, Value y);	
template<class Value> inline Value Max(Value x, Value y);
template<class Value> inline void  Swap(Value &x, Value &y);
template<class Value> inline Value Mix(Value x, Value y, Real s);
template<class Value> inline Value Abs(Value x);
template<class Value> inline Value Clip(Value x, Value min, Value max);
template<class Value> inline Value sqr(Value x);


// --- Inlines -------------------------------------------------------


template<class Value>
	inline Value Min(Value x, Value y)
	{
		if (x <= y)
			return(x);
		else
			return(y);
	};
	
template<class Value>
	inline Value Max(Value x, Value y)
	{
		if (x >= y)
			return(x);
		else
			return(y);
	};
	
template<class Value>
	inline void Swap(Value &x, Value &y)
	{
		Value t;
		
		t = x;
		x = y;
		y = t;
	};
	
template<class Value>
	inline Value Mix(Value x, Value y, Real s)
	{
		return(x + (y - x) * s);
	};

template<class Value>
	inline Value Abs(Value x)
	{
		if (x < 0)
			return(-x);
		else
			return(x);
	};

template<class Value>
	inline Value Clip(Value x, Value min, Value max)
	{
		if (x < min)
			return(min);
		else if (x > max)
			return(max);
		else
			return(x);
	};

template<class Value>
	inline Value sqr(Value x)
	{
		return(x * x);
	};


#ifdef VL_TEMPLATE
#include "Basics.cc"
#endif

#endif
