/*
	File:			Array.cc

	Function:		See header file

	Author(s):		Andrew Willmott

	Copyright:		Copyright (c) 1995-1996, Andrew Willmott

	Notes:			

	Change History:
		31/01/96	ajw		Started
*/


#include "Array.h"
#include <ctype.h>


TMPLArray TArray::Array(Int size, Int alloc) : items(size),
					allocated(alloc)
{
	Assert(size > 0, "(Array) Initial array size must be positive!");
	if (allocated < size)
		allocated = size;
	
	item = new T[allocated];
	Assert(item != 0, "(Array) Out of memory");
}

TMPLArray TArray::Array(const TArray &array) : items(array.items), allocated(array.allocated)
{
	Int i;
	
	item = new T[allocated];
	Assert(item != 0, "(Array) Out of memory");
	
	for (i = 0; i < array.items; i++)
		item[i] = array.item[i];
}

TMPLArray TArray::~Array()
{
	delete[] item;
}

TMPLArray TArray &TArray::operator >> (TAction &a)
{
	Int i;
	
	a.Start();
	
	for (i = 0; i < items; i++)
		a.Process(item[i]);
		
	a.Stop();
	
	return(SELF);
}

TMPLArray TArray &TArray::operator = (const TArray &array)
{
	Int i;

	if (allocated < array.allocated)
	{
		delete[] item;
		allocated = array.allocated;	
		item = new T[allocated];	
		Assert(item != 0, "(Array) Out of memory");
	}
			
	for (i = 0; i < array.items; i++)
		item[i] = array.item[i];

	items = array.items;
	
	return(SELF);
}

TMPLArray ostream &operator << (ostream &s, TArray &array)
{	
	Int i;

	s << '[';
	
	if (array.NumItems() > 0)
	{
		s << array[0];

		for (i = 1; i < array.NumItems(); i++)
			s << ' ' << array[i];
	}
	
	s << ']';

	return(s);
}

TMPLArray istream &operator >> (istream &s, TArray &array)
{
    Char	c;
	
	//	Expected format: [a b c d ...]
	
    while (isspace(s.peek()))			// 	chomp white space
		s.get(c);
		
    if (s.peek() == '[')						
    {
    	s.get(c);
    	array.Clear();
    	
	    while (isspace(s.peek()))		// 	chomp white space
			s.get(c);
    	
		while (s.peek() != ']')
		{			
			array.Add(1);
			s >> array.Top();			//	read an item
    	
			if (!s)
			{
				Expect(false, "Couldn't read array component");
				return(s);
			}
	
		    while (isspace(s.peek()))	// 	chomp white space
				s.get(c);
		}			
		s.get(c);
	}
    else
	{
	    s.clear(ios::failbit);
	    Expect(false, "Error: Expected '[' while reading array");
	    return(s);
	}
	
    return(s);
}

TMPLArray void TArray::SetSize(Int newSize)
{
	Int	i;
	T	*newArray;
	
	if (newSize > allocated)
	{
		if (allocated == 0)
			allocated = kFirstAllocation;
		else
			allocated *= 2;	
		
		while (newSize > allocated)
			allocated *= 2;	
		
		newArray = new T[allocated];
	
		for (i = 0; i < items; i++)
			newArray[i] = item[i];	
		
		delete[] item;
		item = newArray;
	}
	items = newSize;
}

TMPLArray void TArray::Add(Int n)
{
	SetSize(items + n);
}

TMPLArray void TArray::Shrink(Int n)
//	take away n items.
{
	items -= n;
}

TMPLArray void TArray::Insert(Int i, Int n)
//	Make space at position i for n items.
{
	Assert(i >= 0 && i <= items, "(Array:InsertSpace) Illegal index");

	Int j;
	
	Add(n);
	
	for (j = items - 1; j >= i + n; j--)
		item[j] = (item - n)[j];
}

TMPLArray void TArray::Delete(Int i, Int n)
//	Delete n items at position i.
{
	Assert(i >= 0 && i <= items, "(Array:InsertSpace) Illegal index");

	Int j;
	
	items -= n;
		
	for (j = i; j < items; j++)
		item[j] = (item + n)[j];
}

TMPLArray void TArray::ShrinkWrap()
//	Shrink allocated space to be only the current size of array
{
	// There is no realloc version of new in C++, so this involves another copy...
	
	Int	i;
	T	*newArray;
	
	allocated = items;	
	
	newArray = new T[allocated];

	for (i = 0; i < items; i++)
		newArray[i] = item[i];	
	
	delete[] item;
	item = newArray;
}

TMPLArray void TArray::Grow()
//	Allocate more space for the array. Used internally prior to an items++.
{
	Int	i;
	T	*newArray;
	
	if (allocated == 0)
		allocated = kFirstAllocation;
	else
		allocated *= 2;	
	
	newArray = new T[allocated];

	for (i = 0; i < items; i++)
		newArray[i] = item[i];	
	
	delete[] item;
	item = newArray;
}
