// Phong illumination model written as a BRDF

// reference:
// Bui-Tuong Phong,
// Illumination for Computer-Generated Pictures,
// Comm. of the ACM,
// vol. 18, no. 6, June 1975, pp. 311-317.

// Paul Heckbert	22 Oct 1996

#include <math.h>
#include "phong.H"

Phong::Phong() {
    iname = "Phong";
    nparam = 6;
    reset();
}

void Phong::reset() {			// set defaults
    kdp().reset("kd", .5, 0., 1.);
    ksp().reset("ks", .5, 0., 1.);
    exponp().reset("expon", 100., 0., 200.);
    mat_rp().reset("mat_r", 1., 0., 1.);
    mat_gp().reset("mat_g", 1., 0., 1.);
    mat_bp().reset("mat_b", 1., 0., 1.);
}

Vec3 Phong::brdf(const Vec3 &n, const Vec3 &l, const Vec3 &v) const {

    // Note that we rename the arguments i (incident) and o (outgoing)
    // as listed in illum.H to l (light) and v (viewer), respectively,
    // in keeping with convention.

    Vec3 h, rho(0., 0., 0.);

    double nl = dot(n, l);
    if (nl > 0. && dot(n, v) > 0.) {
	// if light illuminates point and viewer can see point
	// then it's not black
	// (we're assuming opaque surface here)

	rho = Vec3(mat_r(), mat_g(), mat_b()) * kd();	// diffuse part
	if (ks() > 0.) {
	    h = l+v;			// halfway vector
	    double nh = dot(n, h);
	    if (nh > 0.)
		rho += Vec3(1., 1., 1.) * ks() * pow(nh/len(h), expon()) / nl;
		    // specular part has highlight color = light color
	}
	// rho has been divided through by cos(theta_i)=nl, since this is a brdf
	// otherwise the formula would be matRGB*kd*nl + whiteRGB*ks*pow(...)
    }
    return rho;
}

void Phong::print(ostream &s) const {        // print current settings
    s << "Phong:"
	<< " kd=" << kd()
	<< " ks=" << ks()
	<< " expon=" << expon()
	<< " mat=" << Vec3(mat_r(), mat_g(), mat_b())
	<< endl;
}

#ifdef PHONGTEST

// compile with -DPHONGTEST to get a little test program that
// prints samples of BRDF in xz plane
//
// try "phongtest 30"

inline double deg_to_rad(double deg) {
    return deg*M_PI/180.;
}

void main(int argc, char **argv) {
    // read polar angle and azimuth of light from argument list, if any
    double theta_l = argc>1 ? deg_to_rad(atof(argv[1])) : 0.;
    double phi_l = argc>2 ? deg_to_rad(atof(argv[2])) : 0.;
    Vec3 light(sin(theta_l)*cos(phi_l), sin(theta_l)*sin(phi_l),
	cos(theta_l));
    cout << "light=" << light << endl;

    Phong phong;
    // we'll use default reflectance parameters, except:
    phong.mat_r() = 1.;	// material has orange color
    phong.mat_g() = .5;
    phong.mat_b() = .1;
    cout << phong << endl;

    Vec3 normal(0,0,1);
    int i;
    for (i=-10; i<=10; i++) {	// from -100 deg to 100 deg
	double theta_v_deg = i*10.;
	double theta_v = deg_to_rad(theta_v_deg);
	Vec3 view = cos(theta_v)*normal + sin(theta_v)*Vec3(1,0,0);
	cout << "theta_v=" << theta_v_deg << " view=" << view << endl;
	Vec3 rho = phong.brdf(normal, light, view);
	cout << "  rho=" << rho << endl;
    }
}

#endif
