/*
	File:		heBRDF.C

	Function:	Implements the He-Torrance Reflectance Model

	Notes:		Adapted from Greg Ward's .cal file,
			and the paper
			"He-Torrance Reflectance Model (Siggraph 1991)"

	The equation numbers in the comments are from the original
	paper by He, Torrance, Sillion and Greenberg.

	It would be nice if all coefficients that depend only on the
	model parameters could be precomputed.

	Notes:
	    rhoDD as written has a singularity when cos(ti) = 0 or cos(tr) = 0
	    We currently ignore rhoSp, as it's infinitely thin. Perhaps
		    we could return a narrow cone around V instead.

	WARNING: We haven't tested this code much, so there very well could
	be bugs!  You are advised to test it a lot.
    
	Andrew Willmott, Nov. 1996
*/

#include <values.h>	// defines _IEEE
#include <nan.h>
#include "heBRDF.H"
#include "complex.h"	// For fresnel function

#ifdef HE_DEBUG
#define DUMP(X, Y) cout << X << " = " << Y << endl;
#else
#define DUMP(X, Y)
#endif


const Vec3	lambdaRGB(.67, .55, .43);
const Real	z0err = 0.0001;			// accepted error in value of z0

Real HeBRDF::lambda = 0.5;
Real HeBRDF::epsilon = 0.000001;	


HeBRDF::HeBRDF()
{
	reset();
	nparam = 5;
	iname = "He-Torrance";
}

void HeBRDF::reset()
{
	//	These parameters are for aluminium

    param[0].reset("n_r", 0.770058, 0.0, 10.0);
    param[1].reset("n_i", 6.08351, 0.0, 10.0);
    param[2].reset("tau", 1.77, 0.0, 20.0);
    param[3].reset("sigma0", 0.28, 0.0, 5.0);
    param[4].reset("diffuse", 0.3, 0.0, 1.0);
}

Vec3 HeBRDF::brdf(const Vec3 &n, const Vec3 &i, const Vec3 &o) const
{
	Real	result;
	
	lambda = lambdaRGB[0];	//	sample at red

	result = ((HeBRDF*) this)->rhoDD(-norm(i), norm(o), n)
		+ ((HeBRDF*) this)->rhoUD(-norm(i), norm(o), n);

	return(Vec3(result, result, result));
}

void HeBRDF::print(ostream &s) const
{
	s << "He-Torrance:" << endl
		<< " n_r = " << nr()
		<< " n_i = " << nr()
		<< " tau = " << nr()
		<< " sigma0 = " << nr()
		<< " diffuse = " << nr()
		<< endl;
}

 


// --- The He-specific stuff -----------------------------------


Real HeBRDF::shadowf(Real t)
//	Shadowing function: Eqn (23)
{
	if (sigma0() == 0)
		return(1.0);
	else if (Abs(t) < 0.0001)
		return(1.0);
	else if (Abs(t) > (M_PI / 2.0 - 0.0001))
		return(0.0);
	else
	{
		Real	a = tau() / (2.0 * sigma0() * tan(t));
		Real	b = erfc(a);
		
		return(
			(1 - 0.5 * b) / 
			((exp(-sqr(a)) / (a * sqrt(M_PI)) - b) / 2.0 + 1.0)
		);
	}
}

Real HeBRDF::kK(Real t)
//	Eqns (83), (84)
{
	if (Abs(t) < epsilon)
		return(0);
	else
		return(
			tan(t) * erfc(tau() / (2.0 * sigma0() * tan(t)))
		);
}

Real HeBRDF::gFunc(Real cti, Real ctr)
//	g; Eqn (79)
{
	Real sigmaIR = sigma(acos(cti), acos(ctr));
	
	DUMP("sigma", sigmaIR);
	
	return(
		sqr(2 * M_PI / lambda * sigmaIR * (cti + ctr))
	);
}


#if 0

//	Greg Ward's code for the infinite series sum in D

Real Dsum2(Int m, Real lastT, Real c, Real t, Real e, Real g)
{
/*
	t(m) = pow(gt, m) / (fact(m) * m)
		 * exp(tf / m) * exp(-gt);
*/	

#if defined(HE_DEBUG) && 0
	cout << "step " << m << ": t = " << t << endl;
	cout << "step " << m << ": c = " << c << endl;
	cout << "step " << m << ": e = " << e << endl;
	cout << "step " << m << ": g = " << g << endl;
	cout << exp(-g) << endl;
#endif
	
	if (m > Dsummax) // || t < Dsumlim)
		return(t);
	else
        return(
        	t + Dsum2(m + 1, t, 
        	c * g / Real(m + 1),
			c * g / Real(m + 1) * exp(-g - e / Real(m + 1)),
        	e, g)
        );
}

Real Dsum(Real e, Real g)
{
	Real sum;
	
	sum = Dsum2(1, 0, g, g * exp(-g + e), -e, g);

#ifdef HE_DEBUG
	cout << "*** sum = " << sum << endl;
#endif

	return(sum);
}

#endif

// my version of the same

Real myDsum(Real tf, Real gt)
{
	/*
		Purpose: form

			sum 1..infinity: term(i)
		
			term(i) = pow(gt, i) / Real(fact(i) * i)
				 * exp(tf / Real(i)) * exp(-gt);
			 
		Better than this would be to start with term i = gt, and
		work outwards. (The max in gpowfact will occur around this
		term in the series.)
		
	*/

	Int		i;
	Real	sum = 0, term, lastTerm = 0;
	Real	gpowfact;
	
	gpowfact = gt;
	
	for (i = 1; i < 400; i++)
	{
		term = gpowfact * exp(-gt + tf / Real(i));
		// sometimes get underflow here
		if (IsNANorINF(term))
		    return sum;
		gpowfact *= gt * Real(i) / sqr(Real(i + 1));
		 
		sum += term;

		//	If we're over the hump of term(i), and the term
		//	is small enough, bail.

		if (term < 0.000001 && term < lastTerm)
			break;
	
		lastTerm = term;	
	}

#ifdef HE_DEBUG	
	cout << "*** sum = " << sum << endl;
#endif	
	
	return(sum);
}

Real HeBRDF::kD(const Vec3 &ki, const Vec3 &kr, const Vec3 &ns)
//	Distribution function D, eqn (78)
{
	Real gt, mf, tf, cti, ctr;
	Vec3 v;
	
	cti = -dot(ki, ns);
	ctr =  dot(kr, ns);
	v = kr - ki;
		
	gt = gFunc(cti, ctr);	
	mf = sqr(M_PI) * sqr(tau()) / (4.0 * sqr(lambda));
	tf = sqr(2.0 * M_PI / lambda) *
		-(sqrlen(v) - sqr(dot(ns, v))) * sqr(tau()) / 4.0;

#if defined(HE_DEBUG) && 1
	
	DUMP("cti", cti);
	DUMP("ctr", ctr);
	cout << "gt = " << gt << endl;
	cout << "mf = " << mf << endl;
	cout << "tf = " << tf << endl;
#endif

	return(mf * myDsum(tf, gt));
}

Real HeBRDF::kG(const Vec3 &ki, const Vec3 &kr, const Vec3 &ns)
// Geometrical factor G, eqn (76)
{
#ifdef PARANOID
	Vec3	si, sr, kc, pi, pr;
	Real	skc;
	
	si = norm(cross(ki, ns));
	sr = norm(cross(kr, ns));
	pi = cross(si, ki);
	pr = cross(sr, kr);
	kc = cross(kr, ki);
	skc = sqrlen(kc);
	Real top =
	    (sqr(dot(sr, ki)) + sqr(dot(pr, ki))) *
	    (sqr(dot(si, kr)) + sqr(dot(pi, kr)));
	Assert(Abs(top - sqr(skc)) < 1e-4, "we were wrong\n");

	// NOTE! last three factors of He's formula 76 multiplied together
	// equal 1!
	// We use the simplified formula below.
#endif
	
	Vec3 v = kr - ki;
	
	return(sqr(sqrlen(v) / dot(ns, v)));
	
#if 0
	cout << "kG:" << endl;
	DUMP("si", si);
	DUMP("sr", sr);
	DUMP("kc", kc);
	DUMP("v", v);
	DUMP("pi", pi);
	DUMP("pr", pr);
		
#endif
}


/*
	Code for finding z0,  the root to the following equation:

	sqrt(M_PI_2) * z0 == 
		sigma0 / 4.0 * (kK(ti) + kK(tr)) * exp(-sqr(z0 / sigma0) / 2.0)
*/

Real HeBRDF::z0d(Real Ki, Real z)
{
	return(
		-Ki / (4.0 * sigma0()) * z * exp(-sqr(z / sigma0()) / 2.0) - sqrt(M_PI / 2.0)
	);
}

Real HeBRDF::z0lim(Real x)
{
	if (x > 0)
		return(Max(x, z0err));
	else
		return(Min(x, -z0err));
}

Real HeBRDF::z0off(Real Ki, Real z)
{
	return(
		(sigma0() / 4.0 * Ki * exp(-sqr(z / sigma0()) / 2) - sqrt(M_PI / 2) * z) /
			z0lim(z0d(Ki, z))
	);
}

Real HeBRDF::z0rootf(Real Ki, Real x0, Real x1, Int i)
{
	if (i <= 0)
		return(0);
	else if (z0err > Abs(x1 - x0))
		return(x1);
	else
		return(z0rootf(Ki, x1, x1 - z0off(Ki, x1), i - 1));
}

Real HeBRDF::z0root(Real ti, Real tr)
{
	if (Abs(ti) < 0.6 && Abs(tr) < 0.6)
		return(0.0);
	else if (Abs(ti) >= (M_PI_2 - 0.0001) || Abs(tr) >= (M_PI_2 - 0.0001))
		return(1.0);
	else
	 	return(z0rootf(kK(ti) + kK(tr), 0.1, -z0off(kK(ti) + kK(tr), 0.1), 100));
}

Real HeBRDF::sigma(Real ti, Real tr)
//	Roughness factor sigma; eqn (80)
{
	if (Abs(sigma0()) < epsilon)
		return(sigma0());
	else
	{
		Real z0 = z0root(ti, tr);
		Real lhs, rhs;

		lhs = sqrt(M_PI_2) * z0;
		rhs = sigma0() / 4.0 * (kK(ti) + kK(tr)) * exp(-sqr(z0 / sigma0()) / 2.0);

#ifdef HE_DEBUG
		cout << "ti = " << ti << endl;
		cout << "tr = " << tr << endl;
		cout << "z0 = " << z0 << endl;
		cout << "lhs = " << lhs << endl;
		cout << "rhs = " << rhs << endl;
		cout << "kti = " << kK(ti) << endl;
		cout << "ktr = " << kK(tr) << endl;
#endif
		
		return(sigma0() / sqrt(1 + sqr(z0 / sigma0())));
	}
}

Real fresnel(Real ct, Real nr, Real ni)
//	The fresnel factor! 
{
	complex n(nr, ni);
	complex	ct1(ct, 0);
	complex	ct2 = sqrt(1 - (1 - sqr(ct1)) / sqr(n));
	complex a, b;
	
	a = (n * ct1 - ct2) / (n * ct1 + ct2);
	b = (n / ct1 - 1 / ct2) / (n / ct1 + 1 / ct2);
	
	return((sqr(real(a)) + sqr(imag(a)) + sqr(real(b)) + sqr(imag(b))) / 2.0);
}


Real HeBRDF::rhoSp(const Vec3 &ki, const Vec3 &kr, const Vec3 &ns)
// Eqn (70)
{
	Real fr, sti, str, cti, ctr;
	
	fr = fresnel(len(kr - ki) / 2.0, nr(), ni());
	cti = -dot(ki, ns);
	ctr =  dot(kr, ns);
	sti = shadowf(acos(cti));
	str = shadowf(acos(ctr));

	return(
		exp(-gFunc(cti, ctr)) * sti * str * fr / cti
	);
}

Real HeBRDF::rhoDD(const Vec3 &ki, const Vec3 &kr, const Vec3 &ns)
{
	Real ti, tr, cti, ctr;
	Real fr, kg, kd, sti, str;
	
#ifdef HE_DEBUG
	cout << "ki = " << ki << endl;
	cout << "kr = " << kr << endl;
	cout << "ns = " << ns << endl;
#endif

	cti = -dot(ki, ns);
	ctr =  dot(kr, ns);
	ti = acos(cti);
	tr = acos(ctr);

	fr = fresnel(len(kr - ki) / 2.0, nr(), ni());
	kg = kG(ki, kr, ns);
	kd = kD(ki, kr, ns);
	sti = shadowf(ti);
	if (sti < epsilon)
		return(0.0);
	str = shadowf(tr);
	if (str < epsilon)
		return(0.0);
	
#ifdef HE_DEBUG
	cout << "ti = " << ti << endl;
	cout << "tr = " << tr << endl;
	cout << "fr = " << fr << endl;
	cout << "kg = " << kg << endl;
	cout << "kd = " << kd << endl;
	cout << "sti = " << sti << endl;
	cout << "str = " << str << endl;
#endif

//	Eqn (72)

	Real result = 
		fr / M_PI *
		kd * sti * str * kg /
		(cti * ctr);
		
	return(result);
}

Real HeBRDF::rhoUD(const Vec3 &ki, const Vec3 &/*kr*/, const Vec3 &ns)
{
	Real result = -dot(ki, ns) * diffuse();
	
	return(result);
}

#ifdef HE_DEBUG

void main()
{
	HeBRDF	he;
	Real	x, result;
	
#if 0
	// Ceramic

	he.param[0].val = 1.5;
	he.param[1].val = 0;
	he.param[2].val = 1;
	he.param[3].val = 1.45;
#endif
#if 1
	// Aluminium
	
	he.param[0].val = 0.770058;
	he.param[1].val = 6.08351;
	he.param[2].val = 1.77;
	he.param[3].val = 0.28;
#endif

	he.lambda = 0.5;

#if 1
	for (x = 0.1; x < M_PI; x += 0.1)
	{
		result = he.rhoDD(norm(Vec3(1, -1, 0)), norm(Vec3(cos(x), sin(x), 0)), Vec3(0, 1, 0));	
		cout << "*** " << x << " = " << result << endl;
	}
#else
	for (x = 0.1; x < 1.0; x += 0.1)
	{
		result = he.gFunc(x, 0.4);
		cout << "*** " << x << " = " << result << endl;
	}
	x = x / 0.0;
#endif
}

#endif
