// Cook-Torrance illumination model

// reference:
// Robert L. Cook, Kenneth E. Torrance
// A Reflectance Model for Computer Graphics
// SIGGRAPH '81
// (also ACM Trans. on Graphics, 1(1), Jan. 1982, pp. 7-24)

// Noah Gibbs, Nov 1997

#include <math.h>
#include "cook_torrance.H"

// Arguments:
// --- What we calculate ---
//   s*Rs + d*Rd
// --- Passed as Params ---
//   m = RMS slope of facets (a higher slope signifies a rougher surface)
//   d = coefficient of diffuse reflectance compared to specular
//   n = index of refraction of material
//   k = extinction coefficient (not currently used)
//   Rd_red, Rd_blue, Rd_green = Diffuse reflectance (surface color)
// --- Passed by the call to brdf ---
//   N = surface Normal
//   L = vector to Light source
//   V = vector to Viewer
//   wavlen = lambda = wavelength of incoming EM radiation (like, say, light)
// --- Implicit in other information ---
//   s = 1.0 - d = coefficient of specular reflectance
//   G = Geometric facet self-shadowing function, function of N, L, and V
//   D = surface distribution function, function of m, N, L, and V
//   F = fresnel function, function of n, V, and L
//   Rs = specular reflectance, function of N, L, V, F, D, and G

CT::CT() {
  iname = new char[14];
  iname = "Cook-Torrance";
  nparam = 6;
  reset();
}

void CT::reset() {
  // reset params
  mp().reset("m", 0.4, 0.01, 3.0);   // Don't know a good high-end value
  dp().reset("d", 0.45,  0.0, 1.0);
  np().reset("n", 3.5,  0.5, 5.0);   // Ditto
  Rdrp().reset("Rd red", 1.0, 0.0, 1.0);
  Rdgp().reset("Rd blue", 1.0, 0.0, 1.0);
  Rdbp().reset("Rd green", 1.0, 0.0, 1.0);
}

double inline quickmin(double a, double b) {
  return (a<b) ? a : b;
}

// Here we calculate (according to the paper I've got) s*Rs + d*Rd.
// That's the Cook-Torrance BRDF.
Vec3 CT::brdf(const Vec3 &N, const Vec3 &L, const Vec3 &V) const {

  Vec3 rho(0.0, 0.0, 0.0);

  /* Current total cost */
  /* 2 dot products, 2 compares */
  /* 1 sub */
  /* 5 adds, 3 mults, 1 div, 1 sqrt */
  /* 2 dot products */
  /* 2 compares, 4 mults, 2 divs */
  /* 1 mult, 1 sub */
  /* 1 div */
  /* 1 exp, 1 negate, 4 mults, 2 divs */
  /* 1 sqrt, 6 mults, 2 divs, 10 adds */
  /* 3 mults, 1 add */
  /* 3 mults, 3 divs */
  /* 3 mults, 3 adds */

  /* Total: 4 dot products, 4 compares, 2 sqrts, 1 exp,
     1 negate, 11 divs, 27 mults, 20 adds */

  /* Precalculation, and checks for visibility */
  /* 2 dot products, 2 compares */
  double N_dot_V = dot(N,V);
  if(N_dot_V <= 0) return rho;
  double N_dot_L = dot(N,L);
  if(N_dot_L <= 0) return rho;

  /* one subtract */
  double s = 1.0 - d();

  /* Precalculation for finding G */
  /* 5 adds, 3 mults, 1 div, 1 sqrt */
  Vec3 H;
  H = V;
  H += L;  /* H = V + L */
  H /= sqrt(H[0]*H[0] + H[1]*H[1] + H[2]*H[2]);  /* H is now normalized */

  /* 2 dot products */
  double N_dot_H = dot(N,H);
  double V_dot_H = dot(V,H);

  /* Calculate geometric self-shadowing coefficient, min of 3 possibilities */
  /* 2 compares, 4 mults, 2 divs */
  double G = quickmin(quickmin(1.0,(2.0*N_dot_H*N_dot_V)/V_dot_H),
		      (2.0*N_dot_H*N_dot_L)/V_dot_H);

  /* Precalculation for finding D */
  /* 1 mult, 1 sub */
  double N_dot_H_2 = N_dot_H * N_dot_H;
  double sin2_alpha = 1.0 - N_dot_H_2;  /* sin^2(a) = 1.0 - cos^2(a) */

  /* tan^2(a) = sin^2(a) / cos^2(a) */
  /* 1 div */
  double tan2_alpha = sin2_alpha / N_dot_H_2;

  /* Calculate Beckmann distribution function */
  /* 1 exp, 1 negate, 4 mults, 2 divs */
  double D = exp(-tan2_alpha/(m() * m()))
    / (m() * m() * (N_dot_H_2 * N_dot_H_2));

  /* Precalculation for F */
  /* 1 sqrt, 6 mults, 2 divs, 10 adds */
  double c = V_dot_H;
  double g = sqrt(n() * n() + c * c - 1.0);
  double temp_term = ((g - c) * (g - c)) / ((g + c) * (g + c));
  double tt2 = (c * (g + c) - 1.0) / (c * (g - c) + 1.0);

  /* Calculate F.  This Fresnel function is valid only for k = 0, but it's
     apparently pretty close no matter what k is, within physical reason.
     I don't possess any of the references for the Fresnel term given, and
     I don't have time to look 'em up.  I'll do so later if this looks good
     enough or bad enough to warrant it. */
  /* 3 mults, 1 add */
  double F = 0.5 * temp_term * (1 + tt2 * tt2);

  //   Rs = specular reflectance, function of N, L, V, F, D, and G
  /* 3 mults, 3 divs */
  double Rs = (F / 3.14159265359) * (D / N_dot_L) * (G / N_dot_V);
  double s_times_Rs = s * Rs;

  /* return s * Rs + d() * Rd; */
  /* 3 mults, 3 adds */
  rho[0] = (d() * Rdr() + s_times_Rs);
  rho[1] = (d() * Rdg() + s_times_Rs);
  rho[2] = (d() * Rdb() + s_times_Rs);
  return rho;
}


void CT::print(ostream &s) const {    // print current settings
    s << "Cook_Torrance:"
	<< " m=" << m()
	<< " d=" << d()
	<< " n=" << n()
	<< " Rd=" << Vec3(Rdr(), Rdg(), Rdb())
	<< endl;
}



#ifdef CTTEST

// compile with -DPHONGTEST to get a little test program that
// prints samples of BRDF in xz plane
//
// try "phongtest 30"

inline double deg_to_rad(double deg) {
    return deg*M_PI/180.;
}

void main(int argc, char **argv) {
    // read polar angle and azimuth of light from argument list, if any
    double theta_l = argc>1 ? deg_to_rad(atof(argv[1])) : 0.;
    double phi_l = argc>2 ? deg_to_rad(atof(argv[2])) : 0.;
    Vec3 light(sin(theta_l)*cos(phi_l), sin(theta_l)*sin(phi_l),
	cos(theta_l));
    cout << "light=" << light << endl;

    rm_CT ct;
    // we'll use default reflectance parameters, except:
    ct.Rdr() = 1.;	// material has orange color
    ct.Rdg() = .5;
    ct.Rdb() = .1;

    cout << ct << endl;

    Vec3 normal(0,0,1);
    int i;
    for (i=-10; i<=10; i++) {	// from -100 deg to 100 deg
	double theta_v_deg = i*10.;
	double theta_v = deg_to_rad(theta_v_deg);
	Vec3 view = cos(theta_v)*normal + sin(theta_v)*Vec3(1,0,0);
	cout << "theta_v=" << theta_v_deg << " view=" << view << endl;
	Vec3 rho = ct.brdf(normal, light, view);
	cout << "  rho=" << rho << endl;
    }
}

#endif
