package info.collide.coolmodes.palettes.calc;

import java.awt.*;
import java.awt.event.*;
import java.net.*;
import javax.swing.*;
import javax.swing.border.*;

import info.collide.coolmodes.palettes.calc.*;
import info.collide.coolmodes.shared.*;
import info.collide.coolmodes.core.*;

import com.spiriteam.graph.*;

/**
 *
 * @author  westheide/schmidt
 * @version
 */

public class CalcPalette extends AbstractPalette implements ActionListener {
    
    /* The JGraph where all the nodes of our palettes will be placed for drag-and-drop */
    private JGraph palette;
    private JPanel ui;
    private TitledBorder border;
    private JScrollPane pane;
    /* toggle buttons for switching to addition edge/subtraction edge/delete edge mode and back to node mode */
    private JToggleButton drawAddEdgeButton;
    private JToggleButton drawSubEdgeButton;
    private JToggleButton deleteEdgeButton;
    /* for enabling or disabling once mode */
    private JCheckBox onceBox;
    private boolean onceMode = false;
    private CalcProp prop;
    
    /* A new instance of CalcPalette is loaded. The palette JGraph is loaded from an XML file and set to COPYONLY_MODE since the
     * JGraph is only for drag-and-drop and not to be manipulated
     */
    public CalcPalette() {
    }
    
    public void init() {
        palette=loadJGraph(prop.getProp("location.nodegraph"));
        palette.setInteractionMode(JGraph.COPYONLY_MODE);
        initUI();
    }
    
    /* We switch to node mode when the palette is selected */
    public void enterPalette() {
        toNodeMode();
    }
    
    protected String getIconLocation() {
        return "info/collide/coolmodes/palettes/calc/resources/calc.png";
    }
    
    public String getToolTipText() {
        return "Graphical Calculator";
    }
    
    /* Here we create the graphical user interface of our palette: a COPYONLY JGraph containing our nodes, a JPanel containing
     * the toggle buttons and the checkbox and two JGraphs for the graphical representation of our two kinds of edges
     */
    private void initUI() {
        ui = new JPanel();
        ui.setLayout(new BorderLayout());

        JPanel southPanel = new JPanel();
        southPanel.setLayout(new GridLayout(2,2));
        southPanel.setBorder(border = BorderFactory.createTitledBorder(prop.getProp("titledborder.edges")));

        drawAddEdgeButton = new JToggleButton(prop.getProp("button.additionedge"), new ImageIcon(getClass().getResource("/info/collide/coolmodes/palettes/calc/resources/Icon.Edges.Add.gif")), false);
        drawAddEdgeButton.addActionListener(this);
        drawAddEdgeButton.setMargin(new Insets(1, 1, 1, 1));
        drawAddEdgeButton.setIconTextGap(2);
        drawAddEdgeButton.setHorizontalAlignment(SwingConstants.LEFT);
        southPanel.add(drawAddEdgeButton);
        
        drawSubEdgeButton = new JToggleButton(prop.getProp("button.subtractionedge"), new ImageIcon(getClass().getResource("/info/collide/coolmodes/palettes/calc/resources/Icon.Edges.Substract.gif")), false);
        drawSubEdgeButton.addActionListener(this);
        drawSubEdgeButton.setMargin(new Insets(1, 1, 1, 1));
        drawSubEdgeButton.setIconTextGap(2);
        drawSubEdgeButton.setHorizontalAlignment(SwingConstants.LEFT);
        southPanel.add(drawSubEdgeButton);

        deleteEdgeButton = new JToggleButton(prop.getProp("button.deleteedge"), new ImageIcon(getClass().getResource("/info/collide/coolmodes/palettes/calc/resources/Icon.Edges.Delete.gif")), false);
        deleteEdgeButton.addActionListener(this);
        deleteEdgeButton.setMargin(new Insets(1, 1, 1, 1));
        deleteEdgeButton.setIconTextGap(2);
        deleteEdgeButton.setHorizontalAlignment(SwingConstants.LEFT);
        southPanel.add(deleteEdgeButton);

        onceBox = new JCheckBox(prop.getProp("checkbox.once"),false);
        onceBox.addActionListener(this);
        southPanel.add(onceBox);
        
        ui.add(pane = new JScrollPane(palette), BorderLayout.CENTER);
        ui.add(southPanel, BorderLayout.SOUTH);
    }
    
    public Component getUI() {
        return ui;
    }
    
    /* when we switch back to node mode, this means that the toggle buttons for each of our edge modes are to be deselected and the once
     * mode checkbox to be enabled.
     */
    public void setNodeModeUI() {
        drawAddEdgeButton.setSelected(false);
        drawSubEdgeButton.setSelected(false);
        deleteEdgeButton.setSelected(false);
        onceBox.setEnabled(true);
    }
    
    public void changeLanguage() {
        try {
            border.setTitle(prop.getProp("titledborder.edges"));
            drawAddEdgeButton.setText(prop.getProp("button.additionedge"));
            drawSubEdgeButton.setText(prop.getProp("button.subtractionedge"));
            deleteEdgeButton.setText(prop.getProp("button.deleteedge"));
            onceBox.setText(prop.getProp("checkbox.once"));
            palette = loadJGraph(prop.getProp("location.nodegraph"));
            palette.setInteractionMode(JGraph.COPYONLY_MODE);
            pane.setViewportView(palette);
        }
        catch (Exception ex) {
            ex.printStackTrace();
        }
    }
    
    /* Here we process the ActionsEvents coming from our toggle buttons and the checkbox: switching to the selected mode and accordingly
     * changing the user interface. After each such ActionEvent, the last active workspace is activated again
     */
    public void actionPerformed(ActionEvent event) {
        if (event.getSource().equals(onceBox)) {
            onceMode = onceBox.isSelected();
        }
        if (event.getSource() == drawAddEdgeButton) {
            if (drawAddEdgeButton.isSelected()) {
                drawSubEdgeButton.setSelected(false);
                deleteEdgeButton.setSelected(false);
                toEdgeMode(new AdditionEdge(),onceMode);
                onceBox.setEnabled(false);
            }
            else {
                toNodeMode();
                onceBox.setEnabled(true);
            }
        }
        else if (event.getSource() == drawSubEdgeButton) {
            if (drawSubEdgeButton.isSelected()) {
                drawAddEdgeButton.setSelected(false);
                deleteEdgeButton.setSelected(false);
                toEdgeMode(new SubtractionEdge(),onceMode);
                onceBox.setEnabled(false);
            }
            else {
                toNodeMode();
                onceBox.setEnabled(true);
            }
        }
        else if (event.getSource() == deleteEdgeButton) {
            if (deleteEdgeButton.isSelected()) {
                drawAddEdgeButton.setSelected(false);
                drawSubEdgeButton.setSelected(false);
                toEdgeMode(null,onceMode);
                onceBox.setEnabled(false);
            }
            else {
                toNodeMode();
                onceBox.setEnabled(true);
            }
        }
        activateLastWorkspace();
    }
}


