/*
** PT.C : Interrupt-level support for the posicionador
**  tridimensionale board
**
** see pt.h for info
**
** mostly cloned from mpu.c - nix
*/
#ifdef PT

#include "switches.h"
#include <stdlib.h>     /* for ANSI malloc */
#include "cext.h"
#include <stdio.h>      /* for NULL */
#include <signal.h>     /* for ^C trapping */
#include <dos.h>        /* for FP_SEG, FP_OFF */
#include <bios.h>       /* for GETTIMEOFDAY */
#include "pt.h"
#include "ptbuff.h"
#include "userio.h"
#pragma check_stack (off)               /* stack checking reports errors calling DOS */

/* same as in mpu.c ... should it be in a header file? */
#ifdef MICROSOFT     /* for Microsoft C V5.1 and above */
#define DISABLEINTS     _disable
#define ENABLEINTS      _enable
#define INTERRUPT	interrupt far
#define GETVECTOR       _dos_getvect
#define SETVECTOR       _dos_setvect
#define GETTIMEOFDAY(a) _bios_timeofday(_TIME_GETCLOCK, &(a))
#else                /* for Turbo C V2.0 and above */
#define DISABLEINTS     disable
#define ENABLEINTS      enable
#define INTERRUPT       interrupt
#define GETVECTOR       getvect
#define SETVECTOR       setvect
#define GETTIMEOFDAY(a) (a = biostime(0, 0))
#endif

#define CLEAR_DATA 0xFF0000FFL
#define LOWBITS 0x0F

#undef DEBUG			        /* cancel userio defn. */
#define PP if(TRUE)
#define DEBUG		FALSE	        /* TRUE to write to screen RAM for debugging */
#define DBG_SCRNSEG 0xb800              /* segment of diag screen (CGA at present) */
#define DBG_RECTIME     0x7d4           /* record time assign debug */
#define DBG_ISRBYTE     0x7d6           /* where to dump the ISR byte debug */
#define DBG_UNH_SYS     0x7d8           /* unhandled sys-ex debug */
#define DBG_RETDATA 0x7da               /* returned data debug */
#define DBG_CLK2HST 0x7dc               /* clock to host called debug */

#if DEBUG
static char far *screen;
/*
** A useful debugging operation, prints a character on the screen
** to indicate where the Program Counter is in the ISR.
** a is character to display, b is screen attribute, c is address on screen
*/
#define debug(a, b, c)  { FP_SEG(screen) = DBG_SCRNSEG;\
	FP_OFF(screen) = (c);\
	*screen = (a);\
	FP_OFF(screen) = (c)+1;\
	*screen = (b); }
#else
#define debug(a, b, c)
#endif


/* address of original ISR, NULL if not assigned, to indicate no interrupt */
/* has been set up */
static void (INTERRUPT *Old_Intr)() = NULL;

static void INTERRUPT PTisr(void);     /* prototype of our interrupt */
/* interrupt will run on a dedicated stack */
#define mystacksize 512
static unsigned oldss, oldsp, mystack[mystacksize];
static int  Intr = PT_IRQ + 8;           /* interrupt used by PT */
static byte pic_mask,               /* mask to enable/disable the interrupt */
	 pic_irq_save;           /* save status of IRQ n in 1st PIC (any PC) */
static unsigned int
	 ptbase = PT_BASE,      /* base address of PT */
	 pic_IMR;                     /* The port of the PIC mask register */

/* PT input buffer */
/* data buffer, declared long to get 32-bit alignment: */
pt_value ptbuff[PTBUFF_SIZE];
int ptbuffhead = 0;     /* buffer head and tail pointers */
int ptbufftail = 0;
short pt_error_flags;

/* extern int musictrace; */

#define set_error(bit) pt_error_flags |= (1 << bit);

/* pt_flush -- empty out buffers */
/**/
void pt_flush()
{
	 ptbuffhead = 0;
	 ptbufftail = 0;
}

/* ptGetValue - read the PT buffer and put the values in
 * the passed pt_value */
int ptGetValue(pt_value *val)
{
	register pt_value *pt = ptbuff + ptbuffhead;

	if (ptbuffhead == ptbufftail) {
		return 0;
	} else {
		val->x = pt->x;
		val->y = pt->y;
		val->a = pt->a;
		ptbuffhead = (ptbuffhead + 1) & PTBUFF_MASK;
		return 1;
	}
}
/* pt_read - read the PT registers and put the values in ptbuff */
static void pt_read()
{
	 register pt_value *pt = ptbuff + ptbufftail;

	 ptbufftail = (ptbufftail + 1) & PTBUFF_MASK;
	 if (ptbufftail == ptbuffhead) {
		/* filled buffer faster than client emptied it */
		set_error(PTBUFFOVFL);
	 }
	 /* read the PT registers.  COORXL is read last
	  * to ack the interrupt */
	 pt->a = inp(ptbase + PT_COORZH) << 8;
	 pt->a += inp(ptbase + PT_COORZL);
	 pt->y = inp(ptbase + PT_COORY);
	 pt->x = inp(ptbase + PT_COORXH) << 8;
	 pt->x += inp(ptbase + PT_COORXL);
}

/* ----------------- ISR Initialise and Close down routines ----------------- */

/*
** Function to close down the interrupt service routine, return FALSE if
** unable to close routine, if signal complains. If err is non-zero,
** print the corresponding error message returned by ptOpen. The interrupt
** is only replaced if it was actually changed.
*/
int ptClose(int err)
{
	 int i;
	 static char *errmsgs[] = {      /* ensure error codes align with the msgs */
	"PT: illegal base address request\n"
	 };

	 if (Old_Intr != NULL) {
	DISABLEINTS();

	/* restore irq status bit in pic */
	/* printf("\n  pic: old %02x save %02x ",inp(pic_IMR),pic_irq_save); */
	outp(pic_IMR, (inp(pic_IMR) & ~pic_mask) | pic_irq_save);
	/* printf("new %02x\n",inp(pic_IMR)); */

	ENABLEINTS();
	SETVECTOR(Intr, Old_Intr);
	Old_Intr = NULL;    /* allow for multiple ptOpen and ptClose */
    }

	 if (err > 0 && err <= (sizeof(errmsgs) / sizeof(char *)))
	gprintf(ERROR, errmsgs[err - 1]);
    else if (err)
	gprintf(ERROR, "ptClose err %d\n",err);
/* if (signal(SIGINT, SIG_DFL) == SIG_ERR)
	return FALSE;
 */
	 return TRUE;
}

/*
** Check if we have a PC/AT architecture, by checking for the second PIC.
*/
static int UsingATpic(void)
{
	 byte save_pic;
	 int pic_there;

	 DISABLEINTS();
	 save_pic = inp(PICIMR_AT);
	 outp(PICIMR_AT, 0xAA);
	 pic_there = inp(PICIMR_AT) == 0xAA;
    outp(PICIMR_AT, save_pic);
    ENABLEINTS();
	 return pic_there;
}

/*
** Set up the PT hardware.  Returns an error number as defined in pt.h
** or 0 if things are ok:
** since the PT is so simple we can't really detect much.  Waiting
** to see if we get an interrupt is wrong because we don't really want
** to require to camera to be on when we start the software.
*/
int ptOpen(unsigned int BaseAddr)
{
	 byte Old_IMR;

	 /* validate parameters */
	 if (BaseAddr & 0x7) return (PTBASE_ERR);

	 ptbase = BaseAddr;

	 /* set up PTisr in interrupt vector */
	 /* PP printf("TestIntr\n");  */
	 Old_Intr = _dos_getvect(((unsigned) Intr));
	 /* Old_Intr = GETVECTOR(((unsigned) Intr)); /* save old address of ISR */
	 SETVECTOR(Intr, PTisr);		/* put in new address */

	 /* calculate IRQ's mask and status */
	 DISABLEINTS();
	pic_IMR = PICIMR;
	pic_mask = (1 << (Intr - 8));
	 Old_IMR = inp(pic_IMR);
	 pic_irq_save = Old_IMR & pic_mask;
	 outp(pic_IMR, Old_IMR & ~pic_mask);	/* unmask the irq */
	 ENABLEINTS();
	 /* printf("pic %02x now %02x\n",pic_IMR,inp(pic_IMR)); */

	 /* ack once to get things started */
	 (void) inp(ptbase + PT_COORXL);

	 return PTINITED;		/* success */
}


/*
** Interrupt service routine for data from PT.
*/
static void INTERRUPT PTisr(void)
{
/* This code moves the stack to an area known to be large enough.
	For unknown reasons, this doesn't work with Quick C, so we just
   hope that the system stack on which interrupts are processed is
   big enough.  This is probably true with recent versions (5.0) of DOS.
 */
#ifndef _QC
    _asm {
	mov ax,ss
	mov oldss,ax
	mov ax,sp
	mov oldsp,ax
	push ds
	pop ss
	mov sp,OFFSET mystack+(mystacksize*2)-2
    }
#endif
    ENABLEINTS();
	 pt_read();		/* actually grab the registers */
	 DISABLEINTS();          /* prevent an interrupt for EOI */
	 outp(PICCNTRL, EOI);    /* nonspecific End Of Interrupt to 8259 */
#ifndef _QC
	 _asm {
	mov ax,oldss
	push ax
	pop ss
	mov sp,oldsp
    }
#endif
    /* returning will reenable the interrupt */
}
#endif /* PT */

