/*
** MPU.C : These functions are the interupt handling routines to process
** interrupt requests for a Roland MPU-401 MIDI processing unit.
** The routines are: (see mpu.h for each function's prototype)
**
** mOpen()      - Setup the Interupt Service Routine (ISR) handling MPU-401
** mClose()     - Disable the ISR for the MPU-401, optionally print errors
** mGetData()   - Retrieve a byte from the MPU-401 interface DataPort
** mPutData()   - Send a data byte to the MPU-401
** mPutCmd()    - Send a command to the MPU-401
** mPutGetCmd() - Send a command and return the returned data byte.
**
** mOpen should be called before any of the other routines here are called,
** and mClose should be called before the program finishes.
**
** This source file contains non standard code and functions for the
** MicroSoft C V5.1 and Turbo C V1.5 compilers and versions above.
**
*/
/*
** Modifications by Roger Dannenberg:
**
** In Leigh's design, sysex messages must be emptied from the input buffer
** as fast as they come in; however, the CMT program ExGet
** waits for all data to be received before writing data to disk.  In
** general, it can be hard for a machine to keep up with full-speed
** MIDI sysex dumps, and these can far exceed 64K.  We could always
** allocate a huge buffer for all MIDI data, or we could save sysex
** messages into a separate buffer that is provided by the application
** program.  Other CMT implementations for small machines do the latter,
** so that's what I've done for DOS.  Sysex messages are ignored until
** the application provides a buffer, which will be declared as HUGE so
** that the 64K segment size is not a limitation.  As a signal that a
** sysex message has arrived, the first 4 bytes are placed in the ordinary
** MIDI message buffer.  When the application sees the 1st 4 bytes, it
** should retrieve data from the buffer.  (This is enforced in the midifns.c
** level, but not here.)  Since sysex buffers must be huge to be greater
** than 64K, I do not reuse the existing ring buffer structures.
**
** Error codes in midifns.h are used to set bits in midi_error_flags
** when errors are detected.  The important errors are BUFOVFL and
** SYSEXOVFL.
*/
/* Mods by GWL
**
** The MPU normally interrupts at IRQ 2; however, it is desirable to allow
** different IRQ lines.  Scanning the various IRQ interrupt vectors for an
** MPU sometimes tickles other devices and could cause problems.  Thus, we
** have opted to require the user to communicate the MPU's base port address
** and IRQ line with environment variables MPUBASE and MPUIRQ respectively.
** The user is cautioned to be careful: MPUisr does not chain to any other
** device that may share the MPU's IRQ, and there is only minimal checking
** for valid values.
*/

#include "switches.h"
#include <stdlib.h>     /* for ANSI malloc */
#include "cext.h"
#include <stdio.h>      /* for NULL */
#include <signal.h>     /* for ^C trapping */
#include <dos.h>        /* for FP_SEG, FP_OFF */
#include <bios.h>       /* for GETTIMEOFDAY */
#include "mpu.h"        /* for MPU-401 constants */
#include "midibuff.h"
#include "midicode.h"
#include "userio.h"     /* for abort_flag */
#include "midierr.h"    /* error flags */
#pragma check_stack (off)               /* stack checking reports errors calling DOS */

#ifdef MICROSOFT     /* for Microsoft C V5.1 and above */
#define DISABLEINTS     _disable
#define ENABLEINTS      _enable
#define INTERRUPT       interrupt far
#define GETVECTOR       _dos_getvect
#define SETVECTOR       _dos_setvect
#define GETTIMEOFDAY(a) _bios_timeofday(_TIME_GETCLOCK, &(a))
#else                /* for Turbo C V2.0 and above */
#define DISABLEINTS     disable
#define ENABLEINTS      enable
#define INTERRUPT       interrupt
#define GETVECTOR       getvect
#define SETVECTOR       setvect
#define GETTIMEOFDAY(a) (a = biostime(0, 0))
#endif

#define CLEAR_DATA 0xFF0000FFL
#define LOWBITS 0x0F

#undef DEBUG                            /* cancel userio defn. */
#define PP if(TRUE)
#define DEBUG           FALSE           /* TRUE to write to screen RAM for debugging */
#define DBG_SCRNSEG 0xb800              /* segment of diag screen (CGA at present) */
#define DBG_RECTIME     0x7d4           /* record time assign debug */
#define DBG_ISRBYTE     0x7d6           /* where to dump the ISR byte debug */
#define DBG_UNH_SYS     0x7d8           /* unhandled sys-ex debug */
#define DBG_RETDATA 0x7da               /* returned data debug */
#define DBG_CLK2HST 0x7dc               /* clock to host called debug */

#if DEBUG
static char far *screen;
/*
** A useful debugging operation, prints a character on the screen
** to indicate where the Program Counter is in the ISR.
** a is character to display, b is screen attribute, c is address on screen
*/
#define debug(a, b, c)  { FP_SEG(screen) = DBG_SCRNSEG;\
	FP_OFF(screen) = (c);\
	*screen = (a);\
	FP_OFF(screen) = (c)+1;\
	*screen = (b); }
#else
#define debug(a, b, c)
#endif

#define MPU_ACK         0xFE
#define IRQ9MASK        (2)


/* address of original ISR, NULL if not assigned, to indicate no interrupt */
/* has been set up */
static void (INTERRUPT *Old_Intr)() = NULL;

static void INTERRUPT MPUisr(void);     /* prototype of our interrupt */
/* interrupt will run on a dedicated stack */
#define mystacksize 512
static unsigned oldss, oldsp, mystack[mystacksize];
static void (*ClockToHostFn)(void);     /* Ptr to fn called each MPU Clock */
static int  Intr,           /* interrupt used by MPU-401 */
    UartMode = FALSE,       /* TRUE when MPU is in UART mode */
    CmdAck,                 /* MPU-401 command acknowledge flag */
    ExpectingDataAfterCmd,  /* Flag indicating MPU will return a byte */
    IsAT,                   /* Flag set for AT */
    trace;                  /* true if -trace switch entered (to show IRQ) */
static byte ReturnedData,   /* The byte MPU has returned */
    pic_mask,               /* mask to enable/disable the interrupt */
    pic_irq_save,           /* save status of IRQ n in 1st PIC (any PC) */
    AT_irq9_save;           /* save status of IRQ 9 in 2nd PIC (AT only) */
static unsigned int
    DataPort = MPUBASEADDR,      /* addr of MPU data port */
    ComPort = MPUBASEADDR + 1,   /* addr of MPU command port */
    StatPort = MPUBASEADDR + 1,  /* addr of MPU status port */
    pic_IMR;                     /* The port of the PIC mask register */

byte huge *xbuff = NULL; /* application-supplied sysex buffer */
long xbufmask;           /* mask for circular buffer */
long xbufhead = 0;       /* buffer head and tail offsets */
long xbuftail = 0;
static long old_xbuftail;

boolean xbuf_flush = false;   /* says to flush remainder of sysex message */
MidiMsg SysexHeader;    /* saves 1st 3 bytes of sysex messages for buffer */
byte SysexCounter = 0;  /* index into SysexHeader */

/* midi input buffer */
/* data buffer, declared long to get 32-bit alignment: */
long buff[BUFF_SIZE/4]; 
int buffhead = 0;     /* buffer head and tail pointers */
int bufftail = 0;

private MidiMsg CurrentMsg, RealTimeMsg;
private int data_len;
private int data_rcv;
private boolean firstbyte;

extern boolean ctrlFilter;
extern boolean exclFilter;
extern boolean realFilter;

extern int musictrace;

#define set_error(bit) midi_error_flags |= (1 << bit);

/***************
* LOCAL TABLES *
***************/

private byte VoiceTable[8] = {2,2,2,2,1,1,2,0};
/* note: length entry for EOX MUST BE ZERO: */
private byte CommonTable[8] = {0,1,2,1,0,0,0,0};

/* midi_flush -- empty out buffers */
/**/
void midi_flush()
{
    buffhead = 0;
    bufftail = 0;
    xbufhead = 0;
    xbuftail = 0;
    xbuf_flush = true;  /* in case sysex continuation messages are still coming */
}

/*
 * #define dbmax 32
 * int dbb[dbmax];
 * char * dbs[dbmax];
 * int dbx = 0;
 * int dbh = 0;
 */

/* ------------------------- MPU Interface Routines ----------------------- */

/*
** Retrieve a byte from the MPU-401 interface DataPort
*/
byte mGetData(void)
{
    while(inp(StatPort) & DSR)
	;                        /* wait until MPU-401 ready to send */
    return(inp(DataPort));
}

/*
** Put out data to MPU-401, refer pp9 MPU-401 Technical Reference Manual
*/
void mPutData(byte dat)
{
    while(inp(StatPort) & DRR)
		;     /* wait until MPU-401 ready to recieve */
    outp(DataPort, dat);
}
/*
** Send a command and return a byte from the MPU-401. As this routine calls
** mPutCmd, don't call this from within MPUisr either. Because getting the
** record counter value will reset it, we should add this record counter value
** to RecordTime to keep it accurate.
*/
byte mPutGetCmd(byte cmd)
{
    ExpectingDataAfterCmd = TRUE;
    mPutCmd(cmd);
    while (ExpectingDataAfterCmd)
		;
    return ReturnedData;
}

int NotCmdAck() { return !CmdAck; } /* used to defeat optimizer, see below */

/*
** Put out commands to the MPU-401, waiting until the command is acknowledged
** DON'T CALL THIS FUNCTION FROM WITHIN MPUisr! - It waits for MPUisr
** to reset CmdAck, calling mPutCmd via MPUisr could cause some tail chasing.
** Don't use this function to retrieve data from the MPU (commands 0xA0-0xAF)
** Use mPutGetCmd() above.
*
* This function busy waits on the interrupt to set CmdAck.  Under the
* default Microsoft compiler options, the compiler "over"optimizes the
* loop by not doing the test!  I'm sure this could be fixed by disabling
* certain optimizations, but I decided to move the test into a routine
* to (hopefully) defeat any optimization attempts.  If small routines were
* in-lined by the compiler, an optimizer could still lead you astray.
*/

extern int mPutCmd_Loop_Count = 0; /* see note above */

void mPutCmd(byte cmd)
{
    if(cmd != MPURESET && UartMode) {
	return;
    }
    while(inp(StatPort) & DRR)
	;        /* wait until MPU-401 ready to receive */
    CmdAck = FALSE; /* reset the acknowledge flag */
    outp(ComPort, cmd);
    if(cmd == MPURESET && UartMode) /* if a UartMode Reset is being sent */
	UartMode = FALSE;     /* no longer in UartMode, don't check for */
    else                      /* an ACK in this one case */
	while (NotCmdAck());  /* wait until command recvd is acknowledged */
    if(cmd == UARTMODE)
	UartMode = TRUE;
}

/* PutMidiMsg -- copy complete message from interrupt to buffer */
/**/
static void PutMidiMsg(MidiMsg msg)
{
    register byte *cmt_data = ((byte *) buff) + bufftail;

    /* filter out control changes */
    if (ctrlFilter) {
	register int data1 = databyte1(msg);
	register int hibits = statusbyte(msg) & 0xF0;
	if (hibits == 0xD0 ||   /* Chan Pressure */
	    hibits == 0xE0 ||       /* Pitch Bend */
	    hibits == 0xA0 ||       /* Poly Pressure */
	    ((hibits == 0xB0) &&    /* Control change (don't count switches) */
	     ((data1 < 64) || (data1 > 121)))) {
	    /* CONTROL MESSAGE HAS BEEN FILTERED */
	    return;
	}
    }
    if (realFilter) {
	if (statusbyte(msg) >= 0xF8) return;
    }
    if (exclFilter) {
	if (statusbyte(msg) == MIDI_SYSEX) return;
    }
    *((long *) cmt_data) = msg.l;
    bufftail = (bufftail + 4) & BUFF_MASK;
    if (bufftail == buffhead) {
	/* filled buffer faster than client emptied it */
	set_error(BUFFOVFL);
    }
}



/* ----------------- ISR Initialise and Close down routines ----------------- */

/*
** Function to close down the interrupt service routine, return FALSE if
** unable to close routine, if signal complains. If err is non-zero,
** print the corresponding error message returned by mOpen. The interrupt
** is only replaced if it was actually changed.
*/
int mClose(int err)
{
    int i;
    static char *errmsgs[] = {      /* ensure error codes align with the msgs */
	"Can\'t set up an abnormal end trap\n",
	"MPU-401 will not reset - it may not be connected\n",
	"MPU-401 will not acknowledge commands\n",
	"MPU-401 is not installed on an expected interrupt\n",
	"Can\'t allocate enough memory for MPU-401 operation",
	"Illegal IRQ request",
	"Illegal base address request"
    };

    if (Old_Intr != NULL) {
	DISABLEINTS();

	/* restore irq status bit in pic */
	/* printf("\n  pic: old %02x save %02x ",inp(pic_IMR),pic_irq_save); */
	outp(pic_IMR, (inp(pic_IMR) & ~pic_mask) | pic_irq_save);
	/* printf("new %02x\n",inp(pic_IMR)); */

	/* restore AT 2nd pic IRQ 9 * /
	if (IsAT)
	/* printf("ATpic: old %02x save %02x ",inp(PICIMR_AT),AT_irq9_save); */
	    outp(PICIMR_AT, (inp(PICIMR_AT) & ~IRQ9MASK) | AT_irq9_save);
	/* printf("new %02x\n",inp(PICIMR_AT)); */
	ENABLEINTS();
	SETVECTOR(Intr, Old_Intr);
	Old_Intr = NULL;    /* allow for multiple mOpen and mClose */
    }

    if (err > 0 && err <= (sizeof(errmsgs) / sizeof(char *)))
	gprintf(ERROR, errmsgs[err - 1]);
    else if (err)
	gprintf(ERROR, "mClose err %d\n",err);
    if (signal(SIGINT, SIG_DFL) == SIG_ERR)
	return FALSE;
    return TRUE;
}

/*
** If a Control-C or Abort interrupt occurs, exit gracefully,
** disabling the interrupt if it had been assigned.
*/
static void AbEnd(int i)
{
    abort_flag = true;
}

/*
** Function to use the time of day clock to check to see if a timeout
** has occured. Returns TRUE if the TimeOutPeriod has expired, FALSE
** if still within time. TimeOutPeriod is in system clock ticks, each
** 54.9 milliseconds.
*/
static int TimeOut(long StartTime, long TimeOutPeriod)
{
    long CurrentTime;

    GETTIMEOFDAY(CurrentTime);
    return CurrentTime - StartTime > TimeOutPeriod;
}

/*
** Check if we have a PC/AT architecture, by checking for the second PIC.
*/
static int UsingATpic(void)
{
    byte save_pic;
    int pic_there;

    DISABLEINTS();
    save_pic = inp(PICIMR_AT);
    outp(PICIMR_AT, 0xAA);
    pic_there = inp(PICIMR_AT) == 0xAA;
    outp(PICIMR_AT, save_pic);
    ENABLEINTS();
    return pic_there;
}

/*
** Test the given interrupt number to be connected to the MPU-401.
** Returns TRUE if the interrupt is connected, FALSE if not.
*/
static int TestIntr(int Intr, int UsingAT)
{
   byte Old_IMR;
   long StartTime;

    /* set up MPUisr in interrupt vector */
    /* PP printf("TestIntr\n");  */
    Old_Intr = _dos_getvect(((unsigned) Intr));
    /* Old_Intr = GETVECTOR(((unsigned) Intr)); /* save old address of ISR */
    SETVECTOR(Intr, MPUisr);            /* put in new address */

    /* calculate IRQ's mask and status */
    DISABLEINTS();
    if (UsingAT && Intr == 0x0A) {      /* if an AT with IRQ-2 */
	pic_IMR = PICIMR_AT;            /* program second PIC */
	pic_mask = IRQ9MASK;
    } else {                            /* an XT or an AT not on IRQ-2 */
	pic_IMR = PICIMR;
	pic_mask = (1 << (Intr - 8));
    }
    Old_IMR = inp(pic_IMR);
    pic_irq_save = Old_IMR & pic_mask;
    outp(pic_IMR, Old_IMR & ~pic_mask); /* unmask the irq */
    ENABLEINTS();
    /* printf("pic %02x now %02x\n",pic_IMR,inp(pic_IMR)); */
    /*
     * If MPUUART then this is a "dumb" MPU compatible, so you
     * cannot use a metronome command to get a response. Just exit now
     */
    if (getenv("MPUUART")) return TRUE;

    /*
    ** Send a metronome off command (reasonably benign), and see if we get
    ** a response from the ISR within a timeout period to determine if the
    ** interrupt is correct
    */

    while (inp(StatPort) & DRR)
	;
    CmdAck = FALSE;                     /* reset the acknowledge flag */
    outp(ComPort, METROOFF);
    GETTIMEOFDAY(StartTime);
    while (!CmdAck && !TimeOut(StartTime, TIMEUP))
	;                  /* wait for ISR to respond or timeout */
    if (!CmdAck) {
	inp(DataPort);         /* throw away the acknowledge sent */
	DISABLEINTS();
	outp(pic_IMR, Old_IMR);
	ENABLEINTS();
	if (musictrace)
	    printf("Didn't find on Interrupt %02X\n", Intr);
	SETVECTOR(Intr, Old_Intr);      /* put back old address */
	return FALSE;         /* we didn't find the ISR on this int */
    }
    if (musictrace)
	printf("Found on Interrupt %02X\n", Intr);
    return TRUE;               /* we've found the ISR */
}

/*
** Routine to prime the internal variables and attempt to establish
** communications with the MPU-401 and set up the interrupt to MPUisr.
** Returns an error number for the different error messages if unable to
** communicate with the MPU, and 0 if everything went ok.
** BaseAddr gives the base address of the MPU interface, IrqNo the interrupt
** number to use, SEARCHIRQ if searching is to be used.
*/
int mOpen(unsigned int BaseAddr, int IrqNo)
{
    int TimedOut = FALSE, NoOfResets = 2;
    long StartTime;

    setstatus(CurrentMsg, 0);
    firstbyte = TRUE;

    /* validate parameters */
    if (IrqNo!=SEARCHIRQ && (IrqNo<2 || IrqNo==6 || IrqNo>7) )
	return (MPUIRQ_ERR);
    if (BaseAddr & 0xf) return (MPUBASE_ERR);

    StatPort = ComPort = (DataPort = BaseAddr) + 1;
    UartMode = FALSE;
    /* SIGINT handler is set up by USERIO.C.  It causes a cu_cleanup
     * that in turn calls mClose, so we shouldn't override the handler
     * here.  On the other hand, I don't set up a handler for SIGABRT
     * elsewhere, so I'm leaving in this handler.  Depending on how
     * and when SIGABRT can be raised, it is probably a bug to call
     * AbEnd and not do other cleanup as well.
     */
    /* if(signal(SIGINT, AbEnd) == SIG_ERR)
	return(ABEND_ERR); */
    if (signal(SIGABRT, AbEnd) == SIG_ERR)
	return ABEND_ERR;

    /* if it's an AT, disable IRQ9 in 2nd PIC */

    IsAT = UsingATpic();
#if DEBUG
    puts(IsAT ? "Running on an AT" : "Running on an XT");
    puts("setting up ISR");
#endif
    DISABLEINTS();
    if (IsAT)
    {
	AT_irq9_save = inp(PICIMR_AT);
	outp(PICIMR_AT, AT_irq9_save | IRQ9MASK);
	AT_irq9_save &= IRQ9MASK ;
    }
    ENABLEINTS();

    /* see if MPU allows us to send a reset within a timeout period */
    do {
	GETTIMEOFDAY(StartTime);
	while ((inp(StatPort) & DRR) &&
	       !(TimedOut = TimeOut(StartTime, TIMEUP))) {
	    if ((inp(StatPort) & DSR) == 0)
		inp(DataPort);     /* read any garbage found & throw away */
	    }
#if DEBUG
	puts("sending reset");
#endif
	outp(ComPort, MPURESET);   /* send reset anyway, to see if resets */
    } while (--NoOfResets && TimedOut);
    if (!NoOfResets)
	return RESET_ERR;

    /*
    ** Send a metronome off command (reasonably benign), and see if we get a
    ** response from the MPU within a timeout period to determine if there is
    ** MPU-401 connected
    */

#if DEBUG
    puts("sending metronome off");
#endif
    while (inp(StatPort) & DRR)
	if ((inp(StatPort) & DSR) == 0)
	    inp(DataPort);                  /* read ACK & throw away */
    outp(ComPort, METROOFF);
    GETTIMEOFDAY(StartTime);
    while ((inp(StatPort) & DSR) && !TimeOut(StartTime, TIMEUP))
		;    /* wait until MPU-401 ready to send */
    if (inp(DataPort) != MPU_ACK)
	return(ACK_ERR);

    /* try setting up the ISR on a range of interrupts */

    if (musictrace)
	printf("About to search for IRQ %02x\n",IrqNo);
    if (IrqNo != SEARCHIRQ) {
	if (TestIntr(Intr=(IrqNo+8), IsAT)) /* if only on a single interrupt */
	    return(MPUINITED);
    } else /* test over a range of possible interrupts */
	for (Intr = MINTR; Intr <= MINTRLAST; Intr++)
	    if (TestIntr(Intr, IsAT))
		return MPUINITED;
    return INT_ERR;
}

static void UndoSysexMsg()
{
    xbuftail = old_xbuftail;
}

static boolean PutSysExByte(byte value)
{
    if ((SYSEX_ERR & midi_error_flags) ||
	!xbuff || exclFilter) return FALSE;
    /* the 1st 3 bytes will be saved to the normal midi buffer */
    if (SysexCounter < 3) {
	if (SysexCounter == 0) old_xbuftail = xbuftail;
	SysexHeader.b[SysexCounter++] = value;
    }
    xbuff[xbuftail++] = value;
    xbuftail &= xbufmask;
    if (xbuftail == xbufhead) {
	set_error(SYSEXOVFL);
	UndoSysexMsg();
	return FALSE;
    }
    return TRUE;
}


/************************************************************
*                           MidiParse
*
* Input : A midi byte.
* Output: none
* Return: none
* Effect:
***************************************************************/

private void MidiParse(int c)
{
#define CLEAR_DATA 0xFF0000FFL
#define LOWBITS 0x0F

   /* REALTIME MESSAGES */
    if ((c & MIDI_REALTIME) == MIDI_REALTIME) {
	setstatus(RealTimeMsg, c);
	PutMidiMsg(RealTimeMsg);
	return;
    }

    /* STATUS MESSAGES */
    if (c & MIDI_STATUS_BIT) {
	if (statusbyte(CurrentMsg) == MIDI_SYSEX) { /* END OF EXCLUSIVE */
	    if (PutSysExByte(MIDI_EOX))
		PutMidiMsg(SysexHeader);
	    firstbyte = TRUE;
	}
	setstatus(CurrentMsg, c);
	data_rcv = 0;
	if ((c & MIDI_COMMON) == MIDI_COMMON) {
	    data_len = CommonTable[c & LOWBITS];
	    if (data_len == 0) {
		if (c != MIDI_EOX) {
		    if (c != MIDI_SYSEX) PutMidiMsg(CurrentMsg);
		} else setstatus(CurrentMsg, 0); /* make sure EOX is ignored */
	    }
	} else data_len = VoiceTable[(c & MIDI_COMMON) >> 4];
	return;
    }

    /* EXCLUSIVE DATA */
    if (statusbyte(CurrentMsg) == MIDI_SYSEX) { /* PUT IN EXCL BUFFER */
	if (firstbyte) {
	    xbuf_flush = false;
	    PutSysExByte(MIDI_SYSEX);
	    firstbyte = FALSE;
	}
	PutSysExByte((byte) c);
	return;
    }

    /* VOICE MESSAGE DATA */
    if (statusbyte(CurrentMsg)) { /* use zero to mean "status unknown" */
	setdata(CurrentMsg, ++data_rcv, c);
	if (data_rcv >= data_len) {
	    PutMidiMsg(CurrentMsg);
	    midimsg(CurrentMsg) &= CLEAR_DATA;
	    data_rcv = 0;
	}
    }
}


/*
** Interrupt service routine for data from MPU-401.
** refer algorithm pp13 MPU-401 TRM.
*/
static void INTERRUPT MPUisr(void)
{
    /* this is static for 2 reasons: keep the interrupt stack small,
     *  and we want MPUisr_dat to be addressable after we swap stacks.
     */
    static byte MPUisr_dat;
/* This code moves the stack to an area known to be large enough.
   For unknown reasons, this doesn't work with Quick C, so we just
   hope that the system stack on which interrupts are processed is
   big enough.  This is probably true with recent versions (5.0) of DOS.
 */
#ifndef _QC
    _asm {
	mov ax,ss
	mov oldss,ax
	mov ax,sp
	mov oldsp,ax
	push ds
	pop ss
	mov sp,OFFSET mystack+(mystacksize*2)-2
    }
#endif
    ENABLEINTS();
    MPUisr_dat = inp(DataPort);
    debug(MPUisr_dat, 2, DBG_ISRBYTE);
    if (UartMode) {             /* if we are in UART mode, */
	MidiParse(MPUisr_dat);  /* parse the data */
    } else if (MPUisr_dat == MPU_ACK) { /* o.w. interpret mode */
	CmdAck = TRUE;                  /* command has been acknowledged */
	if (ExpectingDataAfterCmd) {                     /* there is data following */
	    ReturnedData = mGetData();              /* get the next byte sent */
	    ExpectingDataAfterCmd = FALSE;
	    debug(ReturnedData, 3, DBG_RETDATA);
	}
    }
    DISABLEINTS();                                  /* prevent an interrupt for EOI */
    outp(PICCNTRL, EOI);                    /* nonspecific End Of Interrupt to 8259 */
#ifndef _QC
    _asm {
	mov ax,oldss
	push ax
	pop ss
	mov sp,oldsp
    }
#endif
    /* returning will reenable the interrupt */
}
