/*

MIDIMAC_ Serial Driver Source Release 2.1
Copyright _1986 Dave Oppenheim
Modified for Mac II and Multifinder 9/21/89 Dave Oppenheim
Modified and used in the CMU Midi Toolkit by permission
All Rights Reserved

This is the C source code for an interrupt-driven serial driver for
a MIDI port on the Macintosh.  Both receive and transmit are interrupt-
driven.

This code is superceded by the MIDI Manager code and
THIS FILE SHOULD NOT BE INCLUDED IN THE CMT.LIB PROJECT
unless you want NOT to use the MIDI Manager.  The reason
I switched to the MIDI Manager is that this code seemed
to be unreliable, especially as newer faster MACs come along.

For receive, this means that the client supplied routine "process_byte"
is called for each received character.  Since this is an interrupt level
routine, care should be taken to avoid overly long computations. Furthermore,
Toolbox routines which cause memory to be allocated or freed must not
be called from process_byte.  See "Inside Macintosh" for further details
on what is and is not allowed during interrupt processing.

For transmit, each call to Xmit places the character in the output queue,
and, if the serial output is not busy, starts its transmission.  Whenever
the transmission of a character is complete, a transmit interrupt starts
the transmission of the next character, if any.  When the queue is empty,
no further interrupts occur.  If the queue is full, Xmit waits until there
is room before inserting another character. This means that Xmit can hang
indefinitely if the device connected to the output port is not ready.
During debugging, the output queue serves as a convenient record of the
last N characters transmitted.

There are three exported routines in this file:
    Xmit		- transmits a character to a serial port
    setupMIDI	- initializes a serial port for MIDI including configuring
		- the SCC chip and initializing the interrupt vectors
    restoreMIDI	- disables interrupts, for use when done with a serial port

In all routines, "port" is 0 for the Modem Port or 1 for the Printer Port.

In setupMIDI, "clockDivide" is 0x40 for 500 kHz, 0x80 for 1 MHz, or
0xC0 for 2 MHz.

It is a good idea to call restoreMIDI before doing a disk operation, and
setupMIDI afterwards, because interrupts can be disabled for quite awhile.

Have fun!

*/

#include "switches.h"
/* #include "Proto.h" */
#include "cext.h"
#include "mididriver.h"
#include "midiparse.h"

#define portA	0
#define portB	1

public void Pause();

/* where to find A5 */
/* extern Ptr CurrentA5 : 0x904; */

/* size of the output buffer and a mask for computing wrap arounds
 * NOTE: because we use a mask to compute the modulus operation,
 *		 the buffer size must be a power of two. */
#define OUT_BUF_SIZE 64
#define OUT_BUF_MASK (OUT_BUF_SIZE - 1)

/* SCC Serial Chip Addresses */

/* SCC base read & write addresses */
#define SCCRBase	0x1D8L
#define SCCWBase	0x1DCL

/* offsets for A & B control & data */
#define aData		0x6
#define aCtl		0x2
#define bData		0x4
#define bCtl		0x0

/* interrupt level 2 dispatch table [32 bytes] */
#define Lvl2DT		0x1B2L

/* pre-computed pointers to the control and data registers
 * NOTE: these are unintialized until setupMIDI */
private byte *aControlReg = NULL;
private byte *bControlReg = NULL;
private byte *aReadReg = NULL;
private byte *bReadReg = NULL;
private byte *aWriteReg = NULL;
private byte *bWriteReg = NULL;

public int aErrors = 0;
public int bErrors = 0;
public int waitCount = 0;

struct ioq {
    ushort qhead;
    ushort qtail;
    short nitems;	/* signed */
    byte data[OUT_BUF_SIZE];
} xmtQ[2];

boolean Sending[2];		/* transmit interrupt pending for given port */

void	RcvAISRguts(void);
void	RcvBISRguts(void);
void	RcvErrAISRguts(void);
void	RcvErrBISRguts(void);
void	XmitAISRguts(void);
void	XmitBISRguts(void);
void	EnQ(struct ioq *qp, byte value);
byte	DeQ(struct ioq *qp);
void	WriteSCCReg(short port, byte reg, byte value);

#ifdef LIGHTSPEED
void	RcvAISR(void);
void	RcvBISR(void);
void	RcvErrAISR(void);
void	RcvErrBISR(void);
void	XmitAISR(void);
void	XmitBISR(void);
#endif

#ifndef LIGHTSPEED
/* addresses of assembly language interrupt service routines */
extern XmitAISR();
extern XmitBISR();
extern RcvAISR();
extern RcvBISR();
extern RcvErrAISR();
extern RcvErrBISR();
extern RememberGReg();

#else

/* code for getting A5 (or A4, in a code resource or driver) at interrupt level in MultiFinder */

/* change A5 to A4 for a driver */
#define GREG A5

__GetGReg(void);
static
__GetGReg()
{
    asm {
	bsr.s	@1
	dc.l	0			;  store GREG here
@1		move.l	(sp)+,a1
    }
}


#define RememberGReg()	__GetGReg(); asm { move.l GREG,(a1) }
#define SetUpGReg()		asm { move.l GREG,-(sp) } __GetGReg(); asm { move.l (a1),GREG }
#define RestoreGReg()		asm { move.l (sp)+,GREG }


#endif

public void setupMIDI(port, clockDivide)
    int port;
    int clockDivide;
{

    RememberGReg();
    aControlReg = (*(byte **)SCCWBase) + aCtl;
    bControlReg = (*(byte **)SCCWBase) + bCtl;
    aReadReg = (*(byte **)SCCRBase) + aData;
    bReadReg = (*(byte **)SCCRBase) + bData;
    aWriteReg = (*(byte **)SCCWBase) + aData;
    bWriteReg = (*(byte **)SCCWBase) + bData;

    if (port == portA) {
	*((ulong *)Lvl2DT + 4) = (ulong) XmitAISR;
	*((ulong *)Lvl2DT + 6) = (ulong) RcvAISR;
	*((ulong *)Lvl2DT + 7) = (ulong) RcvErrAISR;
    } else {
	*((ulong *)Lvl2DT + 0) = (ulong) XmitBISR;
	*((ulong *)Lvl2DT + 2) = (ulong) RcvBISR;
	*((ulong *)Lvl2DT + 3) = (ulong) RcvErrBISR;
    }

    xmtQ[port].nitems = 0;
    xmtQ[port].qhead = 0;
    xmtQ[port].qtail = 0;
    Sending[port] = false;

    if (port == portA) {
	WriteSCCReg(0, 9, 0x82);	/* channel A reset, no vector */
    } else {
	WriteSCCReg(0, 9, 0x42);	/* channel B reset, no vector */
    }
    WriteSCCReg(port, 10, 0x00);	/* misc receive bits */
    
    WriteSCCReg(port, 11, 0x28);	/* external clock on TRxC */
    WriteSCCReg(port, 4, clockDivide + 4);
				    /* clock divided by whatever, 1 stop bit */
    WriteSCCReg(port, 3, 0xC1);		/* receive 8 data bits, rcv enabled */
    WriteSCCReg(port, 5, 0x6A);		/* transmit 8 bits, xmit enabled, rts */
    WriteSCCReg(port, 1, 0x13);		/* receive, transmit, and external
				     * interrupts enabled */
    WriteSCCReg(port, 15, 8);		/* all ext/stat but DCD disabled */
    WriteSCCReg(port, 0, 0x10);		/* reset ext/stat interrupts */
    WriteSCCReg(0, 9, 0x0A);		/* master interrupt enable, no vector */
}

public void restoreMIDI(long_port)
    long long_port;
{
    int port = long_port;	/* parameter is long for compatibility with cleanup.c */
    int i;
    /* disable rcv and xmit interrupts;
     * keep external interrupts for mouse */
    WriteSCCReg(port, 1, 1);
    /* wait for activity to cease */
    for (i = 0; i < 100; i++) Pause();
}

public void Xmit(port, value)
    int port;
    byte value;
{
    register struct ioq *qptr;

    qptr = &xmtQ[port];
    while (qptr->nitems >= OUT_BUF_SIZE)
	waitCount++;
    EnQ(qptr, value);
    if (!Sending[port]) {
	Sending[port] = true;
	if (port == portA) {
	    *aWriteReg = DeQ(qptr);
	} else {
	    *bWriteReg = DeQ(qptr);
	}
    }
}

public void XmitAISRguts()
{
    if (xmtQ[portA].nitems <= 0) {
	Sending[portA] = false;
	WriteSCCReg(portA, 0, 0x28);	/* reset xmit int pending */
    } else {
	*aWriteReg = DeQ(&xmtQ[portA]);
    }
}

public void XmitBISRguts()
{
    if (xmtQ[portB].nitems <= 0) {
	Sending[portB] = false;
	WriteSCCReg(portB, 0, 0x28);	/* reset xmit int pending */
    } else {
	*bWriteReg = DeQ(&xmtQ[portB]);
    }
}

public void RcvAISRguts()
{
    process_byte((int) *aReadReg);
}

public void RcvBISRguts()
{
    int c;

    c = *bReadReg;	/* read the byte to clear interrupt */
    /* NOTE: this is a noop because process_byte can't handle
     * data from multiple ports yet */
}

public void RcvErrAISRguts()
{
    register byte junk;

    junk = *aReadReg;		/* read byte */
    *aControlReg = 0x30;	/* clear error */
    aErrors++;
}

public void RcvErrBISRguts()
{
    register byte junk;

    junk = *bReadReg;		/* read byte */
    *bControlReg = 0x30;	/* clear error */
    bErrors++;
}

private void EnQ(qp, value)
    register struct ioq *qp;
    register byte value;
{
    qp->data[(qp->qtail++)] = value;
    qp->qtail &= OUT_BUF_MASK;	/* wrap */
    qp->nitems++;
}

private byte DeQ(qp)
    register struct ioq *qp;
{
    register byte value;

    value = qp->data[(qp->qhead++)];
    qp->qhead &= OUT_BUF_MASK;	/* wrap */
    qp->nitems--;
    return (value);
}

private void WriteSCCReg(port, reg, value)
    register short port;
    register byte reg;
    register byte value;
{
    if (port == portA) {
	*aControlReg = reg;
	Pause();
	*aControlReg = value;
	Pause();
    /* printf("sccput(%d, %d, %x, %lx)\n", port, reg, value, aControlReg); */
    } else {
	*aControlReg = reg;
	Pause();
	*aControlReg = value;
	Pause();
    /* printf("sccput(%d, %d, %x, %lx)\n", port, reg, value, bControlReg); */
    }
}

public void Pause()
{
    long junk, *junkPtr1, *junkPtr2;
    
    junkPtr1 = junkPtr2 = &junk;
    /* twiddle our thumbs for a bit */
    *junkPtr1 = *junkPtr2;
    *junkPtr2 = *junkPtr1;
    /* John Maloney says without the following we're ok up to 33MHz 68020.
       I threw in some extra writes in case the 68030 is faster.
     */
     
    *junkPtr1 = *junkPtr2 + 1;
    *junkPtr2 = *junkPtr1 + 1;
}

#ifdef LIGHTSPEED

private void RcvAISR()
{
    SetUpGReg();
    RcvAISRguts();
    RestoreGReg();
}

private void RcvBISR()
{
    SetUpGReg();
    RcvBISRguts();
    RestoreGReg();
}

private void RcvErrAISR()
{
    SetUpGReg();
    RcvErrAISRguts();
    RestoreGReg();
}

private void RcvErrBISR()
{
    SetUpGReg();
    RcvErrBISRguts();
    RestoreGReg();
}

private void XmitAISR()
{
    SetUpGReg();
    XmitAISRguts();
    RestoreGReg();
}

private void XmitBISR()
{
    SetUpGReg();
    XmitBISRguts();
    RestoreGReg();
}

#endif
