/* excldesc.c -- describe a system exclusive message if possible */
/* Copyright 1989 Carnegie Mellon University */

/*****************************************************************************
*       Change Log
*  Date | Change
*-----------+-----------------------------------------------------------------
* 13-Jun-86 | Created Change Log
*  5-Apr-91 | JDW : Further changes
*  4-Mar-92 | GWL : Clean up for Lattice
*****************************************************************************/

/* DOCUMENTATION:

Starting with version 3.3, system exclusive messages are retrieved via calls
to get_excl(), which returns sysex messages a block at a time.  It is no
longer allowed to "peek" in the sysex buffer as before, so the sysex message
is no longer visible all in one contiguous block.  This impacts the design
of the checksum facility in this module.  Before, checksum was called from
excldesc when a check-sumable message was found.  Now, checksumming needs
help from the sysex message reader as follows:

It is assumed that clients of this module call excldesc to print the message
type, manufacturer code, etc.  The buffer passed to excldesc must contain at
least 161 bytes (at this time) because that's how far ahead into the message 
excldesc looks.  If it's appropriate to do a checksum, excldesc calls 
checksum_start(), but it is the responsibility of the client to call checksum()
on each block, including the first, to do the job.  After the last block,
checksum_end() should be called to test the final result.

Note that the client does not call checksum_start.

*/

#include "ctype.h"
#include "cext.h"
#include "stdio.h"
#include "midicode.h"
#include "userio.h"
#include "midifns.h"
#include "excldesc.h"

/* length of Exclusive message preamble: */
#define HDRSIZ 6

/****************************************************************************
*           checksum_start
* Inputs:
*    char *msg: 1st block of the system exclusive message
* Effect: 
*    gets ready to check for parity in yamaha messages
****************************************************************************/

private int cksum;      /* check sum of message */
private long datalen;   /* number of bytes in message */
private int offset;	/* where to start in message (start at 6 1st time) */
private boolean started = false;	/* says ifwe're doing a check */


/* checksum_start -- tell system to begin checking message */
/**/
void checksum_start(msg)
  byte *msg;
{
    datalen = (msg[4] << 7) + msg[5]; /* yamaha checksum stored here */
    datalen += HDRSIZ;	/* yamaha length and checksum doesn't include  */
    offset = HDRSIZ;	/*   first part of message */
    datalen += 1;	/* include the checksum byte in the summation */
    cksum = 0;
    started = true;
}


/* checksum - process one buffer of sysex message */
/**/
void checksum(msg, len)
  byte *msg;
  long len;
{
    int n;          /* loop variable */
    int stop = min(len, datalen);
    if (!started) return;
    for (n = offset; n < stop; n++) cksum = (cksum + msg[n]) & 0x7f;
    datalen -= len;
    offset = 0;		/* sum all but first block from the beginning */
}


/* checksum_end - see if checksum and length are ok */
/**/
void checksum_end()
{
    /* final datalen is -1 because we subtracted the actual number of
	byte from the yamaha count, which does not include 
	the final end-of-sysex byte: */
    if (started && ((cksum != 0) || (datalen != -1))) {
	gprintf(ERROR, "Data has been garbled -- please start over.\n");
    }
    started = false;
}



/****************************************************************************
*               excldesc
* Inputs:
*    char *msg: the system exclusive message
* Effect: 
*    prints manufacturer and device code, if known
****************************************************************************/

void excldesc(msg)
byte *msg;
{
    static char *sizemsg = " Size : %d\n";
    int i;
    int datalen;
    if (msg[1] == YAMAHA) {
	gprintf(TRANS,"YAMAHA, ");
	datalen = (msg[4] << 7) + msg[5];
	switch (msg[3]) {
	case DX7_VOICE:
	    gprintf(TRANS,"single voice data (");
	    for (i = HDRSIZ+145; i < HDRSIZ+155; i++) {
		gprintf(TRANS,"%c",msg[i]);
	    }
	    gprintf(TRANS,")");
	    gprintf(TRANS, sizemsg, datalen);
	    checksum_start(msg);
	    break;
	case DX7_PERF:
	    gprintf(TRANS,"single performance data: ");
	    for (i = HDRSIZ+64; i < HDRSIZ+94; i++) gprintf(TRANS,"%c", msg[i]);
	    gprintf(TRANS,")");
	    gprintf(TRANS, sizemsg, datalen);
	    checksum_start(msg);
	    break;
	case DX7_PERF32:
	    gprintf(TRANS,"64 performance data");
	    gprintf(TRANS, sizemsg, datalen);
	    checksum_start(msg);
	    break;
	case DX27_VOICE:
	    gprintf(TRANS,"DX27 1 voice bulk data");
	    gprintf(TRANS, sizemsg, datalen);
	    checksum_start(msg);
	    break;
	case DX27_VOICE32:
	    gprintf(TRANS,"DX27 32 voice bulk data");
	    gprintf(TRANS, sizemsg, datalen);
	    checksum_start(msg);
	    break;
	case DX7_VOICE32:
	    gprintf(TRANS,"32 voice data");
	    gprintf(TRANS, sizemsg, datalen);
	    checksum_start(msg);
	    break;
	case DX7_PARAM:
	    gprintf(TRANS,"Parameter Change");
	    break;
	default:
	    gprintf(TRANS,"format %x", msg[3]);
	    break;
	}
    } else {
	gprintf(TRANS,"I.D.: %x", msg[1]);
    }
    gprintf(TRANS,"\n");
}


/****************************************************************************
*               isdangerous
* Inputs:
*    byte *msg: pointer message to check
* Outputs:
*    returns true if message can destroy lots of info (block data message)
****************************************************************************/

boolean isdangerous(msg)
byte *msg;
{
    /* look for 32 voice and 64 performance data msgs */
    return (boolean)(msg[1] == YAMAHA &&
		     (msg[3] == DX7_PERF32 || msg[3] == DX27_VOICE32 ||
		      msg[3] == DX27_VOICE32));
}
