#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include "switches.h"

#define FORMAT  "%lE"

/*
 * Conversion routines between various representations of pitch
 */

/*
 * Main variables
 */

typedef enum {
	Vfreq, Vsteps, Vperiod, Vpitchname, Vlast
} VariableName;

double  freq;           /* Pitch frequency, in HZ                       */
double  steps;          /* half step notation, 60 = middle C            */
double  period;         /* period, in samples */
struct pitchname {
	int class, acc, octave, approx;
} pitchname;            /* symbolic pitch,  'A' natural 4 = 440 Hz */

/*
 * Auxilary variables 
 */

int     nharm = 10;             /* number of harmonics to print out */
double  srate = 44100.0;        /* sample rate */

/*
 * Constants
 */

double p1, p2;          /* used in pitch to step conversion */
int DefaultOctave = 4;  /* default used in symbolic pitches */

void init();
void command();
void print();
void getdata(char *prompt, char *line, char *format, char *pointer);
void update();

main()
{
	init();
	for(;;)
		command();
}


void init()
{
    /* compute constants p1 and p2:
	pitchconvert(0) * 2 = pitchconvert(12)  - octaves
		exp(p2) * 2 = exp(12 * p1 + p2)
			  2 = exp(12 * p1)
		     log(2) = 12 * p1                           */

	 p1 = log(2.0)/12;

    /*     pitchconvert(69) gives 440Hz
	  exp(69 * p1 + p2) = 440
	       69 * p1 + p2 = log(440)       */

	p2 = log(440.0) - (69 * p1);

	steps = 60.0;
	update(Vsteps);
}

void command()
{
	char line[100];
	char buf[100];
	int i;
	double ftos();

	printf("? ");

	if(gets(line) == NULL)  /* EOF */
		line[0] = 'q';

	switch(line[0]) {
	case 'f':       /* freq to steps */
		getdata("freq (Hz)? ", line, FORMAT, (char *) &freq);
		update(Vfreq);
		break;

	case 's':
		getdata("half steps? ", line, FORMAT, (char *) &steps);
		update(Vsteps);
		break;

	case 'P':
		getdata("period in samples? ", line, FORMAT, (char *) &period);
		update(Vperiod);
		break;

	case 'p':
		do {
			getdata("pitch name? ", line, "%s", buf);
			pitchname.octave = DefaultOctave;
			pitchname.approx = 0;
			pitchname.class = buf[0];
			if(isupper(pitchname.class))
				pitchname.class = tolower(pitchname.class);
			buf[0] = '\0';
			line[0] = '\0'; line[1] = '\0';
		} while(! ( pitchname.class >= 'a' && pitchname.class <= 'g'));

		if(buf[1] == '#' || buf[1] == 's' ||
		   buf[1] == 'f' || buf[1] == 'b') {
			if(buf[1] == 'f' || buf[1] == 'b')
				pitchname.acc = -1;
			else
				pitchname.acc = 1;
			sscanf(&buf[2], "%d", &pitchname.octave);
		}
		else {
			pitchname.acc = 0;
			sscanf(&buf[1], "%d", &pitchname.octave);
		}
		update(Vpitchname);
		break;

	case 'h':       /* harmonics */
		if(freq == 0.0) {
			printf("freq == 0.0\n");
			break;
		}
		printf("harmonic freq            steps           period\n");
		for(i = 1; i <= nharm; i++)
			printf("%-8d %-15f %-15f %f\n",
				i, freq*i, ftos(freq*i), srate/(freq*i));
		break;

	case 'n':       /* change number of harmonics */
		getdata("number of harmonics? ", line, "%d", (char *) &nharm);
		break;

	case 'r':
		getdata("sample rate? ", line, FORMAT, (char *) &srate);
		update(Vsteps);
		break;

	 case 'v':
		update(Vsteps);
		break;

	case 'V':       /* print out current state variables */
		update(Vsteps);
		printf("freq = %f Hz, steps = %g, period = %f samples/period\n",
				freq, steps, period);
		printf("srate = %g Hz, nharm = %d\n",
			srate, nharm);

		printf("p1 = log(2.0)/12 = %f, ", p1);
		printf("p2 = log(440.0) - (69 * p1) = %f\n", p2);
		printf("steps = (log(freq) - p2)/p1, ");
		printf("freq = exp(steps * p1 + p2)\n");
		break;

	case 'q':
		exit(0);
		break;

	default:        /* help */
		printf("f - enter frequency in Hz\n");
		printf("s - enter half step number (48 = middle c)\n");
		printf("m - enter midi half step number (60 = middle c)\n");
		printf("p - enter pitch name (example: Df4 for D flat above middle C)\n");
		printf("P - enter period in samples per second\n");

		printf("\nn - change number of harmonics\n");
		printf("r - change sample rate\n");

		printf("\nh - print harmonics\n");
		printf("v - print out variables\n");

		printf("\nq - quit\n");
		break;
	}
}

double
ftos(freq)
double freq;
{
	if(freq == 0.0) {
		printf("freq == 0.0\n");
		return -999.0;
	}
	return (log(freq) - p2)/p1;
}

double
stof(steps)
double steps;
{
	return exp(steps * p1 + p2);
}

void StepsToFreq()
{
	freq = stof(steps);
}

void FreqToSteps()
{
	steps = ftos(freq);
}

void PitchNameToSteps()
{
	static cl[] = { 9, 11, 0, 2, 4, 5, 7, };
	/* this next line due to a Manx compiler bug - replacing the 
	 * index expression [clx] with [pitchname.class - 'a'] is
	 * simpler, but will not work...
	 */
	int clx = pitchname.class - 'a';

	steps = (pitchname.octave + 1) * 12 + cl[clx] +	pitchname.acc;
}

#define MANYOCTAVES     1200    /* avoid mods and int divisions by negatives */

void StepsToPitchName()
{
	int isteps = (int) floor(steps + .5);
	int imod12 = (isteps + MANYOCTAVES) % 12;
	static char pn[] = "ccddeffggaab";
	static char pa[] = "n#n#nn#n#n#n";

	pitchname.approx = (isteps != steps);
	pitchname.octave = (isteps - 12 + MANYOCTAVES)/12 - MANYOCTAVES/12;
	pitchname.class = pn[imod12];
	switch(pa[imod12]) {
	case 'n': pitchname.acc = 0;    break;
	case '#': pitchname.acc = 1;    break;
	case 'f': pitchname.acc = -1;   break;
	}
}

void FreqToPeriod()
{
	if(freq == 0.0) {
		printf("freq == 0.0\n");
		period = 9999999.;
	}
	else
		period = srate/freq;
}

void PeriodToFreq()
{
	if(period == 0.0) {
		printf("period == 0.0\n");
		freq = 9999999.;
	}
	else
		freq = srate/period;
}
	

void     FreqToSteps(), StepsToFreq();
void     FreqToPeriod(), PeriodToFreq();
void     PitchNameToSteps(), StepsToPitchName();

struct conv {
	VariableName    input;
	VariableName    output;
	void            (*convert)();
	char            *name;
} conv[] = {
	Vfreq,          Vsteps,         FreqToSteps,    "FreqToSteps",
	Vsteps,         Vfreq,          StepsToFreq,    "StepsToFreq",
	Vfreq,          Vperiod,        FreqToPeriod,   "FreqToPeriod",
	Vperiod,        Vfreq,          PeriodToFreq,   "PeriodToFreq",
	Vpitchname,     Vsteps,         PitchNameToSteps,       "PitchNameToSteps",
	Vsteps,         Vpitchname,     StepsToPitchName,       "StepsToPitchName",
	Vlast,          Vlast,          NULL,           NULL
};

#define VLAST   ((int) Vlast)
#define DONE(v) done[(int) (v)]

void update(v)
VariableName v;
{
	int     done[VLAST];
	int     i;
	int     nchanged;

	/* invalidate all but the one that has just been set */
	for(i = 0; i < VLAST; i++)
		DONE(i) = 0;
	DONE(v) = 1;
	/* printf("changed Var %d\n", (int) v); */

	do {
		nchanged = 0;
		for(i = 0; conv[i].convert; i++) {
			if(DONE(conv[i].input) && ! DONE(conv[i].output)) {
				(*conv[i].convert)();
				DONE(conv[i].output) = 1;
				nchanged++;
				/*
				print();
				printf("Executed %s\n", conv[i].name);
				*/
			}
		}
	} while(nchanged != 0);

	for(i = 0; i < VLAST; i++)
		if( ! DONE(i) )
			printf("Var %d not computed!!\n", i);

	print();
}

void print()
{
	char buf[10];

	printf("pitch  steps      freq       Period     rate       nharm\n");
	sprintf(buf, "%s%c%s%d",
		pitchname.approx ? "~" : "",
		pitchname.class,
		pitchname.acc < 0 ? "f" : pitchname.acc == 0 ? "" : "#",
		pitchname.octave);
	printf("%-4s   ", buf);
	printf("%-10.5g ", steps);
	printf("%-10.5g ", freq);
	printf("%-10.5g ", period);
	printf("%-10.5g ", srate);
	printf("%d", nharm);
	printf("\n");
}

void getdata(prompt, line, format, pointer)
  char *prompt, *line, *format, *pointer;
{
	line++;
	while(sscanf(line, format, pointer) != 1) {
		printf("%s", prompt);
		gets(line);
	}
}
