/* mapper -- map midi events into output sequences */

#include "switches.h"
#include "cext.h"
#include "stdio.h"
#include "midifns.h" /* to get time_type */
#include "timebase.h"
#include "moxc.h"
#include "userio.h"
#include "scan.h"
#include "cmdline.h"
#include "seq.h"
#include "mapper.h"
#include "ctype.h"
#include "string.h"

/* we're already taking precautions, so inline version of toupper is ok: */
/* AZTEC C version of _toupper (the inline form) is broken */
#ifndef toupper
#define toupper(x) ((x)-'a'+'A')
#endif

#define clip(v, lo, hi) ((v) < (lo) ? (lo) : ((v) > (hi) ? (hi) : (v)))

#define stack_depth 16

char stacks[17][stack_depth];   /* actual storage for map stacks */
char *active_maps[17];          /* pointers into map stacks */

map_type maps[26];

void map_play_init();
boolean mapevt_is_free();
mapevt_type mapevt_new();
void mapper_play();
void wait_for_noteoff();
void wait_for_noteoff_score();
void future_wrapup();

map_type map_free_list = NULL;

#define score_max 100
seq_type score_table[score_max];
char *score_names[score_max];
char score_valid[score_max];
int score_num = 0;

/* A pool of seq_type objects is kept so that they will be preallocated
 * when it is time to play a score.  The seq_busy array tells whether or
 * not each member of seq_pool is busy.  The seq_busy flag is cleared by
 * the "at_end" method of each sequence.
 */
#define max_seq 100
seq_type seq_pool[max_seq];
char seq_busy[max_seq];

void free_player(seq_type score);


/* find_player -- looks for or allocates a free seq_type to play a score */
/**/
seq_type find_player()
{
    int i;
    for (i = 0; i < max_seq; i++) {
	if (!seq_busy[i]) {
	    if (!seq_pool[i]) {
		if (seq_pool[i] = seq_init(seq_alloc(), false)) {
		    seq_at_end(seq_pool[i], free_player);
		}
	    }
	    seq_busy[i] = true;
	    /* gprintf(TRANS, "seq %d busy\n", i); */
	    return seq_pool[i];
	}
    }
    return NULL;
}


/* free_player -- called when score finishes so we can recycle score struct */
/*
 * NOTE: since there is no provision for passing a parameter to this routine
 * other than the seq itself, we must search to find the index of this seq
 * in seq_pool so that we can clear the corresponding member of seq_busy.
 * (This should be fixed in seq.c, seq.h!)
 */
void free_player(score)
  seq_type score;
{
    int i;
#ifdef CYCLEBUG
    /* check cycleflag, if set then cycle instead of free */
    if (score->cycleflag) {
	seq_reset(score);
	set_virttime(score->timebase, 1000);
	seq_play(score);
    } else {
#endif
	/* fix refcount so chunks can be freed later */
	seq_free_chunks(score);
	/* find the score in the seq_pool and mark it free */
	for (i = 0; i < max_seq; i++) {
	    if (seq_pool[i] == score) {
		seq_busy[i] = false;
		/* gprintf(TRANS, "seq %d free\n", i); */
		return;
	    }
	}
#ifdef CYCLEBUG
    }
#endif
}


/* map_create -- allocate and create a new map */
/**/
map_type map_create() 
{
    int i;
    map_type result = (map_type) memget(sizeof(map_node));
    result->chan = 0;
    for (i = 0; i < map_len; i++) result->keymap[i] = NULL;
    return result;
}


/* mapevt_is_free -- test to see if mapevt is already freed */
/*
 * NOTE: the CLASS designation in the mapper input language says
 * to use the same set of mapevt's for each octave, so we end up
 * with multiple pointers to the same mapevt list.  To avoid freeing
 * the same list 3 times, this routine is called to detect when a
 * mapevt is already on the freelist.
 */
boolean mapevt_is_free(mapevt)
  mapevt_type mapevt;
{
    register mapevt_type p = (mapevt_type) 
	(mem_free_list + ((sizeof(mapevt_node) - 1) >> 2));
    while (p) {
	if (mapevt == p) return true;
	p = p->next;
    }
    return false;
}


/* mapevt_free_a_list -- free a list if non-null and not shared */
/**/
void mapevt_free_a_list(mapevt_ptr)
  mapevt_type *mapevt_ptr;
{
    while (*mapevt_ptr) {
	mapevt_type mapevt = *mapevt_ptr;
	if (!mapevt_is_free(mapevt)) {
	    *mapevt_ptr = mapevt->next;
	    mapevt_free(mapevt);
	} else return;
    }
}


/* noteoff_type -- records a pending noteoff.  These structures are
 *   hung onto future_nodes and are processed and freed when the
 *   duration of the event is determined.
 */
typedef struct noteoff_struct {
    struct noteoff_struct *next;        /* next in the list */
    short note_not_score;                       /* note or score? */
    long start;                         /* when this note started */
    short chan;                         /* channel of this note */
    short key;                          /* key number of this note */
    union {
	long dur;                       /* the duration field from NT cmd */
	seq_type score;                 /* the score to be cut off */
    } u;
} noteoff_node, *noteoff_type;


/* A future_node is used to represent information about the performance
 * of one note.  A future_node is created for each note event and it 
 * waits for the corresponding note-off.  It is used to remember that 
 * other notes must be turned off when the note-off arrives.  It also
 * remembers the duration of the note so that event sequences that 
 * were triggered by the note event have a duration to reference even
 * long after the note-off occured.
 */
typedef struct future_struct {
    struct future_struct *next;
    short key_chan;     /* used to identify this event */
    short ref_count;    /* how many references are outstanding */
    boolean sounding;   /* initially true, false after note-off */
    long start;         /* start time of the note */
    long dur;           /* duration, set when note-off received */
    noteoff_type pending;       /* list of pending note-off's */
} future_node, *future_type;

future_type future_list;

#define future_alloc() (future_type) memget(sizeof(future_node))

/* future_create -- create a future */
/* 
 * NOTE: result is at head of future list (see map_play_init())
 */
future_type future_create(key_chan)
{
    register future_type result = future_alloc();
    result->key_chan = key_chan;
    result->ref_count = 2;      /* one reference for mapper_play, one for
					future_list */
    result->sounding = true;
    result->start = virttime;
    result->pending = NULL;
    /* add to head of future_list: */
    result->next = future_list;
    future_list = result;
    return result; 
}


/* future_free -- free the future type by adding to free list */
/**/
void future_free(future)
  future_type future;
{
    if (--future->ref_count) return;
    memfree((char *) future, sizeof(future_node));
}


/* map_destroy -- deallocate a map, including its events */
/**/
void map_destroy(map)
  map_type map;
{
    int i;
    for (i = 0; i < map_len; i++) {
	mapevt_free_a_list(&(map->keymap[i]));
    }
    map_free(map);
}

/* decode_pitch -- perform pitch relative processing */
/*
 * inputs are the pitch specification (pitch) and the note's future
 * structure, which contains the actual pitch played (pitch).  Bits
 * in pitch say whether or not this is absolute or an offset.  The result
 * is not clipped!
 */
int decode_pitch(pitch, future)
  register int pitch;
  register future_type future;
{
    if (pitch & mapevt_transpose_flag) {
	if (pitch & mapevt_sign) {
	    /* extend sign */
	    pitch |= ~mapevt_number;
	} else pitch &= mapevt_number;  /* mask off flags */
	pitch += (future->key_chan & 0x7F);
    } 
    return pitch;
}


/* decode_trans -- decode a transposition */
/*
 * inputs are the transposition specification (trans) and the note's future
 * structure, which contains the actual pitch played (pitch).  Bits
 * in pitch say whether or not this is absolute or an offset.  The result
 * is not clipped!  See mapread.c: parse_trans() for encoding details.
 */
int decode_trans(trans, future)
  register int trans;
  register future_type future;
{
    if (trans & mapevt_transpose_flag) {
	/* high order bits (not including mapevt_sign) must be ones */
	trans |= ~(mapevt_sign | mapevt_number);
	trans += (future->key_chan & 0x7F);
    } else if (trans & mapevt_sign) {
	trans |= ~mapevt_number;
    } else trans &= mapevt_number;
    return trans;
}


/* decode_vel -- perform velocity relative processing */
/*
 * inputs are the velocity specification (vel_spec) and the actual
 * velocity played (velocity).  Bits in vel_spec say whether or not
 * this is absolute or an offset.  The result is not clipped!
 */
int decode_vel(vel_spec, velocity)
  register int vel_spec;
  register int velocity;
{
    if (vel_spec & mapevt_transpose_flag) {
	/* sign extend if it's negative */
	if (vel_spec & mapevt_sign) vel_spec |= ~mapevt_number;
	/* or clear the flag bits if it's positive */
	else vel_spec &= mapevt_number;
	vel_spec = (vel_spec << 1) + velocity;
    } else vel_spec = vel_spec << 1;
    return vel_spec;
}


static int map_event_nesting = 0;
static char last_map = ' ';

/* map_event -- map an input event to output events */
/**/
void map_event(chan, key, vel, level, future)
  int chan;
  int key;
  int vel;
  int level;  /* the level at which to start searching for mapping */
  future_type future;   /* set to nil unless future already exists */
{
    map_type map = NULL;
    int key_chan = (chan << 8) + key;
    if (!future) future = future_create(key_chan);
    if (++map_event_nesting > 10) {
	gprintf(TRANS, 
	    "REDO loop detected and terminated: channel %d, key %d, map %c\n",
	    chan, key, last_map);
	map_event_nesting--;
	future_free(future);  /* lose the reference */
	return;
    }
    last_map = *active_maps[chan];
    switch (level) {
      case 0: /* look for a pitch specific, channel specific handler */
	if ((last_map != EOS) && (map = maps[last_map - 'A'])) {
	    mapevt_type cmds = map->keymap[key % 36];
	    if (cmds) {
		mapper_play(cmds, future, vel, 0);
		break;
	    }
	}
      case 1: /* look for a channel specific, pitch non-specific handler */
	if (map || (last_map != EOS && (map = maps[last_map - 'A']))) {
	    mapevt_type cmds = map->keymap[map_any];
	    if (cmds) {
		mapper_play(cmds, future, vel, 1);
		break;
	    }
	}
      case 2: /* look for a channel non-specific, pitch specific handler */
	if ((last_map = *active_maps[0]) && (map = maps[last_map - 'A'])) {
	    mapevt_type cmds = map->keymap[key % 36];
	    if (cmds) {
		mapper_play(cmds, future, vel, 2);
		break;
	    }
	}
      case 3: /* look for a handler for all channels and any pitch */
	if (map || ((last_map = *active_maps[0]) &&
	      (map = maps[last_map - 'A']))) {
	    mapevt_type cmds = map->keymap[map_any];
	    if (cmds) {
		mapper_play(cmds, future, vel, 3);
		break;
	    }
	}
      case 4:
	future_free(future);    /* give up single reference */
	break;  /* this it the level you give to do nothing */
      default:
	gprintf(TRANS, "bad level parameter to map_event\n");
	break;
    }    
    map_event_nesting--;
}


/* map_install -- activate the map denoted by letter */
/**/
void map_install(letter)
  char letter;
{
    int map_id;
    map_type map;
    int chan;
    if (!isalpha(letter)) return;
    if (islower(letter)) letter = toupper(letter);
    map_id = letter - 'A';
    map = maps[map_id];
    if (!map) return;
    chan = map->chan;
    active_maps[chan] = &(stacks[chan][stack_depth - 1]);
    *active_maps[chan] = EOS;
    active_maps[chan]--;
    *active_maps[chan] = letter;
    map_play_init(maps[letter - 'A']);
}


/* map_play_init -- play the INIT command list of a map */
/*
 * NOTE: the INIT command list is played when the map is installed.
 */
void map_play_init(map)
  map_type map;
{
    /* use middle C for pitch, map->chan may be zero, but that's ok because
	the channel will not be used by anyone because PASS and REDO commands
	are not allowed in INIT command lists (see mapread.c)
     */
    int key_chan = (map->chan << 8) + 60;
    future_type future = future_create(key_chan);
    /* NOTE: now we've allocated a future and it's sitting around waiting for
     * a noteoff that matches key_chan, but (1) no noteoff will be generated
     * because the note was artificially generated, and (2) there might be
     * a real note with the same pitch and channel, so we've created a
     * potentially ambiguous situation.  We must therefore immediately
     * simulate a corresponding note-off.  This happens in about 11 more lines:
     */
    mapevt_type cmds;

    /* simulate knowledge of duration: */
    future->dur = 100;
    future->sounding = false;
    /* DANGER!: assume that the first item on future_list is future (this will
	be true because future_alloc puts new things at the head of its list
	and future is the most recent result of future_alloc.
     */
    future_list = future_list->next;
    if (!map || !(cmds = map->keymap[map_init])) return;
    mapper_play(cmds, future, 100, 3);  /* level 3 not really necessary because
					   we can't PASS */
    future_free(future);
}


/* mapevt_insert_call -- insert a call command into an event list */
/**/
void mapevt_insert_call(mapevt_ptr, letter)
  mapevt_type *mapevt_ptr;
  char letter;
{
    register mapevt_type mapevt = mapevt_new(mapevt_ptr);
    mapevt->cmd = mapevt_call;
    mapevt->value = letter;
}


/* mapevt_insert_dly -- insert a delay command into an event list */
/**/
void mapevt_insert_dly(mapevt_ptr, del)
  mapevt_type *mapevt_ptr;
  long del;
{
    register mapevt_type mapevt = mapevt_new(mapevt_ptr);
    mapevt->cmd = mapevt_dly;
    mapevt->u.dur = del;
}


/* mapevt_insert_go -- insert a go command into an event list */
/**/
void mapevt_insert_go(mapevt_ptr, chan, letter)
  mapevt_type *mapevt_ptr;
  char letter;
{
    register mapevt_type mapevt = mapevt_new(mapevt_ptr);
    mapevt->cmd = mapevt_go;
    mapevt->chan = chan;
    mapevt->value = letter;
}


/* mapevt_insert_nt -- insert a note command into an event list */
/**/
void mapevt_insert_nt(mapevt_ptr, chan, key, vel, dur)
  mapevt_type *mapevt_ptr;
  int chan;
  int key;
  int vel;
  long dur;
{
    register mapevt_type mapevt = mapevt_new(mapevt_ptr);
    mapevt->cmd = mapevt_nt;
    mapevt->chan = chan;
    mapevt->value = key;
    mapevt->vel = vel;
    mapevt->u.dur = dur;
}



/* mapevt_insert_prog -- insert a prog event into an event list */
/**/
void mapevt_insert_prog(mapevt_ptr, chan, prog)
  mapevt_type *mapevt_ptr;
  int chan;
  int prog;
{
    register mapevt_type mapevt = mapevt_new(mapevt_ptr);
    mapevt->cmd = mapevt_prog;
    mapevt->chan = chan;
    mapevt->value = prog;
}


/* mapevt_insert_print -- insert a print event into an event list */
/**/
void mapevt_insert_print(mapevt_ptr, str)
  mapevt_type *mapevt_ptr;
  char *str;
{
    register mapevt_type mapevt = mapevt_new(mapevt_ptr);
    mapevt->cmd = mapevt_print;
    mapevt->u.msg = str;
}


/* mapevt_insert_ctrl -- insert a ctrl command into an event list */
/**/
void mapevt_insert_ctrl(mapevt_ptr, chan, ctrl, value)
  mapevt_type *mapevt_ptr;
  int chan;
  int ctrl;
  int value;
{
    register mapevt_type mapevt = mapevt_new(mapevt_ptr);
    mapevt->cmd = mapevt_ctrl;
    mapevt->chan = chan;
    mapevt->value = ctrl;
    mapevt->vel = value;
}


/* mapevt_insert_ret -- insert a ret command into an event list */
/**/
void mapevt_insert_ret(mapevt_ptr, chan)
  mapevt_type *mapevt_ptr;
  int chan;
{
    register mapevt_type mapevt = mapevt_new(mapevt_ptr);
    mapevt->cmd = mapevt_ret;
    mapevt->chan = chan;
}

/* mapevt_insert_noop -- insert a noop command into an event list */
/**/
void mapevt_insert_noop(mapevt_ptr)
  mapevt_type *mapevt_ptr;
{
    register mapevt_type mapevt = mapevt_new(mapevt_ptr);
    mapevt->cmd = mapevt_noop;
}


/* mapevt_insert_pass -- insert a pass command into an event list */
/**/
void mapevt_insert_pass(mapevt_ptr)
  mapevt_type *mapevt_ptr;
{
    register mapevt_type mapevt = mapevt_new(mapevt_ptr);
    mapevt->cmd = mapevt_pass;
}


/* mapevt_insert_redo -- insert a redo command into an event list */
/**/
void mapevt_insert_redo(mapevt_ptr)
  mapevt_type *mapevt_ptr;
{
    register mapevt_type mapevt = mapevt_new(mapevt_ptr);
    mapevt->cmd = mapevt_redo;
}


/* mapevt_insert_score -- insert a score command into an event list */
/**/
void mapevt_insert_score(mapevt_ptr, score_id, trans, vel, rate)
  mapevt_type *mapevt_ptr;
  int score_id;
  int trans;
  int vel;
  int rate;
{
    register mapevt_type mapevt = mapevt_new(mapevt_ptr);
    mapevt->cmd = mapevt_score;
    mapevt->value = trans;
    mapevt->vel = vel;
    mapevt->gate_and_rate = rate;
    mapevt->u.score_id = score_id;
}


/* mapevt_new -- create an event and link it into the end of event list */
mapevt_type mapevt_new(mapevt_ptr)
  mapevt_type *mapevt_ptr;
{
    while (*mapevt_ptr) mapevt_ptr = &((*mapevt_ptr)->next);
    *mapevt_ptr = mapevt_alloc();
    (*mapevt_ptr)->next = NULL;
    return *mapevt_ptr;
}


/* mapper_find_score -- find score in cache or load from file */
/**/
int mapper_find_score(name)
  char *name;
{
    register int i;
    FILE *score_file = NULL;
    register seq_type *score_table_entry;
    char *suffix;
    for (i = 0; i < score_num; i++) {
	if (strcmp(score_names[i], name) == 0) {
	    if (score_valid[i]) return i;
	    else {
		score_table_entry = &(score_table[i]);
		goto read_ith_score;
	    }
	}
    }
    if (i == score_max) {
	gprintf(TRANS, "Maximum number of scores %d exceeded!\n", score_max);
	return 0;
    }
    score_table_entry = &(score_table[i]);
    *score_table_entry = NULL;
    score_num++;
read_ith_score:
    if (*score_table_entry) {
	seq_free(*score_table_entry);
	gprintf(TRANS, "freed score entry %d\n", i);
    }
    *score_table_entry = seq_create();
    score_names[i] = memget(strlen(name) + 1);
    gprintf(TRANS, "%s is score %d\n", name, i);
    score_valid[i] = true;
    strcpy(score_names[i], name);
    score_file = fileopen(name, "gio", "r", "input adagio file");
    suffix = strrchr(fileopen_name, '.');
    if (suffix && (strcmp(suffix, ".gio") == 0)) {
	seq_read(*score_table_entry, score_file);
    } else {
#ifndef UNIX
	/* oops - reopen as a binary file */
	fclose(score_file);
	score_file = fopen(fileopen_name, "rb");
#endif
	seq_read_smf(*score_table_entry, score_file);
    }
    fclose(score_file);
    return i;
}


/* mapper_flush_scores -- invalidate the cache of scores */
/**/
void mapper_flush_scores()
{
    int i;
    for (i = 0; i < score_num; i++) score_valid[i] = false;
}


/* mapper_init -- set up data structures for this module */
/**/
void mapper_init()
{
    int i;
    for (i = 0; i < 17; i++) {
	active_maps[i] = &(stacks[i][stack_depth - 1]);
	*active_maps[i] = EOS;
    }
    for (i = 0; i < 26; i++) {
	maps[i] = NULL;
    }
    for (i = 0; i < max_seq; i++) {
	seq_pool[i] = NULL;
	seq_busy[i] = false;
    }
}


/* mapper_play -- play an event list */
/*
 * NOTE: some events cannot be scheduled because they depend upon when
 *  the note ends.  These events are linked onto the future, which will
 *  be evaluated when the note-off arrives.
 */
void mapper_play(cmds, future, velocity, level)
  mapevt_type cmds;
  future_type future;
  int velocity;
  int level;
{
    seq_type score;
    while (cmds) {
	switch (cmds->cmd) {
	  case mapevt_prog:
	    midi_program(cmds->chan, cmds->value);
	    break;
	  case mapevt_ctrl:
	    midi_ctrl(cmds->chan, cmds->value, cmds->vel);
	    break;
	  case mapevt_dly:
	    cause(cmds->u.dur, mapper_play, 
		cmds->next, future, velocity, level);
	    return;     /* note this is not a break so we don't loop again,
			   ref_count of future stays the same because we
			   are passing the reference on through cause
			 */
	  case mapevt_nt: {
	    int pitch = decode_pitch(cmds->value, future);
	    int vel = decode_vel(cmds->vel, velocity);

	    pitch = clip(pitch, 0, 127);
	    vel = clip(vel, 1, 127);

	    midi_note(cmds->chan, pitch, vel);
	    if (cmds->u.dur & duration_transpose_flag) {
		wait_for_noteoff(future, cmds->chan, pitch, cmds->u.dur);
	    } else {
		cause(cmds->u.dur, midi_note, cmds->chan, pitch, 0);
	    }
	    break;
	  }
	  case mapevt_call: {
	    map_type map = maps[cmds->value - 'A'];
	    if (map) {
		char **stack_ptr = &(active_maps[map->chan]);
		if (strlen(*stack_ptr) + 1 < stack_depth) {
		    (*stack_ptr)--;
		    **stack_ptr = cmds->value;
		} else gprintf(ERROR, "map stack overflow on chan %d: %s\n",
				    map->chan, *stack_ptr);
	    } else gprintf(ERROR, "call %c: map not defined\n", cmds->value);
	    break;
	  }
	  case mapevt_ret: {
	    char **stack_ptr = &(active_maps[cmds->chan]);
	    if (**stack_ptr != EOS) (*stack_ptr)++;     /* pop the stack */
	    else gprintf(ERROR, "bogus map return\n");
	    break;
	  }
	  case mapevt_go:
	    *(active_maps[cmds->chan]) = cmds->value;
	    map_play_init(maps[cmds->value - 'A']);
	    break;
	  case mapevt_pass:
	    future->ref_count++;        /* about to copy reference */
	    map_event(future->key_chan >> 8, future->key_chan & 0x7f,
		      velocity, level + 1, future);
	    break;
	  case mapevt_print:
	    gprintf(TRANS, "%s\n", cmds->u.msg);
	    break;
	  case mapevt_redo:
	    future->ref_count++;        /* about to copy reference */
	    map_event(future->key_chan >> 8, future->key_chan & 0x7f,
		      velocity, 0, future);
	    break;
	  case mapevt_noop:
	    break;
	  case mapevt_score: {
	    int transpose = decode_trans(cmds->value, future);
	    int level = decode_vel(cmds->vel, velocity);
	    time_type rate = cmds->gate_and_rate & mapevt_rate_mask;

	    if (rate <= 0) rate = 1;
	    rate = (100 << 8) / rate;

	    /* gprintf(TRANS, "start score: transpose %d level %d rate %ld\n",
		transpose, level, rate); */

	    score = find_player();
	    if (!score) {
		gprintf(TRANS, "Out of players!\n");
	    } else {
		seq_type source = score_table[cmds->u.score_id];
		score->chunklist = source->chunklist;
		score->chunklist->u.info.refcount++;
		score->current = seq_events(score);
		/* gprintf(TRANS, "source %lx score %lx current %lx\n",
			   source, score, score->current); */
		/* score->noteoff_count = 0; - done by seq_reset */
		seq_set_rate(score, rate);
		/* set_virttime(score->timebase, 0L); - done by seq_reset */
		seq_set_transpose(score, transpose);
		seq_set_loudness(score, level);
		seq_reset(score);
		seq_play(score);
		score->cycleflag = false; /* just to be safe */
		if (cmds->gate_and_rate & mapevt_gate_flag) {
		    wait_for_noteoff_score(future, score);
		    if (cmds->gate_and_rate & mapevt_cycle_flag) {
#ifdef CYCLEBUG
			score->cycleflag = true;
#else
			/* 0L says recycle immediately after last noteoff */
			seq_cycle(score, true, 0L);
#endif
		    } else {
			/* since wait_for_noteoff_score will call
			 * free_player, let's not do it twice!
			 * NOTE: the score will remain allocated until
			 * the note-off whether or not score is finished.
			 */
			seq_at_end(score, noop);
		    }
		}
	    }
	    break;
	  }
	  default:
	    gprintf(ERROR, "bad mapevt command type\n");
	}
	cmds = cmds->next;
    }
    future_free(future);
}


/* mapper_show_active -- show what maps are bound to each channel */
/**/
void mapper_show_active()
{
    int i;
    for (i = 0; i <= 16; i++) {
	gprintf(TRANS, "%d: %s\n", i, active_maps[i]);
    }
}


/* wait_for_noteoff -- hang a record on a future to process note off */
/**/
void wait_for_noteoff(future, out_chan, out_key, out_dur)
  future_type future;   /* the future for the activating note */
  int out_chan;         /* channel for the future note-off */
  int out_key;          /* key for the future note-off */
  long out_dur;         /* duration for the future note-off */
{
    register noteoff_type noteoff = noteoff_alloc();
    noteoff->note_not_score = true;
    noteoff->start = virttime;
    noteoff->chan = out_chan;
    noteoff->key = out_key;
    noteoff->u.dur = out_dur;
    noteoff->next = future->pending;
    future->pending = noteoff;
    if (!future->sounding) future_wrapup(future);
/*    gprintf(TRANS, "waiting for: %x\n", key_chan); */
}


/* wait_for_noteoff_score -- hang event to stop score onto a future */
/**/
void wait_for_noteoff_score(future, score)
  future_type future;   /* the future to wait on */
  seq_type score;       /* the score to be stopped on note-off */
{
    register noteoff_type noteoff = noteoff_alloc();
    noteoff->note_not_score = false;
    noteoff->u.score = score;
    noteoff->next = future->pending;
    future->pending = noteoff;
    if (!future->sounding) future_wrapup(future);
/*    gprintf(TRANS, "waiting for: %x\n", key_chan); */
}


/* map_noteoff -- process a note-off by activating future */
/**/
void map_noteoff(chan, key)
  int chan;
  int key;
{
    register key_chan = (chan << 8) + key;
    register future_type *future_ptr = &future_list;
/*    gprintf(TRANS, "map_future: "); */
    while (*future_ptr) {
	future_type future = *future_ptr;
/*      gprintf(TRANS, "%x ", future->key_chan); */
	if (future->key_chan == key_chan) {
	    future->dur = virttime - future->start;
	    future->sounding = false;
	    future_wrapup(future);
	    *future_ptr = future->next;
	    future_free(future);
	} else future_ptr = &(future->next);
    }
/*    gprintf(TRANS, "\n"); */
}
	

/* future_wrapup -- called when duration of note is known and there
 *   may be pending noteoff's waiting for this information 
 */
void future_wrapup(future)
  register future_type future;
{
    register noteoff_type noteoff = future->pending;

    while (noteoff) {
	if (noteoff->note_not_score) {
	    register dur = noteoff->u.dur;    
	    if (dur & duration_product_flag) {
		/* clear the product bit */
		dur = dur & duration_duration_field;
		/* multiply source note duration by dur percent */
		dur = (future->dur * dur) / 100;
	    } else { /* not product so must be transpose (+/-) */
		/* get the offset */
		if (dur & duration_sign) {
		    /* sign extend: */
		    dur |= duration_sign_extend;
		} else {
		    /* clear the transpose (+/-) bit */
		    dur = dur & duration_duration_field;
		}
		/* add offset to duration of event */
		dur += future->dur;
	    }
	    dur += noteoff->start;      /* convert to ending time */
	    dur -= virttime;    /* convert to delay */

	    if (dur > 0) cause(dur, midi_note, noteoff->chan, noteoff->key, 0);
	    else midi_note(noteoff->chan, noteoff->key, 0);
	} else {        /* must be time to stop a score */
	    register seq_type score = noteoff->u.score;
	    seq_reset(score);
	    score->cycleflag = false;
	    free_player(score);
	}
	/* now free this pending noteoff and prepare to loop */
	future->pending = noteoff->next;
	noteoff_free(noteoff);
	noteoff = future->pending;
    }
}
