/* smqdemo2.c -- demonstrate output processing for "Still More Questions" */

/***************************************************************************/
/*
This program is derived from Still More Questions, which was written by
Roger Dannenberg in 1985 and first performed that year at UCSD.  My goal
in this composition was to have a computer listen to a performance, do
some analysis of various high-level qualities, and use the resulting
parameters to perform timbre selection.  In this example, I have extracted
code that performs timbre change.  There is no connection to the input
processing (see smqdemo.c), nor is are timbres actually selected as in
the composition.  This would only be possible if the proper synthesizer,
patches, and effects processor were available.  Please contact the composer
if you wish to make performance arrangements.

The problem addressed by this particular program is that of changing 
timbres via MIDI program change commands without affecting notes in 
progress.  (Normally, if a note is sounding and a program change command
is sent, a synthesizer will abruptly stop the note.  This proves to be
very annoying and unacceptable for performance.)

There is no "perfect" solution to this problem.  The approach taken here
is based on the classic readers/writers problem from the operating systems
literature.  In MIDI terms, the idea is that if a program change is
requested while a note is sounding, the program change is delayed until
the note is finished.  To avoid postponing the program change forever, 
new note requests are ignored once a program change is requested.  After
the program change takes effect, new notes are allowed again.  This 
protocol is performed independently on each channel.

A program change request for channel C is indicated by setting 
prg_request[C] to the desired program.  A value of zero indicates
no request.

To keep track of things, if pitch P is playing on channel C, then 
bit C of notes_sounding[P] will be set.  The bit is cleared when
the note is turned off.  Also, note_count[C] keeps track of how 
many notes are sounding on channel C.

To test the program, random notes are generated on channels 1-4, and 
random program changes, choosing from programs 1-10, are generated on
the four channels as well.  Although monophonic melodies are generated,
the technique is designed to work with polyphonic output as well.

*/

#include "stdio.h"
#include "musiprog.h"

void display();
void send_program();
void melody();
void timbre();

char *app_syntax = "";

/* how many different program changes to use, numbered 1..nprograms */
#define nprograms 5

/****************************************************************/
/*  This part of the program implements the functions my_note() and
my_program(), which should be called in place of midi_note() and 
midi_program().  As explained above, these new functions avoid
changing midi programs while a note is playing.
*/

#define nchans 4
/* NOTE: since channels are numbered 1-16 (not 0-15), we'll often declare
 * arrays of size nchans+1 and not use the first element at subscript [0]
 */

/* for the visual display, we'll keep track of the current program
 * on each channel and how many notes are turned on.  notes_on[] is
 * similar to note_cnt[] below, except notes_on[] is decremented as
 * soon as the note is turned off, while note_cnt[] is decremented
 * a while after the note-off message to allow for note decay.
 * notes_on[] is used only for the display printout.
 */
int current_program[nchans+1];
int notes_on[nchans+1];

short notes_sounding[128];

int note_cnt[nchans+1];	/* number of notes on channel */
int prg_request[nchans+1];	/* desired program change */

/* dec_note_cnt -- update to reflect a note-off */
/*
 * NOTE: calls to dec_note_cnt are delayed to allow for note decay.
 */
void dec_note_cnt(channel)
  int channel;	/* the channel of the note off */
{
    note_cnt[channel]--;	/* count total note-on's */
    /* if all notes on channel are off, and if we're waiting
     * to change the program, do it
     */
    if (note_cnt[channel] == 0 && prg_request[channel]) { 
	send_program(channel);
    }
}


void my_note(channel, pitch, velocity)
  int channel, pitch, velocity;
{
    /* bit representation of channel number: */
    int channel_bit = 1 << (channel - 1);
    /* note-on will have non-zero velocity */
    if (velocity != 0) {
	if (prg_request[channel]) {
	    /* ignore note request if program request is pending */
	} else if (notes_sounding[pitch] & channel_bit) {
	    /* ignore request if a note is already on same pitch and channel */
	} else {
	    note_cnt[channel]++;	/* count total note-on's */
	    notes_sounding[pitch] |= channel_bit;	/* keep track of each note */
	    midi_note(channel, pitch, velocity);
	    notes_on[channel]++;	/* this count is for the display */
	    display();
	}
    } else {	/* note-off request */
	/* only do something if a corresponding note is on: */
	if (notes_sounding[pitch] & channel_bit) {
	    midi_note(channel, pitch, 0);
	    notes_on[channel]--;
	    display();
	    /* clear the bit, marking note as off */
	    notes_sounding[pitch] &= ~channel_bit;
	    /* Tricky code here: if we decremented note_cnt[channel],
	     * then it would look as though no note were playing, but
	     * if fact, there is still some note decay going on.  We
	     * don't want to change programs even during note decay,
	     * so schedule a note_cnt decrement a half second in the
	     * future.  (Half a second is arbitrary and should be an
	     * upper bound on decay time.
	     */
	    cause(50, dec_note_cnt, channel);
	}
    }
}


/* my_program -- request a program change */
/*
 * NOTE: the program change will be deferred if a note is sounding on
 * channel.
 */
void my_program(channel, program)
  int channel, program;
{
    prg_request[channel] = program;
    /* if no notes are sounding, do the change right away */
    if (note_cnt[channel] == 0) send_program(channel);
    /* otherwise, send_program will be scheduled by my_note when the
     * last note is turned off on this channel */
}


/* send_program -- send a program change and update data structures */
/*
 * NOTE: this should not be called except from within my_pgrm()
 * and my_note()
 */
void send_program(channel)
  int channel;
{
    midi_program(channel, prg_request[channel]);
    current_program[channel] = prg_request[channel];
    display();
    prg_request[channel] = 0;
}


/****************************************************************/
/* This part of the program exercises the functions my_note() and 
my_program() by generating melodies and program changes.
*/


/* display -- show the current program and whether a note is playing */
/**/
void display()
{
    int i;
    for (i = 1; i <= nchans; i++) {
    gprintf(TRANS, "%2d:%c ", 
	current_program[i], (notes_on[i] ? '*' : ' '));
    }
    gprintf(TRANS, "\r");
}


/* mainscore -- the initial function called after moxc starts */
/**/
void mainscore()
{
    int i;
    /* initialize data structures */
    for (i = 0; i < 128; i++) notes_sounding[i] = 0;
    for (i = 0; i <= nchans; i++) {
	note_cnt[i] = 0;
	current_program[i] = 1;
	notes_on[i] = 0;
    }

    /* start melodies and timbre selection processes: */
    for (i = 1; i <= nchans; i++) {
	/* spread out the pitch centers */
	int center = 24 + 12 * i;
	melody(i, center, center);
	timbre(i);
    } 
}


/* melody -- generate a random, monophonic melody */
/**/
void melody(channel, pitch, center)
  int channel;	/* the channel for this melody */
  int pitch;    /* the pitch */
  int center;	/* the rough pitch center of the melody */
{
    /* Generate the duration of the note.  This method will emphasize
     * small intervals, thus quasi-poisson distribution */
    long duration = random(11, random(11, random(11, 500)));

    /* Generate a new pitch at some random interval from the 
     * previous one.  This particular method will emphasize
     * small intervals, thus quasi-brownian motion.
     */
    short range = random(-7, 7);
    pitch += random((short) -range, range);

    /* bias the choice toward the pitch center occasionally (1 out of 10) */
    if (random(0, 9) == 0) {
	if (pitch < center) pitch++;
	else pitch--;
    }

    /* make sure pitch doesn't get out of range */
    while (pitch < 0) pitch += 12;
    while (pitch >= 128) pitch -= 12;

    /* play the note, make velocity somewhat random too */
    my_note(channel, pitch, random(80, 100));
    /* turn the note off after duration */
    cause(duration, my_note, channel, pitch, 0);
    /* schedule next note of the melody: */
    /* every so often, rest a bit (accomplished by increasing the delay
     * to the next melody note:
     */
    if (random(0, 4) == 0) {
	duration += random(20, random(50, 500));
    }
    /* quantize duration to get a more rhythmic effect: */
    duration = ((duration + 10) / 20) * 20;
    cause(duration, melody, channel, pitch, center);
}


/* timbre -- periodically request a timbre (program) change */
/**/
void timbre(channel)
  int channel;	/* the channel to send changes to */
{
    my_program(channel, random(1, nprograms));
    cause(random(100, 1000), timbre, channel);
}


void keydown(int channel, int key, int velocity)
{
}


void keyup(int channel, int key)
{
}

void asciievent(char c)
{
    if (c == 'q') quit();
    else {
	gprintf(TRANS, "Type q to quit\n");
    }
}
