/* smqdemo.c -- demonstrate input processing for "Still More Questions" */

/************************************************************************/
/*
This program was derived from Still More Questions, which was written by
Roger Dannenberg in 1985 and first performed that year at UCSD.  My goal
in this composition was to have a computer listen to a performance, do
some analysis of various high-level qualities, and use the resulting
parameters to perform timbre selection.  In this example, I have extracted
code that performs the analysis.  I have explicitly eliminated the MIDI
output code to discourage performing the piece without the proper
synthesizer, patches, and effects processor.  Please contact the composer
if you wish to make performance arrangements.
The analysis produces several outputs:

    Attack Density (att_density) -- how many notes were initiated in
	the last 1-second interval?
    Average Density (avg_density) -- an indication of recent history
	of Attack Density.
    Tone Density (ton_density) -- during what percent of the last 
	1-second interval was there a note sounding?
    Pitch Center (pitch_center) -- what is the short term pitch
	center?
    Last Duration (last_duration) -- what was the duration of the 
	last note?
    Time of Last Long Duration (lng_dur_time) -- when was the last
	long duration played?
    High Density Count (dens_count) -- for how many seconds
	has the attack density been high?

Exercises:

    The best way to understand what this program is about is to run
it through its paces.  The following exercises will show what the program
can do, teach how to control the program, and illustrate what it is like
to perform Still More Questions:

1) Objective: control the Attack Density.  Play a note.  Notice the Attack
Density becomes 1 momentarily.  Play about one note per second.  Try to
make the Attack Density stay around 1.  Play rapidly.  Notice the Attack
Density jumps to a higher value.  Experiment with different rates.

2) Objective: control the Average Density.  Make the Attack Density
higher than 4.  Notice that the Average Density increases.  Rest until
the Average Density decreases.  Play a high Attack Density until the 
Average Density reaches at least 5.  Now play a low Attack Density by 
averaging 1 or 2 notes per second.  Notice that the Average Density
stays constant.  Stop playing until the Average Density falls to zero.
The Rules: High Attack Densities increase the Average Density, low 
Attack Densities hold the Average constant, and silence lowers the
Average Density.  (So "Average" is really a misnomer.)  Practice making
the Average Density go to and remain at a chosen value.

3) Objective: control the Tone Density.  Play a melody with long and
sustained tones.  Notice the Tone Density becomes 100.  Play staccato. 
Notice the Tone Density becomes small.  

4) Objective: control Average Density and Tone Density together.  Make the
Average Density and Tone Density low by playing a staccato note every second.
Make only the Tone Density increase by playing a sustained note every second.
Make only the Average Density increase by playing several staccato notes
every second.  (Hint: sometimes grace notes help increase the Attack Density
while minimizing the Tone Density.)  Make both values high by playing a
rapid legato passage.

5) Objective: control the Pitch Center.  Play high.  Play low.  Notice how
the Pitch Center changes.  Every second, the Pitch Center changes in the
direction of the current pitch.

6) Objective: control the Last Duration.  Play a short note.  Play a long
note.  Notice the duration (in hundredths of a second) are measured and
shown as the Last Duration.

7) Objective: control the Time of Last Long Duration.  Play a long note
(more than 3 seconds).  The time of the note (in seconds) is displayed.
Wait 10 seconds and play another long note.  Notice the number changes.

8) Objective: control High Density Count.  Make the Average Density go
to 9.  Hold it there.  Notice that playing a high Attack Density
causes the High Density Count to increase.

9) Objective: making music.  Repeat steps 1 through 8, this time 
concentrating on making musical phrases.  Do you find yourself in
new improvisational territory?  Are you stretching your stylistic
limits to accomplish these tasks?  Really hearing yourself while 
playing is difficult.  Use a tape recorder.  

10) Objective: performance.  If you would like to perform this
piece, contact the composer.

********************************************************************/

#include "stdio.h"
#include "musiprog.h"

char *app_syntax = "";

int att_density = 0;	/* att_density = number of notes per second input */
int ton_density = 0;	/* the time a note was on during last second */
int sil_min = 90;	/* silence required to decrease avg_density */
int pitch_center = 60;	/* the pitch center: slew rate limited follower */
delay_type last_duration = 0;	/* dur of last note */
int sus_min = 300;	/* definition of a long note */
long lng_dur_time = 0;	/* time of last long duration */
int dens_count = 0;	/* high (output) density count */
int avg_density = 0;	/* the output density */

int avg_att_density = 4;/* definition of high attack density */
long last_on_note = 0;	/* time of last note on (used to schedule note_off) */
int sil_total = 0;	/* amount of accumulated silence */
long ton_base = 0;	/* time to integrate from when note off occurs */
int cur_pitch = 0;	/* the current pitch (0 = no pitch) */
int cur_chan = 1;	/* the channel for cur_pitch */
int long_flag = false;

int thru_flag = false;	/* determines whether input is sent to output */

void write_headers();
void input_poll();

void mainscore()
{
    write_headers();
    input_poll();
#ifdef MIDI_THRU /* if defined, turn it off */
    midi_thru(false);
#endif
}


/* this routine is called every second to perform analysis functions */
/**/
void input_poll()
{
    /* Update Attack Density, Average Density, and High Density Count */
    if (att_density > avg_att_density)
	if (avg_density < 9) avg_density++;
    else dens_count++;

    /* Update Pitch Center, Tone Density if note is sounding */
    if (cur_pitch != 0) {
	if (cur_pitch > pitch_center) pitch_center++;
	else if (cur_pitch < pitch_center) pitch_center--;
	ton_density += virttime - ton_base;
    }
    ton_base = virttime;

    /* If silence, update Average Density and High Density Count */
    if (ton_density == 0) {
	sil_total += 100;
	if (sil_total > sil_min && avg_density > 0) {
	    sil_total = 0;
	    dens_count = 0;
	    avg_density--;
	}
    } else sil_total = 0;


    /* If we heard a long note, record time */
    if (long_flag) lng_dur_time = virttime;
    long_flag = false;

    gprintf(TRANS, "%4d %7d %7d %6d %8ld %10ld %9d\r", 
    att_density, avg_density, ton_density, pitch_center, last_duration,
    lng_dur_time, dens_count);

    att_density = 0;
    ton_density = 0;

    /* poll again in 1 sec */
    cause(100, input_poll);
}


/* keydown -- this gets called whenever a MIDI note on is received */
/**/
void keydown(int c, int k, int v)
{
    /* ton_base is used to compute ton_density (see keyup & input_poll) */
    ton_base = virttime;
    /* last_on_note is used to compute last_duration (see keyup) */
    last_on_note = virttime;
    att_density++;
    if (cur_pitch != 0) { /* a note is already in progress, so this must
			     be a polyphonic input.  Call keyup to shut
			     up previous sounding note. */
	keyup(c, cur_pitch);
    }
    cur_chan = c;
    cur_pitch = k;
}


/* keyup -- this gets called whenever a MIDI note off is received */
/**/
void keyup(int chan, int k)
{
    /* this is a test to allow for polyphonic input.  Poly input is
       converted to mono input */
    if (cur_pitch != k) {	/* keyup is not for current note */
	return;		/* so ignore it */
    }
    cur_pitch = 0;
    ton_density += virttime - ton_base;
    last_duration = virttime - last_on_note;
    if (last_duration > sus_min) long_flag = true;
}


void asciievent(char c)
{
    switch (c) {
      case 'q': quit(); break;
      case 'm':
	if (cur_pitch) midi_note(cur_chan, cur_pitch, 0);
	thru_flag = false;
	break;
      case 'M': thru_flag = true; break;
      case 'H':
      case 'h':
      case '?':
	gprintf(TRANS, "\n\nq: to quit\n");
	gprintf(TRANS, "m: enable MIDI thru\n");
	gprintf(TRANS, "M: disable MIDI thru\n");
	gprintf(TRANS, "H,h,?: this help\n\n");
	write_headers();
	break;
    }
}


void write_headers()
{
    gprintf(TRANS, "%s\n%s\n",
	"ATTACK  AVERAGE  TONE   PITCH    LAST    TIME OF  HIGH DENSITY",
	"DENSITY DENSITY DENSITY CENTER DURATION LAST LONG     COUNT");
}
