/* c200key -- midi event handlers for c200 */

/*
    Copyright pending, Intelligistics, Inc.

    Please see the notice in the main module associated with 
    this submodule: c200.c

*/
/*
    Revisions:

    200 -- gwl I*I 24-Nov-89 -- created from c103asc.c
		   15-Jun-90 -- add trigger scan
		   19-Jun-90 -- mode supervisor
		   14-Oct-91 -- parameterized controls
		   08-Jul-93 -- control for "z"
*/

/* definitions */

#include	"cmtprog.h"
#include	"linterp.h"	/* macros for interpolation */
#include	"cscdef.h"	/* definitions and externals */
#include	"keydef.h"
#include	"stdio.h"
#include	"ascdef.h"	/* definitions for keys in */
#include	"ctype.h"	/* for text routines */
#include	"string.h"


/* entry points used below */

void		m000init();
void		m000reset();
void		m000down();
void		m000up();
void		m001init();
void		m001reset();
void		m001down();
void		m001up();
void		m002init();
void		m002reset();
void		m002down();
void		m002up();


/* externals */

struct instdata inst[NINSTS];		/* input instrument data */

struct modedata mode[NMODES];		/* output mode data */

struct modealgs algo[NALGS]={		/* processing entry points */
    m000init, m000reset, m000down, m000up, 
    m001init, m001reset, m001down, m001up, 
    m002init, m002reset, m002down, m002up}; 


/* extrema */

double		contmin=0;		/* continuous controller */
double		contmax=127;
double		bendmin=0;		/* pitch bender */
double		bendnorm=8192;
double		bendmax=16382;

double		sepmin=0;		/* separation */
double		sepmax=1000;
double		nevnmin=1;		/* #events */
double		nevnmax=100;
double		nnotmin=1;		/* #notes */
double		nnotmax=200;

double		sepfromcont;
double		nevnfromcont;
double		nnotfromcont;
int		bendch1=7192;		/* bender < this, ch 1 only */
int		bendch2=9192;		/* bender > this, ch 2 only */

/* controls */

static	int	chanctrl=0;		/* control channel, 0=none */
static	int	chanprgm=0;		/* program change channel, 0=none */
static	int	errorctrl=0;		/* 1=show control errors */
static	int	errorprgm=0;		/* 1=show prgm chnge errors */
static	int	enablebend=1;		/* 1=bender enabled for chan/reset */
static	int	enablepedal=1;		/* 1=sustain pedal 64 down for "n" */
static	int	ctrlstop=65;		/* portmto sw for "y" = stop notes */
static	int	ctrlstart=66;		/* sostenuto ped = "z" = start casc */
static	int	ctrlnext=67;		/* soft pedal for "n" = next parmset */
static	int	ctrlnevents=1;		/* mod whl = # events per cascade */
static	int	ctrlnnotes=4;		/* cont ped = # notes per event */
static	int	ctrlsepabs=14;		/* slider #1 = abs sep csecs */
static	int	ctrlsepran=15;		/* slider #2 = ran sep csecs */
static	int	prgmstop=3;		/* prgm change "y" = stop notes */
static	int	prgmstart=4;		/* prgm change "z" = start casc */
static	int	prgmnext=5;		/* prgm change "n" = next parmset */

/* parameter file */

static	FILE	*prm;			/* points to parameter file */
static	char	*prmname		/* name of parameter file */
		    ="c200prm.dat";


void keyreset();


/******************************************************************************
     bendchange 
******************************************************************************/

void bendchange(ch,val)
int	ch;	/* channel */
int	val;	/* new value */
{

    if (!enablebend) return;		/* only if enabled */
    if (ch!=chanctrl) return;		/* and control channel */

    if (val<bendnorm)			/* down is ch 1 */
    {
	curcp->chanbeg=curcp->chanend=1;
	keyreset();
    }
    else if (val>bendnorm)		/* up is ch 2 */
    {
	curcp->chanbeg=curcp->chanend=2;
	keyreset();
    }
    else				/* norm is both */
    {
	curcp->chanbeg=1;
	curcp->chanend=2;
    }
}


/******************************************************************************
    ctrlchange -- other controllers -- default values

	( 1)	mod wheel	# events
	( 4)	continuous foot	# notes
	(14)	slider 1	absolute separation
	(15)	slider 2	random searation
	(65)	portamento 	'y' stop command
	(66)	sostenuto 	'z' start command
	(67)	soft		'n' next command

******************************************************************************/

void ctrlchange(ch,c,val)
int	ch;	/* channel */
int	c;	/* controller number */
int	val;	/* new value */
{

    if (ch!=chanctrl) 			/* only if control channel */
    {
	if (errorctrl>0)		/* if showing ctrl chan errors */
	{
	    gputchar(BEL);
	    gprintf(TRANS, "ctrlchange: ch %d, cont# %d, val %d\n",ch,c,val);
	}
	return;
    }

    /* portamento pedal does command "y" */
    
    if (c==ctrlstop)
    {
	if (val==127)			/* when on */
	{
	    asciievent('y');
	}
	return;
    }

    /* sostenuto pedal for command "z" */
    
    if (c==ctrlstart)
    {
	if (val==127)			/* when on */
	{
	    asciievent('z');
	}
	return;
    }

    /* soft pedal for command "n" */
    
    if (c==ctrlnext)
    {
	if (val==127)			/* when on */
	{
	    asciievent('n');
	}
	return;
    }

    /* other values change current cascade parameters */

    if (c==ctrlnevents)			/* mod wheel is #events */
    {
	nevnfromcont=finterp(contmin,contmax,nevnmin,nevnmax);
	curcp->nevents
	    =blinterp(val,contmin,contmax,nevnmin,nevnmax,nevnfromcont);
    }
    else if (c==ctrlnnotes)		/* cont pedal is #notes */
    {
	nnotfromcont=finterp(contmin,contmax,nnotmin,nnotmax);
	curcp->nnotes
	    =blinterp(val,contmin,contmax,nnotmin,nnotmax,nnotfromcont);
    }
    else if (c==ctrlsepabs)		/* slider 1 is sepabs */
    {
	sepfromcont=finterp(contmin,contmax,sepmin,sepmax);
	curcp->sepabs
	    =blinterp(val,contmin,contmax,sepmin,sepmax,sepfromcont);
    }
    else if (c==ctrlsepran)		/* slider 2 is sepran */
    {
	sepfromcont=finterp(contmin,contmax,sepmin,sepmax);
	curcp->sepran
	    =blinterp(val,contmin,contmax,sepmin,sepmax,sepfromcont);
    }
    else if (errorctrl)			/* if showing errors */
    {
	gputchar(BEL);
	gprintf(TRANS, "ctrlchange: ch %d, cont# %d, val %d\n",ch,c,val);
    }
}


/******************************************************************************
    infromch -- convert input channel to instrument index
******************************************************************************/

infromch(ch)
int	ch;	/* input channel number (1-16) */
{
    int	j;

    for (j=0; j<NINSTS; j++)		/* locate instrument */
    {
	if(inst[j].channel==ch)
	{
	    return (j);
	}
    }
/*
    gprintf(TRANS, "infromch ch %d\n",ch);
*/
     return(-1); 			/* something wrong */
}


/******************************************************************************
    keydown -- save keys pressed and when
******************************************************************************/

void keydown(ch,p,v)
int	ch;	/* channel */
int	p;	/* pitch */
int	v;	/* velocity */
{
    int				i,j,m,a;
    struct m000data	*mp;
/*
gprintf(TRANS, "keydown ch %d p %d v %d\n",ch,p,v);
*/
    if ((j=infromch(ch))<0)		/* locate instrument */
    {					/* something wrong */	
	gprintf(TRANS, "keydown ch %d p %d v %d\n",ch,p,v);
	return;
    }
/*
gprintf(TRANS, "keydown j %d\n",j);
*/
    for (i=0; i<inst[j].nomodes; i++)
    {
	m=inst[j].omode[i];		/* output mode */
	a=mode[m].algnum;		/* algorithm */
	mp=mode[m].mp.m000p;		/* data pointer */
/*
gprintf(TRANS, "keydown mp %u m %d a %d\n",mp,m,a);
*/
	if (algo[a].keydown)		/* do it */
	    (*algo[a].keydown)(mp,ch,p,v);
    }

}


/******************************************************************************
    keyinit -- initialize modes, etc.
******************************************************************************/

void keyinit()
{
    int			m,a;
    struct m000data	*mp;
    
    /* read parameter values */

    if (prm=fopen(prmname,"ra"))
    {
	fscanf(prm,"%d",&chanctrl);
	fscanf(prm,"%d",&chanprgm);
	fscanf(prm,"%d",&errorctrl);
	fscanf(prm,"%d",&errorprgm);
	fscanf(prm,"%d",&enablebend);
	fscanf(prm,"%d",&enablepedal);
	fscanf(prm,"%d",&ctrlstop);
	fscanf(prm,"%d",&ctrlstart);
	fscanf(prm,"%d",&ctrlnext);
	fscanf(prm,"%d",&ctrlnevents);
	fscanf(prm,"%d",&ctrlnnotes);
	fscanf(prm,"%d",&ctrlsepabs);
	fscanf(prm,"%d",&ctrlsepran);
	fscanf(prm,"%d",&prgmstop);
	fscanf(prm,"%d",&prgmstart);
	fscanf(prm,"%d",&prgmnext);

	if (fclose(prm)<0)
	{
	    gprintf(TRANS, "keyinit: unable to close %s\n",prmname);
	}
    }
    else
    {
	gprintf(TRANS, "keyinit: unable to open %s\n",prmname);
    }
    gprintf(TRANS, "Ctrl change channel %d\n",chanctrl);
    gprintf(TRANS, "Prgm change channel %d\n",chanprgm);
    gprintf(TRANS, "Show ctrl errors?   %s\n",errorctrl?"yes":"no");
    gprintf(TRANS, "Show prgm errors?   %s\n",errorprgm?"yes":"no");
    gprintf(TRANS, "Bender for channel? %s\n",enablebend?"yes":"no");
    gprintf(TRANS, "Sustain for \"n\"?    %s\n",enablepedal?"yes":"no");
    gprintf(TRANS, "Ctrl# for \"y\"       %d\n",ctrlstop);
    gprintf(TRANS, "Ctrl# for \"z\"       %d\n",ctrlstart);
    gprintf(TRANS, "Ctrl# for \"n\"       %d\n",ctrlnext);
    gprintf(TRANS, "Ctrl# for #events   %d\n",ctrlnevents);
    gprintf(TRANS, "Ctrl# for #notes    %d\n",ctrlnnotes);
    gprintf(TRANS, "Ctrl# for sepabs    %d\n",ctrlsepabs);
    gprintf(TRANS, "Ctrl# for sepran    %d\n",ctrlsepran);
    gprintf(TRANS, "Prgm# for \"y\"       %d\n",prgmstop);
    gprintf(TRANS, "Prgm# for \"z\"       %d\n",prgmstart);
    gprintf(TRANS, "Prgm# for \"n\"       %d\n",prgmnext);

    /* initialize modes */

    for (m=0; m<NMODES; m++)		/* scan the mode list */
    {
	if (mp=mode[m].mp.m000p)	/* if there's a data pointer */
	{
	    a=mode[m].algnum;		/* and an algorithm */
	    if (algo[a].init)		/* do it */
		(*algo[a].init)(mp);
	}
    }
}


/******************************************************************************
    keyreset -- initialize/reset key inputs
******************************************************************************/

void keyreset()
{
    int			m,a;
    struct m000data	*mp;
    
    for (m=0; m<NMODES; m++)		/* scan the mode list */
    {
	if (mp=mode[m].mp.m000p)	/* if there's a data pointer */
	{
	    a=mode[m].algnum;		/* and an algorithm */
	    if (algo[a].reset)		/* do it */
		(*algo[a].reset)(mp);
	}
    }
}


/******************************************************************************
    keyup -- save when keys released and check for enough
******************************************************************************/

void keyup(ch,p)
int	ch;	/* channel */
int	p;	/* pitch */
{
    int			i,j,m,a;
    struct m000data	*mp;

    if ((j=infromch(ch))<0) 		/* locate instrument */
    {					/* something wrong */	
	gprintf(TRANS, "keyup   ch %d p %d\n",ch,p);
	return;
    }

    for (i=0; i<inst[j].nomodes; i++)
    {
	m=inst[j].omode[i];		/* output mode */
	a=mode[m].algnum;		/* algorithm */
	mp=mode[m].mp.m000p;		/* data pointer */
	if (algo[a].keyup)		/* do it */
	    (*algo[a].keyup)(mp,ch,p);
    }
}


/******************************************************************************
    peddown -- go to next script item "n" with sustain pedal
******************************************************************************/

void peddown(ch)
int	ch;	/* channel */
{
    if (!enablepedal) return;		/* only if enabled */
    if (ch!=chanctrl) return;		/* and control channel */

    asciievent('n');
}


/******************************************************************************
     pedup -- pedal up -- no effect
******************************************************************************/

void pedup(ch)
int	ch;	/* channel */
{
}


/******************************************************************************
    prgmchange -- program changes
******************************************************************************/

void prgmchange(ch,val)
int	ch;	/* channel (1-16)*/
int	val;	/* value (1-128) */
{

    if (ch!=chanprgm) 			/* only if program change channel */
    {
	if (errorprgm>0)		/* showing prgm chan errors */
	{
	    gputchar(BEL);
	    gprintf(TRANS, "pgmchng ch %d, val %d\n",ch,val);
	}
	return;
    }

    /* interpret */

    if (val==prgmstop)		/* stop all notes */
    {
	asciievent('y');
    }
    else if (val==prgmstart)		/* start a cascade */
    {
	asciievent('z');
    }
    else if (val==prgmnext)		/* next parm set */
    {
	asciievent('n');
    }
    else if (errorprgm)			/* showing errors */
    {
	gputchar(BEL);
	gprintf(TRANS, "pgmchng ch %d, val %d\n",ch,--val);
    }
}


/******************************************************************************
    touchchange -- aftertouch -- no effect
******************************************************************************/

void touchchange(ch,val)
int	ch;	/* channel */
int	val;	/* new value */
{
}

/******************************************************************************
    midievent -- any midi message -- no effect, never enabled anyway
******************************************************************************/
void midievent(data)
byte data[4];
{
}


void sysex(void)
{
}

#ifdef AMIGA
void buttonchange(int number, int value)
{
}

void propchange(int number, int value)
{
	/* insert propchange actions here */
}
#endif
