/****************************************************************************
*
*    score.c -- A performance score abstract data type.
*
*    A performance score consists of an array of Evts and an associated
*    array of compound evts (Cevts). A score also knows its total
*    duration (in centiseconds) and the duration of its longest single
*    evt, which is needed by iter.c. It is possible to read a score
*    from a file or to create one through a Midi recording process
*    (see recorder.c) but there is no program interface for building a
*    performance score note by note. Similarly, there is currently no
*    score_Write operation, although it would be easy to add such an
*    operation which printed a score to an output stream in the human-
*    readable Adagio language (the form expected by score_Read).
*
*    Details: For now, a compound evt is defined to be any set of
*    evts with the same starting time (i.e. a chord). Someday,
*    this may be extended to include glissandos, trills, other ornaments,
*    fast arpeggios, and goodness knows what else. However, using the
*    simple definition, it is possible to generate the Cevt array from
*    the Evt array. The routine score_make_cevts does just that, and is
*    used on Evt arrays derived either from parsing Adagio text or from
*    recording a Midi performance.
*
*    Copyright (C) 1987, John H. Maloney
*    All rights reserved
*
*    Change Log
*    Date   | Change
*-----------+-----------------------------------------------------------------
* 11-Sep-87 | Created change log
****************************************************************************/

#include "stdio.h"
#include "cext.h"
#include "mem.h"
#include "userio.h"
#include "midifns.h" /* to get time_type */
#include "timebase.h"
#include "seq.h"
#include "evt.h"
#include "cevt.h"
#include "score.h"


public void score_destroy(self)
  score_type self;
  /*
   *    Frees a score.
   */
{
    if (self != NULL) {
	free((char *) self->cevt_base);
	free((char *) self->evt_base);
	memfree((char *) self, sizeof(score_node));
    }
}


/* score_convert -- create a new score from a sequence */
/**/
public score_type score_convert(seq, ch)
  seq_type seq; /* the seq to convert */
  int ch;       /* the channel that carries the solo */
{
    int nCevts;        /* number of cevts */
    score_type newScore;

    newScore = (score_type) memget(sizeof(score_node));
    if (newScore == NULL) return NULL;
    newScore->cevt_base = NULL;
    newScore->length = 0L;    /* uninitialized for now! */

    newScore->evt_base = evt_make_evts(seq, ch, &nCevts);
    newScore->cevt_base = cevt_make_cevts(nCevts, newScore->evt_base,
					&(newScore->length));
    return newScore;
}


void score_show_cevt_array(score)
  score_type score;
  /*
   *    Used for debugging. Prints the Cevt array in human-readable
   *    form to the transcript.
   */
{
    cevt_type cevts = score->cevt_base;

    if (cevts == NULL) {
	gprintf(TRANS, "NULL cevt array pointer.\n");
	return;
    }
    while (!evt_is_last(*cevts)) {
	evt_show(*cevts, cevts[1]);
	cevts++;
    }
    evt_show(*cevts, NULL);    /* show last event */
}


public void score_show_evt_array(score)
  score_type score;
  /*
   *    Used for debugging. Prints the Evt array in human-readable
   *    form to the transcript.
   */
{
    evt_type evts = score_first_evt(score);

    if (evts == NULL) {
	gprintf(TRANS, "NULL evts array pointer.\n");
	return;
    }
    while (!evt_is_last(evts)) {
	evt_show(evts, NULL);
	evts++;
    }
    evt_show(evts, NULL);    /* show last evt */
}
