/* sched2 -- do tempo tracking with following rules:
    1 use a buffer of sched_buffer_len notes
    2 set tempo when buffer is full or spans sched_buffer_time milliseconds
    3 change tempo by at most sched_slew_rate
    4 move tempo back to nominal tempo at sched_creep_rate every second
    5 flush buffer when errors occur
    6 tiered catchup policy:
	behind by >ignore_interval: ignore
	behind by <race_interval: change tempo to rate_minimum to catch up
	behind by >race_interval: skip if possible, otherwise just catchup()
 */

#include "stdio.h"
#include "cext.h"
#include "mem.h"
#include "userio.h"
#include "midifns.h" /* to get time_type */
#include "timebase.h"
#include "seq.h"
#ifdef SMARTSEQ
#include "smartseq.h"
#endif /* SMARTSEQ */
#include "moxc.h"
#include "midicode.h"
#include "sched.h"
#ifdef SLAVE
#include "slave.h"
#endif

extern boolean verbose;

typedef struct {
    time_type rtime;
    time_type vtime;
} sched_buffer_node, *sched_buffer_type;

#define sched_buffer_max 4

typedef struct sched2_struct {
    sched_node sched;
    time_type high_water_mark;
    time_type patience;
    int buffer_tail;    /* points to place where next entry will go */
    int buffer_len;     /* number of valid entries in buffer */
    sched_buffer_node buffer[sched_buffer_max];
    time_type buffer_time;
    int slew_rate;      /* how fast rate is allowed to change */
    int creep_rate;     /* move to nominal rate (256) by creep_rate every sec */
    int rate_minimum;   /* fastest we are allowed to play */
    int plan_id;        /* tag for current tempo adjustment plan */
    time_type ignore_interval; /* ignore time errors of this magnitude */
    time_type race_interval;    /* pause or race if error is this magnitude */
    int last_good_rate; /* last computed rate */
} sched2_node, *sched2_type;

char sched_syntax[] = "pat<o>Patience: how long to wait before stopping;\
			acc<o>Acceleration: maximum tempo change per match;\
			pull<o>How much to pull tempo to that of the score;\
			fast<o>Fastest score can go (default 128);\
			ignore<o>Take action if time error is greater;\
			race<o>Race to catch up if time error is less;";

#define sched2_alloc() (sched2_type) memget(sizeof(sched2_node))

#define sched2_free(s) memfree((char *) (s), sizeof(sched2_node))

sched_type sched_create(seq_type seq, time_type patience)
{
    sched2_type sched = sched2_alloc();
    if (!sched) return NULL;
    sched->sched.seq = seq;
    sched->patience = patience;
    sched->buffer_time = 1000;
    sched->slew_rate = cl_int_option("acc", 10L);
    sched->creep_rate = cl_int_option("pull", 1L);
    sched->rate_minimum = cl_int_option("fast", 128L);
    sched->ignore_interval = cl_int_option("ignore", 10L);
    sched->race_interval = cl_int_option("race", 1000L);
    sched_reset(&(sched->sched));
    return &(sched->sched);
}


void sched_destroy(sched_type sched)
{
    sched2_free((sched2_type) sched);
}


/* sched_flush -- flush the match time buffer */
/**/
void sched_flush(sched_type sched)
{
    sched2_type self = (sched2_type) sched;
    self->buffer_len = 0;
}


/* sched_reset -- get ready to run a new accompaniment */
/**/
void sched_reset(sched_type sched)
{
    sched2_type self = (sched2_type) sched;
    self->buffer_len = 0;
    self->buffer_tail = 0;
    self->plan_id = 0;
    self->last_good_rate = 256;
}


/* creeper -- adjust the tempo toward the norm */
/**/
void creeper(sched)
  sched2_type sched;
{
    time_type newrate = seq_rate(sched->sched.seq);
    if (sched->sched.seq->note_enable) {
	if (newrate > 256) {
	    newrate -= sched->creep_rate;
	} else newrate += sched->creep_rate;
	if (verbose) gprintf(TRANS, "creep to %ld\n", newrate);
	seq_set_rate(sched->sched.seq, newrate);
	/* note: seq_set_rate has a side effect of putting the current
	 * virtual time of the sequence into the timebase.  Peeking is
	 * probably not good software engineering, but do it anyway...
	 */
#ifdef SLAVE
	if (slave)
	    slave_update(sched->sched.seq->timebase->virt_base, newrate);
#endif
    }
    if (sched->sched.seq->runflag) cause(1000, creeper, sched);
    else if (verbose) gprintf(TRANS, "creeper finished\n");
}


/* caught_up -- return tempo to normal once the accompaniment has caught up */
/**/
void caught_up(self, plan_id, newrate)
  sched2_type self;
  int plan_id;
  time_type newrate;
{
    if (self->plan_id == plan_id) {
	if (verbose) gprintf(TRANS, "caught_up: plan %d newrate %ld\n", 
			     plan_id, newrate);
	seq_set_rate(self->sched.seq, newrate);
#ifdef SLAVE
	if (slave) slave_update(self->sched.seq->timebase->virt_base, newrate);
#endif
    }
}


/* pause the sequence until it moves beyond high_water_mark */
void checkpoint(sched)
  sched2_type sched;
{
    if (verbose) gprintf(TRANS, "checkpoint virttime %ld hwm %ld", 
				virttime, sched->high_water_mark);
    if (virttime >= sched->high_water_mark) {
	if (verbose) gprintf(TRANS, " (pausing)");
	seq_pause(sched->sched.seq, true);
    }
    if (verbose) gprintf(TRANS, "\n");
}


/* index to a particular starting time */
/**/
void sched_jump_to(sched_type sched, time_type dest)
{
    seq_start_time(sched->seq, dest);
#ifdef SLAVE
    if (slave) slave_start_time(dest);
#endif
}


/* handle a match */
/**/
void sched_match(
  sched_type sched,	/* scheduler object */
  int id,		/* identification number of matcher */
  time_type rtime,	/* real time of the event */
  time_type vtime,	/* virtual time of this match */
  boolean last_flag,	/* tells whether or not this is the end of score */
  time_type next_match)	/* if !last_flag, tells when next match should be */
{
    sched_buffer_type buff_ptr, buff_head;
    register sched2_type self = (sched2_type) sched;
    sgnd_time_type buff_dur, interval, abs_interval;
    long oldrate = self->last_good_rate;
    long newrate = oldrate;  /* ratio rtime/vtime << 8 */

    if (verbose) 
	gprintf(TRANS, "sched_match at rtime %ld vtime %ld pause %d\n", 
		rtime, vtime, self->sched.seq->paused);
    /* compute location of tail, advance tail pointer */
    buff_ptr = &(self->buffer[self->buffer_tail]);
    self->buffer_tail = (self->buffer_tail + 1) & (sched_buffer_max - 1);

    /* insert new data in buffer */
    buff_ptr->rtime = rtime;
    buff_ptr->vtime = vtime;
    if (self->buffer_len < sched_buffer_max) {
	self->buffer_len++;
    }

    /* compute location of head */
    buff_head = &(self->buffer[(self->buffer_tail - self->buffer_len) &
			       (sched_buffer_max - 1)]);
	
    /* how much elapsed time in buffer? */
    buff_dur = rtime - buff_head->rtime;
    /* are we ready to assert new tempo? */
    if ((vtime > (buff_head->vtime + 100 /* ms */)) /* avoid div by 0 */ &&  
	(((self->buffer_len >= sched_buffer_max) && 
	  (buff_dur > 100 /* ms */)) ||
	 buff_dur > self->buffer_time)) {
	newrate = ((rtime - buff_head->rtime) << 8) /
		   (vtime - buff_head->vtime);
	if (newrate > oldrate + self->slew_rate) {
	    newrate = oldrate + self->slew_rate;
	    if (verbose) gprintf(TRANS, "high limit, ");
	} else if (newrate < oldrate - self->slew_rate) {
	    if (verbose) gprintf(TRANS, "low limit, ");
	    newrate = oldrate - self->slew_rate;
	}
	self->last_good_rate = newrate;
	if (verbose) gprintf(TRANS, "new rate: %ld\n", newrate);
    }
    /* catch up to current time */
    timebase_use(seq_timebase(sched->seq));
    abs_interval = interval = vtime - virttime;  /* we are behind by interval */
    if (verbose) gprintf(TRANS, "behind by %ld: ", interval);

    /* abs_interval is |interval| */
    if (interval < 0) abs_interval = -interval;

    if (abs_interval < self->ignore_interval) { /* fix up tempo */
	if (verbose) gprintf(TRANS, "fix up tempo, ignore vtime error\n");
	if (newrate != seq_rate(sched->seq)) {
	    seq_set_rate(sched->seq, newrate);
#ifdef SLAVE
	    if (slave) slave_update(virttime, newrate);
#endif
	}
    } else if (interval > 0 && interval < self->race_interval) {
	time_type delay;
	if (verbose) gprintf(TRANS, "catching up\n");
	seq_set_rate(sched->seq, (time_type) self->rate_minimum);
#ifdef SLAVE
	if (slave) slave_update(virttime, self->rate_minimum);
#endif
	/* compute time at which we will be back on track:
	    estimated score time (EST) =
		vtime + (rtime - eventtime) / newrate
	    accompaniment score time (AST) = 
		virttime + (rtime - eventtime) / rate_minimum
	    when will EST = AST?
		let delta = rtime-eventtime:
		vtime+(delta/newrate) = virttime+(delta/rate_minimum)
		vtime-virttime = (delta/rate_minimum)-(delta/newrate)
		rate_minimum*newrate*(vtime-virttime) = 
			delta*(newrate-rate_minimum)
		delta = rate_minimum*newrate*(vtime-virttime)/
				(newrate-rate_minimum)
		SO AST = virttime +  (rate_minimum*newrate*(vtime-virttime)/
				(newrate-rate_minimum)) / rate_minimum
		  = virttime + newrate * (vtime - virttime) / 
					 (newrate - rate_minimum)
	    watch out for case where pace is actually faster than the
	    rate_minimum, which might cause divide by zero, or negative
	    delay!  In these cases, do not cause caught_up() and rely on
	    future calls to this routine to get things straightened out,
	    e.g. if the pace continues, we'll eventually skip to catch up.
	*/
	if (newrate > self->rate_minimum) {
	    delay = (newrate * (vtime - virttime)) / 
		    (newrate - self->rate_minimum);
	    if (verbose) {
		gprintf(TRANS,
		    "vtime %ld virttime %ld newrate %ld rate_minimum %d\n",
		    vtime, virttime, newrate, self->rate_minimum);
		gprintf(TRANS, "catch up after %ld, plan %d\n", 
		    delay, self->plan_id+1);
	    }
	    cause(delay, caught_up, self, ++(self->plan_id), newrate);
	}
    } else if (interval < 0 && abs_interval < self->race_interval) {
	/* implement pause by jumping back in virtual time: */
	if (verbose) gprintf(TRANS, "pause by jumping back\n");
	set_virttime(sched->seq->timebase, vtime);
	if (newrate != seq_rate(sched->seq)) {
	    seq_set_rate(sched->seq, newrate);
	}
#ifdef SLAVE
	if (slave) slave_update(vtime, newrate);
#endif
    } else if (interval < 0) { /* interval is too large, time to jump */
	if (verbose) gprintf(TRANS, "jump back to location (actually just pause)\n");
	/* implement pause by jumping back in virtual time
	   (later this will actually jump):
	 */
	set_virttime(sched->seq->timebase, vtime);
	if (newrate != seq_rate(sched->seq)) {
	    seq_set_rate(sched->seq, newrate);
	}
#ifdef SLAVE
	if (slave) slave_update(vtime, newrate);
#endif
    } else { /* jump forward */
	if (verbose) gprintf(TRANS, "jump forward\n");
	set_virttime(sched->seq->timebase, vtime);
	sched->seq->note_enable = false;  /* suppress note output */
	catchup();
	sched->seq->note_enable = true;  /* reenable note output */
	if (newrate != seq_rate(sched->seq))
	    seq_set_rate(sched->seq, newrate);
#ifdef SLAVE
	if (slave) slave_jump(vtime, newrate);
#endif
    }
    /* be sure we are set to go */
    if (sched->seq->timebase->rate == STOPRATE) {
	if (verbose) gprintf(TRANS, "unpausing\n");
	seq_pause(sched->seq, false);
#ifdef SLAVE
	if (slave) slave_pause();
#endif
    }

    if (!last_flag) {
	/* now plan to stop at the next_match time */
	self->high_water_mark = next_match + self->patience;
	/* use priority 5 (between note offs and note ons) */
	causepri(next_match + self->patience - virttime, 5, checkpoint, self);
    } else self->high_water_mark += 1;  /* disable all future checkpoints */
}


void sched_nomatch(sched, id, pitch)
  sched_type sched;
  int id;
  int pitch;
{
    sched2_type self = (sched2_type) sched;
    self->buffer_len = 0;
}


#ifdef SLAVE
char slave_message[] = 
    { MIDI_SYSEX, 0x7d, 0,   0, 0, 0, 0,   0, 0, 0, 0,   MIDI_EOX };
#endif


#ifdef SLAVE
/* slave_jump -- send message to slave sequencer */
/**/
void slave_jump(vtime, newrate)
  time_type vtime;
  long newrate;
{
    slave_send_msg_2(slave_jump_cmd, vtime, newrate);    
    gprintf(TRANS, "jump %ld %ld\n", vtime, newrate);
}

/* slave_pause -- send message to slave sequencer */
/**/
void slave_pause()
{
    slave_message[2] = slave_pause_cmd;
    midi_exclusive(slave_message);
    gprintf(TRANS, "pause\n");
}


/* slave_send_msg_1 -- send a one parameter sysex message */
/**/
void slave_send_msg_1(cmd, data)
  char cmd;
  long data;
{
    slave_message[2] = cmd;

    slave_message[3] = data;
    slave_message[4] = (data >> 8);
    slave_message[5] = (data >> 16);
    slave_message[6] = (data >> 24);

    slave_message[7] = MIDI_EOX;

    midi_exclusive(slave_message);
}


/* slave_send_msg_2 -- send a two parameter sysex message */
/**/
void slave_send_msg_2(cmd, data1, data2)
  char cmd;
  long data1, data2;
{
    slave_message[2] = cmd;

    slave_message[3] = data1;
    slave_message[4] = (data1 >> 8);
    slave_message[5] = (data1 >> 16);
    slave_message[6] = (data1 >> 24);

    slave_message[7] = data2;
    slave_message[8] = (data2 >> 8);
    slave_message[9] = (data2 >> 16);
    slave_message[10] = (data2 >> 24);

    midi_exclusive(slave_message);
}


/* slave_start_time -- send message to slave sequencer */
/**/
void slave_start_time(dest)
  time_type dest;
{
    slave_send_msg_1(slave_start_time_cmd, dest);
    gprintf(TRANS, "start_time %ld\n", dest);
}


/* slave_update -- tell slave to reset its timebase */
/**/
void slave_update(vtime, rate)
  time_type vtime;
  long rate;
{
    /* send virtual time and rate */
    slave_send_msg_2(slave_update_cmd, vtime, rate);
    gprintf(TRANS, "update %ld %ld\n", vtime, rate);
}
#endif
