/****************************************************************************
*
*  iter.c -- A score iterator abstract data type.
*
*  A score iterator is a moveable cursor in a score. You can ask the
*  iterator for the Cevt and time at the current position and the
*  Cevt and time immediately preceding and following the current
*  position. You can move the interator forward or backward one
*  Cevt at a time or you can use iter_jump_to to seek to a random
*  point in the score. You can also clone the iterator if you want to
*  remember your current place in a score.
*
*  NOTE: Actually, one seldom needs to know the time of the preceding
*  Cevt, so iter_PrevTime is not implemented. It could easily be added.
*
*  Details: To save space, only time deltas are kept in the evt array
*  structure. This means that we have to sum these deltas from the
*  beginning of the score to compute the absolute time of any evt or
*  Cevt. As long as we move forward or backward incrementally, the time
*  at the current position can be computed incrementally. However, the
*  iter_jump_to operation is potentially quite expensive. The cost is
*  roughly proportional to how far in time the jump destination is from
*  the current position.
*
*  Copyright (C) 1987, John H. Maloney
*  All rights reserved
*
*       Change Log
*  Date | Change
*-----------+-----------------------------------------------------------------
* 11-Sep-87 | Created change log
****************************************************************************/

#include <stdio.h>

#include "cext.h"
#include "mem.h"
#include "stream.h"
#include "midifns.h"
#include "timebase.h"
#include "seq.h"
#include "evt.h"
#include "cevt.h"
#include "score.h"
#include "iter.h"

/****************************************************************************
*
*  Definitions
*
****************************************************************************/

/* Used in a special case of iter_jump_to.
 * Thirty seconds or before is "early" in a long-ish score.
 */
#define EARLY 3000L
#define TWICE_EARLY 6000L

/****************************************************************************
*
*  Private Procedures
*
****************************************************************************/

private ulong iter_cevt_dur();

private ulong iter_cevt_dur(cevt)
    cevt_type cevt;
    /*
     *  Sums the time deltas for all evts in a given Cevt and
     *  returns the total.
     *
     *  Details: Does NOT assume that all evts in a Cevt have
     *  the same starting time.
     */
{
    if (evt_is_last(*cevt)) {
	return 0L;
    } else {
	evt_type thisEvt;
	evt_type firstEvtOfnext_cevt = *(cevt + 1);
	ulong dur = 0L;

	for (thisEvt = *cevt; thisEvt < firstEvtOfnext_cevt; thisEvt++) {
	    dur += evt_dtime(thisEvt);
	}
	return evt_to_vtime(dur);
    }
}


void iter_next_cevt_and_time(self, next_cevt, next_time)
    iter_type self;
    cevt_type *next_cevt;
    ulong *next_time;
    /*
     *  *next_cevt is set to the next Cevt in the score.
     *  *next_time is set to the absolute time of the next Cevt.
     *  If the current Cevt is the last Cevt in the score, the current
     *  Cevt and time are returned.
     */
{
    if (evt_is_last(*(self->curr_cevt))) {
	*next_cevt = self->curr_cevt;
	*next_time = self->curr_time;
    } else {
	*next_cevt = self->curr_cevt + 1;
	*next_time = self->curr_time + iter_cevt_dur(self->curr_cevt);
    }
}

void iter_prev_cevt_and_time(self, prevCevt, prevTime)
    iter_type self;
    cevt_type *prevCevt;
    ulong *prevTime;
    /*
     *  *prevCevt is set to the previous Cevt in the score.
     *  *prevTime is set to the absolute time of the previous Cevt.
     *  If the current Cevt is the first Cevt in the score, the current
     *  Cevt and time are returned.
     */
{
    if (evt_is_first(*(self->curr_cevt))) {
	*prevCevt = self->curr_cevt;
	*prevTime = self->curr_time;
    } else {
	*prevCevt = self->curr_cevt - 1;
	*prevTime = self->curr_time - iter_cevt_dur(*prevCevt);
    }
}

/****************************************************************************
*
*  Exported Operations
*
****************************************************************************/


public iter_type iter_clone(self)
    iter_type self;
    /*
     *  Returns a new copy of iterator or NULL if iterator is NULL.
     */
{
    iter_type new_iter;

    if (self == NULL) {
	return NULL;
    } else {
	new_iter = (iter_type) memget(sizeof(iter_node));
	*new_iter = *self;
	return (iter_type) new_iter;
    }
}


/* iter_create -- return a new iterator for the score */
/**/
public iter_type iter_create(score)
  score_type score;
{
    iter_type new_iter;

    if (score == NULL) {
	return NULL;
    } else {
	new_iter = iter_alloc();
	new_iter->first_cevt = score_first_cevt(score);
	new_iter->curr_cevt = new_iter->first_cevt;
	new_iter->curr_time = 0L;

	return (iter_type) new_iter;
    }
}


/* iter_destroy -- destroy and iterator */
/**/
public void iter_destroy(iterator)
  iter_type iterator;
{
    if (iterator != NULL) {
	iter_free(iterator);
    }
}


/* iter_jump_to -- jumps to a given time in score */
/*
 * NOTE:  If scoreTime is greater than the total
 * duration of the score, the current position becomes the last
 * Cevt.
 *
 * Details: This operation is expensive for positions far from the
 * current position! (If we kept absolute start times in the evt
 * array, we could do a binary search, but we don't...) 
 */
public void iter_jump_to(self, scoreTime)
  iter_type self;
  ulong scoreTime;
{
    cevt_type this_cevt;
    ulong this_start_time, thisDur;

    this_cevt = self->curr_cevt;
    this_start_time = self->curr_time;

    if (scoreTime >= self->curr_time) { /* search forward */
	while (!evt_is_last(*this_cevt)) {
	    thisDur = iter_cevt_dur(this_cevt);
	    if ((this_start_time + thisDur) > scoreTime) {
		break;
	    } else {
		this_cevt++;
		this_start_time += thisDur;
	    }
	}
    } else {    /* search backward */
	while (!evt_is_first(*this_cevt)) {
	    this_cevt--;
	    this_start_time -= iter_cevt_dur(this_cevt);
	    if (this_start_time <= scoreTime) {
		break;
	    }
	}
    }
    self->curr_cevt = this_cevt;
    self->curr_time = this_start_time;
}


void iter_reset(self)
  iter_type self;
{
    self->curr_cevt = self->first_cevt;
    self->curr_time = 0L;
}


#ifdef NOTNEEDED
public ulong iter_next_time(iterator)
    iter_type iterator;
    /*
     *  Returns the time of the Cevt following the current position
     *  (or the current time if we are at the end of the score).
     */
{
    cevt_type next_cevt;
    ulong next_time;

    iter_next_cevt_and_time((iter_type) iterator, &next_cevt, &next_time);
    return next_time;
}


public void iter_Sounding(iterator, scoreTime, pMap)
    iter_type iterator;
    ulong scoreTime;
    PitchMap *pMap;     /* *pMap is modified */
    /*
     *  *pMap is set to reflect the set of notes sounding at scoreTime.
     *
     *  Details: We use a clone of ourself to locate the last Cevt before
     *  scoreTime, then move forward by evts to find the absolutely
     *  last evt before scoreTime. From this evt, we move backwards
     *  looking for notes which are held over scoreTime. We are done
     *  when we get to a time max_dur before scoreTime, because no note
     *  before that could still be sounding at scoreTime.
     */
{
    iter_type temp;
    ulong max_dur = ((Selfiter_type) iterator)->max_dur;
    ulong earliest_start_time, this_start_time;
    evt_type thisEvt;

    /* Clear the pitch map: */
    pMap->mapWord[0] = 0L;
    pMap->mapWord[1] = 0L;
    pMap->mapWord[2] = 0L;
    pMap->mapWord[3] = 0L;

    earliest_start_time = (scoreTime > max_dur) ? (scoreTime - max_dur) : 0L;

    /* Find the last Cevt before scoreTime and its time: */
    temp = iter_Clone(iterator);
    iter_jump_to(temp, scoreTime);
    this_start_time = iter_curr_time(temp);
    thisEvt = *iter_curr_cevt(temp);
    iter_Destroy(temp);

    /* Find the last evt (not just Cevt) before scoreTime: */
    while ((!evt_is_last(thisEvt)) &&
	   ((this_start_time + evt_dtime(thisEvt)) < scoreTime)) {
		this_start_time += evt_dtime(thisEvt);
		thisEvt++;
    }

    /* Scan backwards from thisEvt looking for notes which are held
     * over scoreTime (and stop when we get to the earliest time such
     * a note could occur), setting the pitch bits for all notes held
     * over scoreTime in pMap:
     */
    while ((this_start_time >= earliest_start_time) &&
	   (!evt_is_first(thisEvt))) {
	if ((evt_isNote(thisEvt)) &&
	    ((this_start_time + thisEvt->u.note.ndur) > scoreTime)) {
		/* Set bit for pitch in pitch map: */
		ushort wordIndex = (thisEvt->u.note.npitch - 1) / 32;
		ushort bitIndex = (thisEvt->u.note.npitch - 1) % 32;

		pMap->mapWord[wordIndex] =
			pMap->mapWord[wordIndex] | (1L << bitIndex);
	}
	thisEvt--;
	this_start_time -= evt_dtime(thisEvt);
    }
}
#endif
