#include "quaternion.h"

Quaternion::Quaternion() {

	int i;
	m_c[0] = 1;
	for (i = 1; i < 4; i++)
		m_c[i]=0;
}

Quaternion::Quaternion(const Quaternion& q) {
	copy(q);
}

Quaternion& Quaternion::operator =(const Quaternion& q) {

	if (this != &q)
		copy(q);

	return *this;
}

void Quaternion::copy(const Quaternion& q) {
	m_c[0]=q[0];
	m_c[1]=q[1];
	m_c[2]=q[2];
	m_c[3]=q[3];
}

void Quaternion::rotation(op_float angle, const Vector& axis) {
	
	Vector rvec;

	if (axis[0]==0.0f && axis[1]==0.0f && axis[2]==0.0f) {
		m_c[0]=1.0f;
		m_c[1] = m_c[2] = m_c[3] = 0.0f;
		return;
	}	

	angle = DEGTORAD(angle);
	m_c[0] = cos(angle*0.5f);

	rvec = sin(angle*0.5f)*axis;
	m_c[1] = rvec[0];
	m_c[2] = rvec[1];
	m_c[3] = rvec[2];

	*this = normalize();
}

void Quaternion::getAxisAngle(op_float& angle, Vector& axis) {
	
	op_float temp_angle;
	op_float scale;

	temp_angle = acos(m_c[0]);

	scale = (op_float)sqrt(m_c[1]*m_c[1]+m_c[2]*m_c[2]+m_c[3]*m_c[3]);
//	scale = (float)sin(temp_angle);

	angle = (op_float)(temp_angle * 2.0f);

	axis[0] = op_float(m_c[1]/scale);
	axis[1] = op_float(m_c[2]/scale); 
	axis[2] = op_float(m_c[3]/scale);

	axis = axis.normalize();
}

Quaternion Quaternion::slerp(const Quaternion& inq1, const Quaternion& inq2, op_float t) const {

	Quaternion qInt,q1,q2;
	op_float result,scale0,scale1;
	op_float theta,sinTheta;

	if (inq1 == inq2)
		return inq1;

	q1 = inq1; q2 = inq2;
	result = q1.dot(q2);

	if (result < 0.0f) {
		q2 = q2.negate();
		result = -result;
	}

	scale0 = 1 - t;
	scale1 = t;

	if (1 - result > 0.1f) {

		theta = (op_float)acos(result);
		sinTheta = (op_float)sin(theta);

		scale0 = (op_float)sin((1-t)*theta)/sinTheta;
		scale1 = (op_float)sin((t*theta))/sinTheta;
	}	

	qInt[0] = (scale0 * q1[0]) + (scale1 * q2[0]);
	qInt[1] = (scale0 * q1[1]) + (scale1 * q2[1]);
	qInt[2] = (scale0 * q1[2]) + (scale1 * q2[2]);
	qInt[3] = (scale0 * q1[3]) + (scale1 * q2[3]);

	return qInt;
}

void Quaternion::toMatrix(Matrix& mat) {

/*		// First row
	pMatrix[ 0] = 1.0f - 2.0f * ( y * y + z * z );  
	pMatrix[ 1] = 2.0f * ( x * y - w * z );  
	pMatrix[ 2] = 2.0f * ( x * z + w * y );  
	pMatrix[ 3] = 0.0f;  

	// Second row
	pMatrix[ 4] = 2.0f * ( x * y + w * z );  
	pMatrix[ 5] = 1.0f - 2.0f * ( x * x + z * z );  
	pMatrix[ 6] = 2.0f * ( y * z - w * x );  
	pMatrix[ 7] = 0.0f;  

	// Third row
	pMatrix[ 8] = 2.0f * ( x * z - w * y );  
	pMatrix[ 9] = 2.0f * ( y * z + w * x );  
	pMatrix[10] = 1.0f - 2.0f * ( x * x + y * y );  
	pMatrix[11] = 0.0f;  

	// Fourth row
	pMatrix[12] = 0;  
	pMatrix[13] = 0;  
	pMatrix[14] = 0;  
	pMatrix[15] = 1.0f;*/

	mat[0][0]=1-2*(m_c[2]*m_c[2]-m_c[3]*m_c[3]); mat[0][1]=2*(m_c[1]*m_c[2]-m_c[0]*m_c[3]); mat[0][2]=2*(m_c[1]*m_c[3]+m_c[0]*m_c[2]); mat[0][3]=0;
	mat[1][0]=2*(m_c[1]*m_c[2]+m_c[0]*m_c[3]); mat[1][1]=1-2*(m_c[1]*m_c[1]-m_c[3]*m_c[3]); mat[1][2]=2*(m_c[2]*m_c[3]-m_c[0]*m_c[1]); mat[1][3]=0;
	mat[2][0]=2*(m_c[1]*m_c[3]-m_c[0]*m_c[2]); mat[2][1]=2*(m_c[2]*m_c[3]+m_c[0]*m_c[1]); mat[2][2]=1-2*(m_c[1]*m_c[1]-m_c[2]*m_c[2]); mat[2][3]=0;
	mat[3][0]=0; mat[3][1]=0; mat[3][2]=0; mat[3][3]=1;

}

op_float& Quaternion::operator[] (int index) {	
	return m_c[index];
}

op_float Quaternion::operator[] (int index) const {
	return m_c[index];
}

bool Quaternion::operator ==(const Quaternion& q) const {

	return (m_c[0]==q[0] && m_c[1]==q[1] && m_c[2]==q[2] && m_c[3]==q[3]);
}

Quaternion operator * (const Quaternion& q1, const Quaternion& q2) {

	Quaternion q;

	q[0] = q1[0]*q2[0] - q1[1]*q2[1] - q1[2]*q2[2] - q1[3]*q2[3];
	q[1] = q1[0]*q2[1] + q1[1]*q2[0] + q1[2]*q2[3] - q1[3]*q2[2];
	q[2] = q1[0]*q2[2] - q1[1]*q2[3] + q1[2]*q2[0] + q1[3]*q2[1];
	q[3] = q1[0]*q2[3] + q1[1]*q2[2] - q1[2]*q2[1] + q1[3]*q2[0];

	return q;
}

Quaternion operator + (const Quaternion& q1, const Quaternion& q2) {

	Quaternion q;

	q[0] = q1[0] + q2[0];
	q[1] = q1[1] + q2[1];
	q[2] = q1[2] + q2[2];
	q[3] = q1[3] + q2[3];

	return q;
}

op_float Quaternion::dot(const Quaternion& q) const {

	return (m_c[0]*q[0]+m_c[1]*q[1]+m_c[2]*q[2]+m_c[3]*q[3]);

}

Quaternion Quaternion::conjugate() const {

	Quaternion q;

	q[0] = m_c[0];
	q[1] = -m_c[1];
	q[2] = -m_c[2];
	q[3] = -m_c[3];

	return q;
}

op_float Quaternion::norm() const {

	return sqrt(m_c[0]*m_c[0]+m_c[1]*m_c[1]+m_c[2]*m_c[2]+m_c[3]*m_c[3]);

}

Quaternion Quaternion::normalize() const {

	Quaternion q;
	op_float n = norm();

	q[0] = m_c[0]/n;
	q[1] = m_c[1]/n;
	q[2] = m_c[2]/n;
	q[3] = m_c[3]/n;

	return q;
}

Quaternion Quaternion::negate() const {

	Quaternion q;

	q[0] = -m_c[0];
	q[1] = -m_c[1];
	q[2] = -m_c[2];
	q[3] = -m_c[3];

	return q;
}