#include "stdafx.h"
#include "glwindow.h"

vector<GLWindow*> GLWindow::m_windows;

GLWindow::GLWindow(HINSTANCE hInstance) : m_hInstance(hInstance), m_fullscreen(false) { 
	m_hDC=NULL;
	m_hWnd=NULL;
	m_hRC=NULL;

	m_frustum = new GLFrustum();
	m_viewer = new GLViewer();
}

void GLWindow::create(const char* caption, op_rect viewport, int depth, bool vfullscreen) {

	WNDCLASS wndclass;
	DWORD dwStyle;

	HICON hIcon;

	hIcon = (HICON) LoadImage(NULL,"eye.ico",IMAGE_ICON,0,0,LR_LOADFROMFILE);

	memset(&wndclass, 0, sizeof(WNDCLASS));			
	wndclass.style = CS_HREDRAW | CS_VREDRAW;		
	wndclass.lpfnWndProc = GLWndProc;						
	wndclass.hInstance = m_hInstance;						
	wndclass.hIcon = hIcon;
	wndclass.hCursor = LoadCursor(NULL, IDC_ARROW);	
	wndclass.hbrBackground = (HBRUSH) (COLOR_WINDOW+1);	
	wndclass.lpszClassName = "Optic GL Window";			

	if (0 == RegisterClass(&wndclass))
		throw WindowCreationException("Unable to register Window class");
	
	m_viewport = viewport;

	if (vfullscreen) {													
		dwStyle = WS_POPUP | WS_CLIPSIBLINGS | WS_CLIPCHILDREN;
		fullscreen(true);						
		ShowCursor(FALSE);							
	}
	else									
		dwStyle = WS_OVERLAPPEDWINDOW | WS_CLIPSIBLINGS | WS_CLIPCHILDREN;						

	RECT rWindow;
	rWindow.left	= 0;							
	rWindow.right	= viewport.width;						
	rWindow.top	    = 0;								
	rWindow.bottom	= viewport.height;						

	AdjustWindowRect(&rWindow, dwStyle, false);		
													
	if (NULL == (m_hWnd = CreateWindow("Optic GL Window", caption, dwStyle, 0, 0,
						rWindow.right  - rWindow.left, rWindow.bottom - rWindow.top, 
						NULL, NULL, m_hInstance, NULL)))
		throw WindowCreationException("CreateWindow() failed");

	m_hDC = GetDC(m_hWnd);								
														
    setup_framebuffer(depth);							

    if (NULL == (m_hRC = wglCreateContext(m_hDC)))
		throw WindowCreationException("Could not obtain a rendering context");

    wglMakeCurrent(m_hDC, m_hRC);

	m_windows.push_back(this);

	resize(viewport.width,viewport.height);
}

void GLWindow::destroy() {
	
	if (m_hRC) {
		wglMakeCurrent(NULL, NULL);						
		wglDeleteContext(m_hRC);						
	}
	
	if (m_hDC) 
		ReleaseDC(m_hWnd, m_hDC);			
		
	if(m_fullscreen) {
		fullscreen(false);				
		ShowCursor(TRUE);			
	}

}

void GLWindow::resize(int width, int height) {

	if (height == 0)
		height = 1;

	m_viewport.width=width; m_viewport.height=height;
	m_viewer->setViewport(m_viewport);

	if (m_viewer->getProjectionMode() == VIEWER_MODE_ORTHO) {
		m_viewer->ortho(0.0f,width,height,0.0f,-1.0f,1.0f);
	}
	else {
		m_frustum->view(40.0f,(op_float)width/(op_float)height,4,5000);
		m_viewer->perspective(m_frustum);
	}
}

void GLWindow::showframebuffer() {
	
	glFlush();
	
	SwapBuffers(m_hDC);

}

void GLWindow::setup_framebuffer(int depth) {

	PIXELFORMATDESCRIPTOR pfd = {0}; 
    int pixelformat; 
 
    pfd.nSize = sizeof(PIXELFORMATDESCRIPTOR);			
    pfd.nVersion = 1;								
													
    pfd.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER; 
    pfd.dwLayerMask = PFD_MAIN_PLANE;					
    pfd.iPixelType = PFD_TYPE_RGBA;					
    pfd.cColorBits = depth;						
    pfd.cDepthBits = depth;					
    pfd.cAccumBits = 0;								
    pfd.cStencilBits = 0;							
 
	if (0 == (pixelformat = ChoosePixelFormat(m_hDC, &pfd)))
		throw WindowCreationException("Invalid pixel format descriptor");

	if (FALSE == (SetPixelFormat(m_hDC, pixelformat, &pfd)))
		throw WindowCreationException("Invalid pixel format descriptor");
   
}

void GLWindow::fullscreen(bool go) {

	DEVMODE dmSettings;							

	memset(&dmSettings,0,sizeof(dmSettings));

	if (go) {
		
		if (FALSE == EnumDisplaySettings(NULL,ENUM_CURRENT_SETTINGS,&dmSettings))
			throw FullScreenException("Could not obtain display modes");

		dmSettings.dmPelsWidth	= m_viewport.width;	
		dmSettings.dmPelsHeight	= m_viewport.height;		
		dmSettings.dmFields     = DM_BITSPERPEL | DM_PELSWIDTH | DM_PELSHEIGHT;

		if (DISP_CHANGE_SUCCESSFUL != ChangeDisplaySettings(&dmSettings,CDS_FULLSCREEN))
			throw FullScreenException("Could not change the display settings");
	}
	else
		ChangeDisplaySettings(NULL,0);

	m_fullscreen = go;
}

void GLWindow::show() {

	ShowWindow(m_hWnd, SW_SHOWNORMAL);				
	UpdateWindow(m_hWnd);								

	SetFocus(m_hWnd);										

}

void GLWindow::hide() {

	ShowWindow(m_hWnd,SW_HIDE);
}

void GLWindow::caption(const char* cap) {

	SetWindowText(m_hWnd,cap);
}

bool GLWindow::isFullscreen() {

	return m_fullscreen;
}

HDC GLWindow::getDeviceContext() {
	return m_hDC;
}

LRESULT CALLBACK GLWndProc(HWND hWnd,UINT uMsg, WPARAM wParam, LPARAM lParam) {
    
	PAINTSTRUCT    ps;
	GLWindow* cur_window;
	vector<Callback*> callbacks;
	int i;

	try {
	
		callbacks = locate_window(hWnd)->m_callbacks[uMsg];
		for (i = 0; i < callbacks.size(); i++) 
			callbacks[i]->call(hWnd,uMsg,wParam,lParam);

		return DefWindowProc (hWnd, uMsg, wParam, lParam); 

	} catch (...) { }

    switch (uMsg) { 
    case WM_SIZE:										
		
		cur_window = locate_window(hWnd);
		if (!cur_window->isFullscreen())
			cur_window->resize(LOWORD(lParam),HIWORD(lParam));
        
		break; 
 
	case WM_PAINT:									
		BeginPaint(hWnd, &ps);								
		EndPaint(hWnd, &ps);						
		break;
 
    case WM_CLOSE:									
        PostQuitMessage(0);							
        break; 
    } 
 
    return DefWindowProc (hWnd, uMsg, wParam, lParam); 	
}

GLWindow* locate_window(HWND hWnd) {
	
	int i;
	GLWindow* cur_window;

	for (i = 0; i < GLWindow::m_windows.size(); i++) {
		cur_window = GLWindow::m_windows[i];
		if (cur_window->getWin32Handle() == hWnd) 
			return cur_window;
	}

	return NULL;
}