#include "stdafx.h"
#include "gltexture.h"

GLTexture::GLTexture() {

	m_cubedirs[0] = GL_TEXTURE_CUBE_MAP_POSITIVE_X_ARB;
	m_cubedirs[1] = GL_TEXTURE_CUBE_MAP_NEGATIVE_X_ARB;
	m_cubedirs[2] = GL_TEXTURE_CUBE_MAP_POSITIVE_Y_ARB;
	m_cubedirs[3] = GL_TEXTURE_CUBE_MAP_NEGATIVE_Y_ARB;
	m_cubedirs[4] = GL_TEXTURE_CUBE_MAP_POSITIVE_Z_ARB;
	m_cubedirs[5] = GL_TEXTURE_CUBE_MAP_NEGATIVE_Z_ARB;

	m_data = NULL;
}

GLTexture::~GLTexture() {

	if (m_data != NULL)
		delete [] m_data;
}

GLTexture::GLTexture(const GLTexture& texture) {
	copy(texture);
}

GLTexture& GLTexture::operator =(const GLTexture& texture) {

	if (&texture != this)
		copy(texture);

	return *this;
}

void GLTexture::copy(const GLTexture& texture) {

	int i;

	for (i = 0; i < 6; i++)
		m_cubedirs[i] = texture.m_cubedirs[i];

	m_bpp = texture.m_bpp;
	m_combine = texture.m_combine;
	m_imagesize = texture.m_imagesize;
	m_id = texture.m_id;

	m_data = new op_byte[m_imagesize];
	memcpy(m_data,texture.m_data,sizeof(op_byte));

	m_name = texture.m_name;
	m_type = texture.m_type;
	m_width = texture.m_width;
	m_height = texture.m_height;
	
	for (i = 0; i < 3; i++) {
		m_wrap[i] = texture.m_wrap[i];
		m_texgen[i] = texture.m_texgen[i];
	}
}

void GLTexture::bind() {

	int gltype;

	gltype = map_to_gl(m_type);

	glEnable(gltype);
	glBindTexture(gltype,m_id);
	
	apply_filter(gltype);
	apply_wrap();
	apply_combine();
	apply_texgen();
}

void GLTexture::create(const char* filename, int type) {

	int gltype;

	switch (type) {
	case TEXTURE_TYPE_2DMIPMAP:
	case TEXTURE_TYPE_2D:

		m_type = type;
		gltype = map_to_gl(type);
		load(filename);

		glGenTextures(1,&m_id);
		glBindTexture(GL_TEXTURE_2D,m_id);

		if (type == TEXTURE_TYPE_2D)
			glTexImage2D(GL_TEXTURE_2D,0,3,m_width,m_height,0,GL_RGB,GL_UNSIGNED_BYTE,m_data);
		else {
			m_filter = TEXTURE_FILTER_TRILINEAR;
			gluBuild2DMipmaps(GL_TEXTURE_2D, 3, m_width, m_height, GL_RGB, GL_UNSIGNED_BYTE, m_data);
		}

		apply_filter(gltype);
		apply_wrap();
		apply_combine();
		apply_texgen();

		break;

	case TEXTURE_TYPE_CUBEMAP:

		m_filter = TEXTURE_FILTER_TRILINEAR;
		m_type = TEXTURE_TYPE_CUBEMAP;
		loadCubeMap(filename);

		break;
	case TEXTURE_TYPE_3D:
		// Unsupported right now
		break;
	
	case TEXTURE_TYPE_RENDER:

		m_type = type;
		
		create_rendertexture();
		break;

	};
}

void GLTexture::apply_filter(int gltype) {
	
	switch (m_filter) {
	case TEXTURE_FILTER_BILINEAR:		
		glTexParameteri(gltype,GL_TEXTURE_MIN_FILTER,GL_LINEAR);
		glTexParameteri(gltype,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
		break;
	case TEXTURE_FILTER_TRILINEAR:
		glTexParameteri(gltype,GL_TEXTURE_MIN_FILTER,GL_LINEAR_MIPMAP_LINEAR);
		glTexParameteri(gltype,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
		break;
	};
}

void GLTexture::apply_texgen() {
	
	if (m_texgen[TEXTURE_COORD_S] != TEXTURE_GEN_MANUAL) {
		glTexGenf(GL_S, GL_TEXTURE_GEN_MODE, map_to_gl(m_texgen[TEXTURE_COORD_S]));
		glEnable(GL_TEXTURE_GEN_S);
	}
	else
		glDisable(GL_TEXTURE_GEN_S);

	if (m_texgen[TEXTURE_COORD_T] != TEXTURE_GEN_MANUAL) {
		glTexGenf(GL_T, GL_TEXTURE_GEN_MODE, map_to_gl(m_texgen[TEXTURE_COORD_T]));
		glEnable(GL_TEXTURE_GEN_T);
	}
	else
		glDisable(GL_TEXTURE_GEN_T);

	if (m_texgen[TEXTURE_COORD_R] != TEXTURE_GEN_MANUAL) {
		glTexGenf(GL_R, GL_TEXTURE_GEN_MODE, map_to_gl(m_texgen[TEXTURE_COORD_R]));
		glEnable(GL_TEXTURE_GEN_R);
	}
	else
		glDisable(GL_TEXTURE_GEN_R);
		
}

void GLTexture::apply_combine() {

	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, map_to_gl(m_combine));
}

void GLTexture::apply_wrap() {
	
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, map_to_gl(m_wrap[TEXTURE_COORD_S]));
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, map_to_gl(m_wrap[TEXTURE_COORD_T]));
	
}

void GLTexture::loadCubeMap(const char* fileprefix) {

	int i;
	char suffix[4],prefix[256],name[256];

//	glEnable(GL_TEXTURE_CUBE_MAP_ARB);

	glGenTextures(1, &m_id);
	glBindTexture(GL_TEXTURE_CUBE_MAP_ARB, m_id);

	for (i = 0; i < 6; i++) {
	
		strcpy(prefix,fileprefix);
		extract_suffix(suffix,prefix);
		sprintf(name, "%s_%d.%s", prefix, i, suffix);

		load(name);

		glTexParameteri(GL_TEXTURE_CUBE_MAP_ARB, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
		glTexParameteri(GL_TEXTURE_CUBE_MAP_ARB, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

		gluBuild2DMipmaps(m_cubedirs[i], GL_RGB, m_width, m_height, GL_RGB, GL_UNSIGNED_BYTE, m_data);
		
	}

//	glDisable(GL_TEXTURE_CUBE_MAP_ARB);
}

void GLTexture::extract_suffix(char* suffix, char* buffer) {

	int length;

	length = strlen(buffer);

	if (length < 3)
		return;

	suffix[0]=buffer[length-3];
	suffix[1]=buffer[length-2];
	suffix[2]=buffer[length-1];
	suffix[3]='\0';

	buffer[length-3]='\0';

}

void GLTexture::create_rendertexture() {

	unsigned int* pTexture = NULL;											
	int size = 512, channels = 3;

	m_height = m_width = size;
	pTexture = new unsigned int [size * size * channels];
	memset(pTexture, 0, size * size * channels * sizeof(unsigned int));	

	glGenTextures(1, &m_id);								
	glBindTexture(GL_TEXTURE_2D, m_id);					
	
	glTexImage2D(GL_TEXTURE_2D, 0, channels, size, size, 0, GL_RGB, GL_UNSIGNED_INT, pTexture);						

	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);

	delete [] pTexture;	

}