#include "stdafx.h"
#include "glrenderer.h"

GLRenderer::GLRenderer(HINSTANCE hInst) : m_camera(NULL), m_texcombiner(NULL) {
	m_window = new GLWindow(hInst);
	m_api = RND_API_OPENGL;
}

void GLRenderer::init() {

	m_camera = new GLTransformer();
	m_camera->identity();

	setFillMode(RND_DEFAULT);
	setShadeModel(RND_DEFAULT);
	setZBuffer(true,RND_DEFAULT);
	setStencilFunc(false,RND_DEFAULT,0,0);
	setBlender(false,RND_DEFAULT,RND_DEFAULT);

	glFrontFace(GL_CCW);
	glClearColor(0.0,0.0,0.0,0.5);
	glClearDepth(1.0);
	glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
	glEnable(GL_COLOR_MATERIAL);
	glHint(GL_PERSPECTIVE_CORRECTION_HINT,GL_NICEST);
	glHint(GL_FOG_HINT,GL_NICEST);

	clearBuffers();
	loadExtensions();

}

void GLRenderer::render(Primitive* prim) {

	int numtex,numverts,i;
	Vertex vertex;
	Material* mat;
	
	mat = prim->getMaterial();
	numtex = mat->getNumTextures();
	numverts = prim->getNumVertices();
	mat->bindAll();
	
	if (m_texcombiner != NULL)
		m_texcombiner->enable();
	
	glBegin(map_to_gl(prim->getType()));
	
	for (i = 0; i < numverts; i++) {
		vertex = prim->getVertex(i);
		render_vertex(vertex,mat,numtex);
	}

	glEnd();

	mat->unbindTextures();
}

void GLRenderer::render_vertex(const Vertex& vertex, Material* material, int numtex) {

	int i,texstart=GL_TEXTURE0_ARB;
	Texture* texture;

	glPushAttrib(GL_COLOR_BUFFER_BIT);

	if (vertex.m_material != NULL)
		vertex.m_material->bindBase();

	if (glFogCoordfEXT != NULL)
		glFogCoordfEXT(vertex.m_fogcoord);

	glNormal3f(vertex.m_normal[0],vertex.m_normal[1],vertex.m_normal[2]);

	if (numtex > 0) {
		if (numtex > 1 && glMultiTexCoord2fARB != NULL) {
			for (i = 0; i < numtex; i++,texstart++) {
				if (i >= vertex.m_texcoords.size())
					break;
				
				if (material->getTexture(i)->getTexGen(TEXTURE_COORD_S) == TEXTURE_GEN_MANUAL)
					glMultiTexCoord2fARB(texstart,vertex.m_texcoords[i][0],vertex.m_texcoords[i][1]);
			}
		}
		else {	
			texture = material->getTexture(0);
			if (vertex.m_texcoords.size() > 0 && texture->getTexGen(TEXTURE_COORD_S) == TEXTURE_GEN_MANUAL) {				
				switch (texture->getType()) {
				case TEXTURE_TYPE_2DMIPMAP:
				case TEXTURE_TYPE_2D:
				case TEXTURE_TYPE_CUBEMAP:
				case TEXTURE_TYPE_RENDER:
					glTexCoord2f(vertex.m_texcoords[0][0],vertex.m_texcoords[0][1]);
					break;
				case TEXTURE_TYPE_3D:
					glTexCoord3f(vertex.m_texcoords[0][0],vertex.m_texcoords[0][1],vertex.m_texcoords[0][2]);
					break;
				};
			}		
		}	
	}
	
	glVertex3f(vertex.m_position[0],vertex.m_position[1],vertex.m_position[2]);

	glPopAttrib();
}

void GLRenderer::setTexCombiner(TextureCombiner* texcombiner) {

	m_texcombiner = texcombiner;
}

void GLRenderer::redirectTexture(Texture* texture) {
	
	int texsize = texture->getHeight();
	op_rect newvp;

	newvp.width = newvp.height = texsize;
	m_savedvp = m_window->getViewer()->getViewport();
	m_window->getViewer()->setViewport(newvp);

	glViewport(0,0,texsize,texsize);								
	
}

void GLRenderer::commitTexture(bool erase, Texture* texture) {

	int texsize = texture->getHeight();

	glFlush();
	glBindTexture(GL_TEXTURE_2D,texture->getID());				

	glCopyTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, 0, 0, texsize, texsize, 0);
	
	if (erase)	
		clearBuffers();	

	m_window->getViewer()->setViewport(m_savedvp);
	glViewport(0, 0, m_savedvp.width, m_savedvp.height);	
}

void GLRenderer::text(int x, int y, Font* font,const char* txt, ...) {

	char buffer[256];
	GLWindow* glwnd = (GLWindow*) m_window;
	va_list arglst;

	font->enable();

	glPushAttrib(GL_TRANSFORM_BIT | GL_VIEWPORT_BIT);

	glMatrixMode(GL_PROJECTION);						
	glPushMatrix();									
	glLoadIdentity();									
	glMatrixMode(GL_MODELVIEW);						
	glPushMatrix();										
	glLoadIdentity();									

	y = glwnd->getViewer()->getViewport().height - font->getHeight() - y;			

	glViewport(x-1,y-1,0,0);				

	glRasterPos4f(0,0,0,1);						

	glPopMatrix();									
	glMatrixMode( GL_PROJECTION );						
	glPopMatrix();										

	glPopAttrib();	
	
	glMatrixMode(GL_MODELVIEW);

	va_start(arglst, txt);				
	vsprintf(buffer, txt, arglst);			
	va_end(arglst);

	glCallLists(strlen(buffer), GL_UNSIGNED_BYTE, buffer);
	
	font->disable();
}

void* GLRenderer::newObject(int otype) {

	switch (otype) {
	case RND_OBJECT_PRIMITIVE:
		return (void*)new GLPrimitive();
	case RND_OBJECT_MATERIAL:
		return (void*)new GLMaterial();
	case RND_OBJECT_LIGHT:
		return (void*)new GLLight();
	case RND_OBJECT_FRUSTUM:
		return (void*)new GLFrustum();
	case RND_OBJECT_TEXTURE:
		return (void*)new GLTexture();
	case RND_OBJECT_TRANSFORMER:
		return (void*)new GLTransformer();
	case RND_OBJECT_VIEWER:
		return (void*)new GLViewer();
	case RND_OBJECT_FOG:
		return (void*)new GLFog();
	case RND_OBJECT_FONT:
		return (void*)new GLFont();
	case RND_OBJECT_TEXTURECOMBINER:
		return (void*)new GLTextureCombiner();
	case RND_OBJECT_VTXSHADER:
		return (void*)new GLVertexShader();
	case RND_OBJECT_RENDERPROC:
		return (void*)new GLRenderProc();
	case RND_OBJECT_PIXELSHADER:
		return (void*)new GLPixelShader();
	case RND_OBJECT_VTXARRAY:
		return (void*)new GLVertexArray();
	};

	return NULL;
}

void GLRenderer::deleteObject(void* obj) {
	delete [] obj;
	obj = NULL;
}

void GLRenderer::clearBuffers() {
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);
}

Transformer* GLRenderer::getCamera() {
	return m_camera;
}

void GLRenderer::setCamera(Transformer* camera) {

	if (m_camera != NULL) {
		delete m_camera;
		m_camera = NULL;
	}

	m_camera = camera;
}

Window* GLRenderer::getWindow() {
	return m_window;
}	

void GLRenderer::setFillMode(int fillmode) {
	
	if (fillmode == RND_DEFAULT)
		fillmode = RND_MODE_FILL;
	
	glPolygonMode(GL_FRONT_AND_BACK, map_to_gl(fillmode));
}

void GLRenderer::setShadeModel(int shade) {

	if (shade == RND_DEFAULT)
		shade = RND_SHADE_GOURAUD;

	glShadeModel(map_to_gl(shade));
}

void GLRenderer::setZBuffer(bool enable, int mask) {

	if (enable) {
		glEnable(GL_DEPTH_TEST);
		
		if (mask == RND_DEFAULT)
			mask = RND_MASK_LEQUAL;
		
		glDepthFunc(map_to_gl(mask));
	}
	else
		glDisable(GL_DEPTH_TEST);
}

void GLRenderer::setStencilFunc(bool enable, int func, int ref, int mask) {

	if (enable) {
		glEnable(GL_STENCIL_TEST);
		
		if (func == RND_DEFAULT)
			func = RND_MASK_LEQUAL;
		
		glStencilFunc(map_to_gl(func),ref,mask);
	}
	else
		glDisable(GL_STENCIL_TEST);
}

void GLRenderer::setStencilOp(int fail, int zfail, int zpass) {
	glStencilOp(map_to_gl(fail),map_to_gl(zfail),map_to_gl(zpass));
}

void GLRenderer::setBlender(bool enable, int srcf, int dstf) {

	if (enable) {
		glEnable(GL_BLEND);

		if (srcf == RND_DEFAULT)
			srcf = RND_BLEND_SRCALPHA;
		if (dstf == RND_DEFAULT)
			dstf = RND_BLEND_ONEMINUSSRCALPHA;

		glBlendFunc(map_to_gl(srcf),map_to_gl(dstf));
	}
	else
		glDisable(GL_BLEND);
}

void GLRenderer::loadExtensions() {

	glh_init_extensions("GL_ARB_multitexture");
	glh_init_extensions("GL_NV_vertex_program");
	if (GL_FALSE == glh_init_extensions("GL_EXT_fog_coord"))
		MessageBox(NULL,"Error","error",MB_OK);

}