#include "terrain.h"

Terrain::Terrain() : m_heightdata(NULL),m_fogdepth(50.0f), m_minheight(999999), m_maxheight(-9999999), m_bias(1.0f) {

	m_rendproc = (RenderProc*) m_rend->newObject(RND_OBJECT_RENDERPROC);
	m_fog = (Fog*) m_rend->newObject(RND_OBJECT_FOG);

	m_fog->setType(FOG_TYPE_VOLUMETRIC);
	m_fog->setColor(Vector(0.7f,0.7f,0.7f,1.0f));
	m_fog->setNear(0);
	m_fog->setFar(m_fogdepth);
}

Terrain::~Terrain() {

	if (m_heightdata != NULL)
		delete [] m_heightdata;
}

void Terrain::build(BuildParameters* params) {

	op_float row,col,s,t;
	op_float width,height,step;
	Vertex verts[4];
	Vector mapcoords[4];
	Vector u,v,n;
	Primitive* prow;
	TerrainBuildParameters* tparams = (TerrainBuildParameters*) params;

	width = tparams->width; height = tparams->height; step = tparams->step;
	init_heightdata(tparams->filename.c_str());
	getMinMax();

	m_rendproc->record();

	for (row = -height/2.0; row < height/2.0-step; row += step) {

		prow = (Primitive*) m_rend->newObject(RND_OBJECT_PRIMITIVE);
		prow->setType(PRIM_QUADSTRIP);
		prow->setMaterial(m_material);

		for (col = -width/2.0; col < width/2.0-step; col += step) {

			mapcoords[0] = getMapCoords(row,col,height,width);
			mapcoords[1] = getMapCoords(row,col+step,height,width);
			mapcoords[2] = getMapCoords(row+step,col+step,height,width);
			mapcoords[3] = getMapCoords(row+step,col,height,width);

			verts[0].m_position = Vector(row,col,m_heightdata[(int)mapcoords[0][1]*m_width+(int)mapcoords[0][0]]);
			verts[1].m_position = Vector(row,col+step,m_heightdata[(int)mapcoords[1][1]*m_width+(int)mapcoords[1][0]]);
			verts[2].m_position = Vector(row+step,col+step,m_heightdata[(int)mapcoords[2][1]*m_width+(int)mapcoords[2][0]]);
			verts[3].m_position = Vector(row+step,col,m_heightdata[(int)mapcoords[3][1]*m_width+(int)mapcoords[3][0]]);

			s = (verts[0].m_position[0]+width/2.0)/width; t = (verts[0].m_position[1]+height/2.0)/height;
			u = (verts[3].m_position - verts[0].m_position).normalize();
			v = (verts[1].m_position - verts[0].m_position).normalize();
			n = u.cross(v).normalize();
			verts[0].m_normal = n;
			verts[0].m_texcoords.push_back(Vector(s,t,0)); verts[0].m_texcoords.push_back(Vector(s,t,0)); verts[0].m_texcoords.push_back(Vector(s,t,0));
			verts[0].m_fogcoord = getFogCoord(verts[0].m_position[2]);

			s = (verts[3].m_position[0]+width/2.0)/width; t = (verts[3].m_position[1]+height/2.0)/height;
			u = (verts[0].m_position - verts[3].m_position).normalize();
			v = (verts[2].m_position - verts[3].m_position).normalize();
			n = u.cross(v).negate().normalize();
			verts[3].m_normal = verts[0].m_normal;
			verts[3].m_texcoords.push_back(Vector(s,t,0)); verts[3].m_texcoords.push_back(Vector(s,t,0)); verts[3].m_texcoords.push_back(Vector(s,t,0));
			verts[3].m_fogcoord = getFogCoord(verts[3].m_position[2]);

			prow->addVertex(verts[0]);
			prow->addVertex(verts[3]); 
			
			verts[0].m_texcoords.clear();
			verts[3].m_texcoords.clear();

		}

		m_rend->render(prow);
		m_rend->deleteObject(prow);
	}

	m_rendproc->stop();
}

Vector Terrain::getMapCoords(op_float row, op_float col, op_float maxrow, op_float maxcol) {

	Vector coords;

	coords[0] = ((m_width)*(col+maxcol/2.0))/maxcol;
	coords[1] = ((m_height)*(row+maxrow/2.0))/maxrow;

	return coords;
}

op_float Terrain::getFogCoord(op_float depth) {
	
	if (depth > m_fogdepth)
		return 0.0;

	return -m_bias*(depth-m_fogdepth);

}

void Terrain::setFogBias(op_float bias) {
	m_bias = bias;
}

void Terrain::getMinMax() {
	
	int i;

	for (i = 0; i < m_width*m_height; i++) {
		if (m_heightdata[i] < m_minheight)
			m_minheight = m_heightdata[i];
		
		if (m_heightdata[i] > m_maxheight)
			m_maxheight = m_heightdata[i];
	}

	m_fogdepth = (m_maxheight-m_minheight)/3.0;

}

void Terrain::render() {

	enableShaders();

	m_fog->enable();
	m_rendproc->execute();
	m_fog->disable();

	disableShaders();
}

void Terrain::init_heightdata(const char* heightfile) {

	FILE* ifile = NULL;

	if (NULL == (ifile = fopen(heightfile,"rb")))
		throw InvalidTerrainFileException(string(heightfile) + "not found");

	fread(&m_width,sizeof(int),1,ifile);
	fread(&m_height,sizeof(int),1,ifile);

	m_heightdata = new op_float[m_width*m_height];
	if (m_height*m_width != (fread(m_heightdata,sizeof(op_float),m_width*m_height,ifile))) {
		fclose(ifile);
		delete [] m_heightdata;
		throw InvalidTerrainFileException(string(heightfile) + "is an invalid terrain file");
	}

	fclose(ifile);
}

Fog* Terrain::getFog() {
	return m_fog;
}

op_float Terrain::getMaxHeight() {
	return m_maxheight;
}

AABoundingBox Terrain::getBoundingBox() {
	return AABoundingBox();
}

int Terrain::getNumComponents() {
	return 1;
}

void Terrain::getFaces(int comp, vector<Face>& faces) {

}

void Terrain::getVertices(int comp, vector<Vertex>& vertices) {

}