#include "octree.h"

vector<RenderableEntity*> Octree::m_objects;
int Octree::m_maxdepth;
int Octree::m_maxtris;

Octree::Octree() : m_isLeaf(true) {

	ACQUIRE_RENDERER(m_rend,g_engine)
}

void Octree::render() {

	Matrix model(4,4),proj(4,4);
	Frustum* frust;
	
	if (m_objects.size() == 0)
		return;

	frust = (Frustum*) m_rend->newObject(RND_OBJECT_FRUSTUM);

	m_rend->getCamera()->bind(TRANS_MATRIX_PROJECTION);
	m_rend->getCamera()->getCurrentMatrix(proj);
	m_rend->getCamera()->bind(TRANS_MATRIX_WORLD);
	m_rend->getCamera()->getCurrentMatrix(model);
	frust->calcFrustum(model,proj);

	render_help(frust);

	m_rend->deleteObject(frust);
}

void Octree::render_help(Frustum* frustum) {

	int i;

	if (m_isLeaf)
		m_rendproc->execute();
	else {
		for (i = 0; i < 8; i++) {
			if (!frustum->cullBox(m_chils[i]->m_box.getLocation(),m_chils[i]->m_box.getExtents()))
				m_chils[i]->render_help(frustum);
		}
	}

}

void Octree::build(int maxtris, int maxdepth) {

	int i;

	m_maxtris = maxtris; m_maxdepth = maxdepth;
	get_toplevel_extents();
	g_log->writeLine("OCTREE: Top Level Box: " + m_box.getLocation().toString() + " Top Level Extents: " + m_box.getExtents().toString());
	subdivide();
	for (i = 0; i < 8; i++) 
		m_chils[i]->assign_nodes(0);
}

void Octree::assign_nodes(int depth) {

	int i,tris;

	if (count_triangles() > m_maxtris && depth < m_maxdepth) {		
		subdivide();
		for (i = 0; i < 8; i++)
			m_chils[i]->assign_nodes(depth+1);
	}
	else {
		m_isLeaf = true;
		tris = finalize_node();
		g_log->writeLine("OCTREE: LEAF TRIANGLES: " + intTostring(tris,false));
	}

}

int Octree::finalize_node() {
	
	int i,j,index,tris=0,totaltris=0;
	bool ownerbind;
	vector<Vertex> verts;
	vector<Face> faces;
	RenderableEntity* obj;
	RenderableEntity* owner;
	VertexShader* vtxshader;
	PixelShader* pxlshader;
	VertexArray* array;
	Material* mat;

	m_rendproc = (RenderProc*) m_rend->newObject(RND_OBJECT_RENDERPROC);
	m_rendproc->record();
	
	for (i = 0; i < m_objects.size(); i++) {
		
		obj = m_objects[i];
		
		faces.clear(); verts.clear();
		obj->getFaces(0,faces);
		obj->getVertices(0,verts);
		owner = (RenderableEntity*) faces[0].m_owner;
		vtxshader = owner->getVertexShader();
		pxlshader= owner->getPixelShader();
		if (vtxshader)
			vtxshader->enable();
		if (pxlshader)
			pxlshader->enable();
		
		ownerbind = false;
		for (j = 0; j < obj->getNumComponents(); j++) {
			
			faces.clear(); verts.clear();
			obj->getFaces(j,faces);
			obj->getVertices(j,verts);
			index = faces[0].m_vertInd[0];
			mat = verts[index].m_material;
			
			tris=0;
			array = build_vtxarray(faces,verts,tris);
			if (tris > 0) {

				if (!ownerbind)
					owner->getMaterial()->bindAll();

				mat->bindAll();
				array->execute(VTXARRAY_TYPE_TRIANGLES);
				mat->unbindTexture(0);
				ownerbind = true;

			}
			totaltris += tris;

			m_rend->deleteObject(array);

		}

		if (ownerbind)
			owner->getMaterial()->unbindTextures();
		
		if (vtxshader)
			vtxshader->disable();
		if (pxlshader)
			pxlshader->disable();
	}
	
	m_rendproc->stop();
	
	return totaltris;
}

VertexArray* Octree::build_vtxarray(const vector<Face>& faces, const vector<Vertex>& verts, int& numtris) {
	
	int i,j,index;
	VertexArray* vtxarray;
	op_float* pverts,*pnorms,*ptexts;
	vector<UINT> preinds;
	UINT* pinds;
	Vector v1,v2,v3;
	bool texcreated;

	RND_NEWVTXARRAY(vtxarray,m_rend)
		
	for (i = 0; i < faces.size(); i++) {
		v1 = verts[faces[i].m_vertInd[0]].m_position;
		v2 = verts[faces[i].m_vertInd[1]].m_position;
		v3 = verts[faces[i].m_vertInd[2]].m_position;
		if (m_box.intersects(v1,v2,v3)) {
			preinds.push_back(faces[i].m_vertInd[0]);
			preinds.push_back(faces[i].m_vertInd[1]);
			preinds.push_back(faces[i].m_vertInd[2]);
			numtris++;
		}
	}
	if (numtris == 0)
		return vtxarray;
	
	pinds = new UINT[preinds.size()];
	for (i = 0; i < preinds.size(); i++)
		pinds[i] = preinds[i];
	
	vtxarray->setIndexPointer(preinds.size(),pinds);
	
	pverts = new op_float[verts.size()*3];
	pnorms = new op_float[verts.size()*3];

	index=0;
	for (i = 0; i < verts.size(); i++) {
		pverts[index] = verts[i].m_position[0];
		pverts[index+1] = verts[i].m_position[1];
		pverts[index+2] = verts[i].m_position[2];

		pnorms[index] = verts[i].m_normal[0];
		pnorms[index+1] = verts[i].m_normal[1];
		pnorms[index+2] = verts[i].m_normal[2];

		index += 3;
	}
	vtxarray->setVertexPointer(pverts);
	vtxarray->setNormalPointer(pnorms);

	texcreated = false;
	for (i = 0; i < verts[0].m_texcoords.size(); i++) {
		
		if (!texcreated) {
			ptexts = new op_float[verts.size()*2];
			texcreated = true;
		}

		index=0;
		for (j = 0; j < verts.size(); j++) {
			ptexts[index] = verts[j].m_texcoords[i][0];
			ptexts[index+1] = verts[j].m_texcoords[i][1];

			index += 2;
		}
		vtxarray->setTexturePointer(i,ptexts);
	}

	return vtxarray;
}

int Octree::count_triangles() {
	
	int i,j,k,numcomps,numtris=0;
	vector<Vertex> verts;
	vector<Face> faces;
	RenderableEntity* obj;
	Vector v1,v2,v3;
	
	for (i = 0; i < m_objects.size(); i++) {

		obj = m_objects[i];
		numcomps = obj->getNumComponents();

		for (j = 0; j < numcomps; j++) {

			faces.clear(); verts.clear();

			obj->getFaces(j,faces);
			obj->getVertices(j,verts);

			for (k = 0; k < faces.size(); k++) {
				v1 = verts[faces[k].m_vertInd[0]].m_position;
				v2 = verts[faces[k].m_vertInd[1]].m_position;
				v3 = verts[faces[k].m_vertInd[2]].m_position;

				if (m_box.intersects(v1,v2,v3))
					numtris++;
			}

			m_rend->deleteObject(verts[faces[0].m_vertInd[0]].m_material);
		}
	}

	return numtris;
}

void Octree::subdivide() {

	Vector loc,ext;

	m_isLeaf = false;

	loc = m_box.getLocation(); ext = m_box.getExtents();

	// Top/Left/Front
	m_chils[0] = new Octree();
	m_chils[0]->m_box.setLocation(Vector(loc[0]-(ext[0]*0.5f),loc[1]+(ext[1]*0.5f),loc[2]+(ext[2]*0.5f)));
	m_chils[0]->m_box.setExtents(ext*0.5f);

	// Top/Right/Front
	m_chils[1] = new Octree();
	m_chils[1]->m_box.setLocation(Vector(loc[0]+(ext[0]*0.5f),loc[1]+(ext[1]*0.5f),loc[2]+(ext[2]*0.5f)));
	m_chils[1]->m_box.setExtents(ext*0.5f);

	// Top/Right/Back
	m_chils[2] = new Octree();
	m_chils[2]->m_box.setLocation(Vector(loc[0]+(ext[0]*0.5f),loc[1]+(ext[1]*0.5f),loc[2]-(ext[2]*0.5f)));
	m_chils[2]->m_box.setExtents(ext*0.5f);

	// Top/Left/Back
	m_chils[3] = new Octree();
	m_chils[3]->m_box.setLocation(Vector(loc[0]-(ext[0]*0.5f),loc[1]+(ext[1]*0.5f),loc[2]-(ext[2]*0.5f)));
	m_chils[3]->m_box.setExtents(ext*0.5f);

	// Bottom/Left/Front
	m_chils[4] = new Octree();
	m_chils[4]->m_box.setLocation(Vector(loc[0]-(ext[0]*0.5f),loc[1]-(ext[1]*0.5f),loc[2]+(ext[2]*0.5f)));
	m_chils[4]->m_box.setExtents(ext*0.5f);

	// Bottom/Right/Front
	m_chils[5] = new Octree();
	m_chils[5]->m_box.setLocation(Vector(loc[0]+(ext[0]*0.5f),loc[1]-(ext[1]*0.5f),loc[2]+(ext[2]*0.5f)));
	m_chils[5]->m_box.setExtents(ext*0.5f);

	// Bottom/Right/Back
	m_chils[6] = new Octree();
	m_chils[6]->m_box.setLocation(Vector(loc[0]+(ext[0]*0.5f),loc[1]-(ext[1]*0.5f),loc[2]-(ext[2]*0.5f)));
	m_chils[6]->m_box.setExtents(ext*0.5f);

	// Bottom/Left/Back
	m_chils[7] = new Octree();
	m_chils[7]->m_box.setLocation(Vector(loc[0]-(ext[0]*0.5f),loc[1]-(ext[1]*0.5f),loc[2]-(ext[2]*0.5f)));
	m_chils[7]->m_box.setExtents(ext*0.5f);

}

void Octree::get_toplevel_extents() {

	int i;
	Vector adder,extents;
	AABoundingBox box;

	for (i = 0; i < m_objects.size(); i++) {
		adder[0] += m_objects[i]->getBoundingBox().getLocation()[0];
		adder[1] += m_objects[i]->getBoundingBox().getLocation()[1];
		adder[2] += m_objects[i]->getBoundingBox().getLocation()[2];
	}
	m_box.setLocation(adder * (1.0f/(op_float)m_objects.size()));

	for (i = 0; i < m_objects.size(); i++) {
		box = m_objects[i]->getBoundingBox();
	
		adder = (box.getLocation()+box.getExtents())-m_box.getLocation();
		adder[0] = fabs(adder[0]); adder[1] = fabs(adder[1]); adder[2] = fabs(adder[2]);

		if (adder[0] > extents[0])
			extents[0] = adder[0];
		if (adder[1] > extents[1])
			extents[1] = adder[1];
		if (adder[2] > extents[2])
			extents[2] = adder[2];
	}
	m_box.setExtents(extents);
}

void Octree::insert(RenderableEntity* ent) {
	m_objects.push_back(ent);
}