#include "mesh.h"

Mesh::Mesh() : m_model(NULL), m_rendproc(NULL) {

}

void Mesh::build(BuildParameters* params) {

	MeshBuildParameters* mbp = (MeshBuildParameters*) params;
	
	if (strstr(mbp->m_filename.c_str(),".3ds")) {
		load3DS(mbp->m_filename.c_str());
	}
	
	RND_NEWVTXARRAY(m_array,m_rend);
	m_material->setDiffuse(Triple(1,1,1));
	make_renderproc();
	make_boundingbox();
}

void Mesh::render() {

	enableShaders();
	
	m_material->bindAll(true);
	if (m_rendproc)
		m_rendproc->execute();
	m_material->unbindTextures(true);

	disableShaders();
}

int Mesh::getNumComponents() {
	return m_model->numOfObjects;
}

void Mesh::getFaces(int comp, vector<Face>& faces) {

	int i;
	Face face;

	face.m_owner = this;
	for (i = 0; i < m_model->pObject[comp].numOfFaces*3; i += 3) {
		face.m_vertInd[0] = m_model->pObject[comp].pIndices[i];
		face.m_vertInd[1] = m_model->pObject[comp].pIndices[i+1];
		face.m_vertInd[2] = m_model->pObject[comp].pIndices[i+2];

		faces.push_back(face);
	}
}

void Mesh::getVertices(int comp, vector<Vertex>& verts) {
	
	int i,index;
	Vertex vert;
	op_byte* bcolor;
	Triple color;
	Vector texcoords;
	
	vert.m_material = (Material*) m_rend->newObject(RND_OBJECT_MATERIAL);
	if (m_model->pObject[comp].materialID > 0) {
		bcolor = m_model->pMaterials[m_model->pObject[comp].materialID].color;
		color[0] = bcolor[0]/255.0f; color[1] = bcolor[1]/255.0f; color[2] = bcolor[2]/255.0f;
		vert.m_material->setDiffuse(color);
	}
	if (m_model->pObject[comp].bHasTexture) {
		index = m_model->pMaterials[m_model->pObject[comp].materialID].textureId;
		vert.m_material->insertTexture(0,m_textures[index]);
	}

	for (i = 0; i < m_model->pObject[comp].numOfVerts; i++) {
		
		vert.m_position[0] = m_model->pObject[comp].pVerts[i].x;
		vert.m_position[1] = m_model->pObject[comp].pVerts[i].y;
		vert.m_position[2] = m_model->pObject[comp].pVerts[i].z;
		vert.m_position += m_orien->getPosition();
		
		vert.m_normal[0] = m_model->pObject[comp].pNormals[i].x;
		vert.m_normal[1] = m_model->pObject[comp].pNormals[i].y;
		vert.m_normal[2] = m_model->pObject[comp].pNormals[i].z;

		if (m_model->pObject[comp].pTexVerts != NULL) {
			texcoords[0] = m_model->pObject[comp].pTexVerts[i].x;
			texcoords[1] = m_model->pObject[comp].pTexVerts[i].y;
			vert.m_texcoords.push_back(texcoords);
		}

		verts.push_back(vert);
		
		if (m_model->pObject[comp].pTexVerts != NULL)
			vert.m_texcoords.pop_back();
	}

}

AABoundingBox Mesh::getBoundingBox() {
	return m_box;
}

void Mesh::make_boundingbox() {
	
	int i,j,totalverts=0;
	Vector adder,extents;

	for (i = 0; i < m_model->numOfObjects; i++) {
		for (j = 0; j < m_model->pObject[i].numOfVerts; j++) {
			adder[0] += m_model->pObject[i].pVerts[j].x;
			adder[1] += m_model->pObject[i].pVerts[j].y;
			adder[2] += m_model->pObject[i].pVerts[j].z;
			totalverts++;
		}
	}
	m_box.setLocation(adder*(1.0f/(op_float)totalverts));

	for (i = 0; i < m_model->numOfObjects; i++) {
		for (j = 0; j < m_model->pObject[i].numOfVerts; j++) {

			adder[0] = m_model->pObject[i].pVerts[j].x;
			adder[1] = m_model->pObject[i].pVerts[j].y;
			adder[2] = m_model->pObject[i].pVerts[j].z;

			adder = adder - m_box.getLocation();
			if (adder[0] > extents[0])
				extents[0] = adder[0];
			if (adder[1] > extents[1])
				extents[1] = adder[1];
			if (adder[2] > extents[2])
				extents[2] = adder[2];

		}
	}
	m_box.setExtents(extents);
}

void Mesh::make_renderproc() {
	
	int i,index;
	op_float* rawptr;
	op_byte* bcolor;
	unsigned int* indptr;
	Triple color;
	Material* mat;

	m_rendproc = (RenderProc*) m_rend->newObject(RND_OBJECT_RENDERPROC);
	mat = (Material*) m_rend->newObject(RND_OBJECT_MATERIAL);

	m_rendproc->record();
	for (i = 0; i < m_model->numOfObjects; i++) {

		if (m_model->pObject[i].materialID > 0) {
			bcolor = m_model->pMaterials[m_model->pObject[i].materialID].color;
			color[0] = bcolor[0]/255.0f; color[1] = bcolor[1]/255.0f; color[2] = bcolor[2]/255.0f;
			mat->setDiffuse(color);
		}
		if (m_model->pObject[i].bHasTexture) {
			index = m_model->pMaterials[m_model->pObject[i].materialID].textureId;
			mat->insertTexture(0,m_textures[index]);
		}
		
		mat->bindAll();

		rawptr = (op_float*)m_model->pObject[i].pVerts;
		m_array->setVertexPointer(rawptr);
		rawptr = (op_float*)m_model->pObject[i].pNormals;
		m_array->setNormalPointer(rawptr);
		rawptr = (op_float*)m_model->pObject[i].pTexVerts;
		m_array->setTexturePointer(0,rawptr);
		
		indptr = m_model->pObject[i].pIndices;
		m_array->setIndexPointer(m_model->pObject[i].numOfFaces*3,indptr);
		m_array->execute(VTXARRAY_TYPE_TRIANGLES);	
		
		if (m_model->pObject[i].bHasTexture) {
			mat->unbindTexture(0);
			mat->removeTexture(0);
		}
	}
	
	//mat->unbindTextures();
	m_rendproc->stop();
}

bool Mesh::QueryInterface(const EntityType type, void** obj) {

	if (type == Entity::RENDERABLE) {
		*obj = (RenderableEntity*)(this);
		return true;
	}

	return false;
}

void Mesh::load3DS(const char* filename) {

	tChunk pri_chunk;

	m_model = new t3DModel;
	m_model->numOfMaterials = 0; m_model->numOfObjects = 0;

	if (NULL == (m_file = fopen(filename,"rb"))) {
		MessageBox(NULL,"Hello","hello",MB_OK);
		return;
	}

	read_3ds_chunk(&pri_chunk);

	if (pri_chunk.ID != MESH_3DS_PRIMARY) {
		MessageBox(NULL,"Hello","hello",MB_OK);
		return;
	}

	proc_3ds_chunk(&pri_chunk);
	proc_3ds_normals();
	proc_3ds_indices();
	proc_3ds_textures();

	fclose(m_file);
}

void Mesh::proc_3ds_chunk(tChunk* prevChunk) {

	tChunk cur_chunk;
	tChunk scratch_chunk;
	t3DObject obj= {0};
	tMaterialInfo texture = {NULL};
	unsigned int version=0;
	int buffer[50000]={0};

	strcpy(texture.strFile,"empty");
	obj.bHasTexture = false;
	obj.pNormals = obj.pVerts = NULL; obj.pTexVerts = NULL; obj.pIndices = NULL; obj.pFaces = NULL;
	while (prevChunk->bytesRead < prevChunk->length) {

		read_3ds_chunk(&cur_chunk);

		switch (cur_chunk.ID) {
		case MESH_3DS_OBJECTINFO:

			read_3ds_chunk(&scratch_chunk);
			scratch_chunk.bytesRead += fread(buffer,1,scratch_chunk.length-scratch_chunk.bytesRead,m_file);
			cur_chunk.bytesRead += scratch_chunk.bytesRead;
			proc_3ds_chunk(&cur_chunk);

			break;
		case MESH_3DS_MATERIAL:

			m_model->numOfMaterials++;
			m_model->pMaterials.push_back(texture);

			proc_3ds_mat_chunk(&cur_chunk);

			break;
		case MESH_3DS_OBJECT:

			m_model->numOfObjects++;
			m_model->pObject.push_back(obj);
			memset(&(m_model->pObject[m_model->numOfObjects-1]),0,sizeof(t3DObject));

			cur_chunk.bytesRead += read_3ds_string(m_model->pObject[m_model->numOfObjects-1].strName);
			proc_3ds_object_chunk(&(m_model->pObject[m_model->numOfObjects-1]),&cur_chunk);

			break;
		case MESH_3DS_VERSION: 	
			
			cur_chunk.bytesRead += fread(&version,1,cur_chunk.length-cur_chunk.bytesRead,m_file);
			if (version > 3) 
				MessageBox(NULL,"Hello","hello",MB_OK);
			
			break;
		case MESH_3DS_EDITKEYFRAME:
			cur_chunk.bytesRead += fread(buffer,1,cur_chunk.length-cur_chunk.bytesRead,m_file);
			break;
		default:
			cur_chunk.bytesRead += fread(buffer,1,cur_chunk.length-cur_chunk.bytesRead,m_file);
			break;

		};

		prevChunk->bytesRead += cur_chunk.bytesRead;
	}

}


void Mesh::proc_3ds_object_chunk(t3DObject* obj, tChunk* prevChunk) {
	
	tChunk cur_chunk;
	int buffer[50000]={0};

	while (prevChunk->bytesRead < prevChunk->length) {

		read_3ds_chunk(&cur_chunk);

		switch (cur_chunk.ID) {
		case MESH_3DS_OBJECT_MESH:
			proc_3ds_object_chunk(obj,&cur_chunk);
			break;
		case MESH_3DS_OBJECT_VERTICES:
			read_3ds_vertices(obj,&cur_chunk);
			break;
		case MESH_3DS_OBJECT_FACES:
			read_3ds_vertex_indices(obj,&cur_chunk);
			break;
		case MESH_3DS_OBJECT_MATERIAL:
			read_3ds_object_material(obj,&cur_chunk);
			break;
		case MESH_3DS_OBJECT_UV:
			read_3ds_texture_coords(obj,&cur_chunk);
			break;
		default:
			cur_chunk.bytesRead += fread(buffer,1,cur_chunk.length-cur_chunk.bytesRead,m_file);
			break;
		};

		prevChunk->bytesRead += cur_chunk.bytesRead;
	}

}

void Mesh::proc_3ds_mat_chunk(tChunk* prevChunk) {

	tChunk cur_chunk;
	int buffer[50000]={0};

	while (prevChunk->bytesRead < prevChunk->length) {

		read_3ds_chunk(&cur_chunk);

		switch (cur_chunk.ID) {
		case MESH_3DS_MATNAME:
			cur_chunk.bytesRead += fread(m_model->pMaterials[m_model->numOfMaterials-1].strName,1,cur_chunk.length-cur_chunk.bytesRead,m_file);
			break;
		case MESH_3DS_MATDIFFUSE:
			read_3ds_color_chunk(&(m_model->pMaterials[m_model->numOfMaterials-1]),&cur_chunk);
			break;
		case MESH_3DS_MATMAP:
			proc_3ds_mat_chunk(&cur_chunk);
			break;
		case MESH_3DS_MATMAPFILE:
			cur_chunk.bytesRead += fread(m_model->pMaterials[m_model->numOfMaterials-1].strFile,1,cur_chunk.length-cur_chunk.bytesRead,m_file);
			break;
		default:
			cur_chunk.bytesRead += fread(buffer,1,cur_chunk.length-cur_chunk.bytesRead,m_file);
			break;
		};

		prevChunk->bytesRead += cur_chunk.bytesRead;
	}

}

void Mesh::read_3ds_chunk(tChunk* chunk) {
	
	chunk->bytesRead = fread(&(chunk->ID),1,2,m_file);
	chunk->bytesRead += fread(&(chunk->length),1,4,m_file);

}

int Mesh::read_3ds_string(char* buffer) {

	int index=0;

	fread(buffer,1,1,m_file);
	while (*(buffer+index++) != 0)
		fread(buffer+index,1,1,m_file);

	return strlen(buffer)+1;
}

void Mesh::read_3ds_color_chunk(tMaterialInfo* mat, tChunk* chunk) {

	tChunk tmp_chunk;

	read_3ds_chunk(&tmp_chunk);
	tmp_chunk.bytesRead += fread(mat->color,1,tmp_chunk.length-tmp_chunk.bytesRead,m_file);

	chunk->bytesRead += tmp_chunk.bytesRead;
}

void Mesh::read_3ds_vertex_indices(t3DObject* obj, tChunk* chunk) {

	unsigned short index=0;
	int i,j;

	chunk->bytesRead += fread(&(obj->numOfFaces),1,2,m_file);
	obj->pFaces = new tFace[obj->numOfFaces];
	memset(obj->pFaces,0,sizeof(tFace)*obj->numOfFaces);

	for (i = 0; i < obj->numOfFaces; i++) {
		for (j = 0; j < 4; j++) {
			chunk->bytesRead += fread(&index,1,sizeof(index),m_file);

			if (j < 3) 
				obj->pFaces[i].vertIndex[j] = index;
		}
	}
}

void Mesh::read_3ds_texture_coords(t3DObject* obj, tChunk* chunk) {

	chunk->bytesRead += fread(&obj->numTexVertex,1,2,m_file);
	obj->pTexVerts = new tVector2[obj->numTexVertex];
	chunk->bytesRead += fread(obj->pTexVerts,1,chunk->length-chunk->bytesRead,m_file);
}

void Mesh::read_3ds_vertices(t3DObject* obj, tChunk* chunk) {

	int i;
	float tmp_y;

	chunk->bytesRead += fread(&(obj->numOfVerts),1,2,m_file);

	obj->pVerts = new tVector3[obj->numOfVerts];
	memset(obj->pVerts,0,sizeof(tVector3)*obj->numOfVerts);

	chunk->bytesRead += fread(obj->pVerts,1,chunk->length-chunk->bytesRead,m_file);

	for (i = 0; i < obj->numOfVerts; i++) {
		tmp_y = obj->pVerts[i].y;
		obj->pVerts[i].y = obj->pVerts[i].z;
		obj->pVerts[i].z = -tmp_y;
	}

}

void Mesh::read_3ds_object_material(t3DObject* obj, tChunk* chunk) {

	char matname[255];
	int buffer[50000]={0};
	int i;

	chunk->bytesRead += read_3ds_string(matname);
	for (i = 0; i < m_model->numOfMaterials; i++) {

		if (strcmp(matname,m_model->pMaterials[i].strName) == 0) {
			obj->materialID = i;

			if (strcmp(m_model->pMaterials[i].strFile,"empty") != 0) 
				obj->bHasTexture = true;
			break;
		}
		else
			obj->materialID = -1;
	}

	chunk->bytesRead += fread(buffer,1,chunk->length-chunk->bytesRead,m_file);
}

void Mesh::proc_3ds_indices() {
	
	int i,j,index=0;
	t3DObject* obj;

	for (i = 0; i < m_model->numOfObjects; i++) {
		index=0;
		obj = &m_model->pObject[i];
		obj->pIndices = new UINT[3*obj->numOfFaces];
		for (j = 0; j < obj->numOfFaces; j++) {
			obj->pIndices[index] = obj->pFaces[j].vertIndex[0];
			obj->pIndices[index+1] = obj->pFaces[j].vertIndex[1];
			obj->pIndices[index+2] = obj->pFaces[j].vertIndex[2];

			index += 3;
		}
	}
}

void Mesh::proc_3ds_normals() {

	int i,j,k,shared;
	Vector u,v,n,zero,sum;
	tVector3 poly[3];
	Vector vPoly[3];
	Vector* normals,*tmp_normals;
	t3DObject* obj;

	if (m_model->numOfObjects <= 0)
		return;

	for (i = 0; i < m_model->numOfObjects; i++) {

		obj = &(m_model->pObject[i]);

		normals = new Vector[obj->numOfFaces];
		tmp_normals = new Vector[obj->numOfFaces];
		obj->pNormals = new tVector3[obj->numOfVerts];

		for (j = 0; j < obj->numOfFaces; j++) {

			poly[0] = obj->pVerts[obj->pFaces[j].vertIndex[0]];
			vPoly[0][0]=poly[0].x; vPoly[0][1]=poly[0].y;vPoly[0][2]=poly[0].z;
			poly[1] = obj->pVerts[obj->pFaces[j].vertIndex[1]];
			vPoly[1][0]=poly[1].x; vPoly[1][1]=poly[1].y;vPoly[1][2]=poly[1].z;
			poly[2] = obj->pVerts[obj->pFaces[j].vertIndex[2]];
			vPoly[2][0]=poly[2].x; vPoly[2][1]=poly[2].y;vPoly[2][2]=poly[2].z;

			u = vPoly[0] - vPoly[2];
			v = vPoly[2] - vPoly[1];
			n = u.cross(v);
			tmp_normals[j] = n;
			normals[j] = n.normalize();

		}

		for (j = 0; j < obj->numOfVerts; j++) {
			for (k = 0; k < obj->numOfFaces; k++) {

				if (obj->pFaces[k].vertIndex[0] == j || obj->pFaces[k].vertIndex[1] == j || obj->pFaces[k].vertIndex[2] == j) {
					sum = sum + tmp_normals[k];
					shared++;
				}
			}

			n = (sum * (1.0f/(float)-shared)).normalize();
			poly[0].x = n[0]; poly[0].y = n[1]; poly[0].z = n[2];

			obj->pNormals[j] = poly[0]; 
			sum = zero;
			shared = 0;
		}

		delete [] normals;
		delete [] tmp_normals;
	}
}

void Mesh::proc_3ds_textures() {

	int i;
	Texture* tex;

	for (i = 0; i < m_model->numOfMaterials; i++) {
		if (strcmp(m_model->pMaterials[i].strFile,"empty") != 0) {
			RND_NEW2DMIPTEXTURE(tex,m_rend,m_model->pMaterials[i].strFile)
			m_textures.push_back(tex);
			m_model->pMaterials[i].textureId = m_textures.size()-1;
		}
	}
}