#include "displaymanager.h"
#include "../core/demo.h"

DisplayManager::DisplayManager(Renderer* rend, InputManager* inp) : m_rend(rend), m_inp(inp) {

}

void DisplayManager::init() {

	vector<int> channels;
	Camera* cam;
	VertexShader* shader;
		
	m_fonts[DMAN_FONT_TNR14B] = (Font*) m_rend->newObject(RND_OBJECT_FONT);
	m_fonts[DMAN_FONT_TNR14B]->create("Times New Roman",14,FONT_STYLE_BOLD,Triple(1,1,1),m_rend->getWindow());
	
	m_fonts[DMAN_FONT_TNR20B] = (Font*) m_rend->newObject(RND_OBJECT_FONT);
	m_fonts[DMAN_FONT_TNR20B]->create("Times New Roman",20,FONT_STYLE_BOLD,Triple(1,1,1),m_rend->getWindow());
	
	m_inp->getChannels(channels,m_inp->getCommonDevice(INPUT_COMMON_MOUSE));
	m_mouse = channels[0];
	
	cam = new Camera(m_rend,m_inp);
	m_camera = new BasicViewer(m_rend,cam);
	
	shader = createVertexShader("basic.vsd");
	shader->trackMatrix(0,TRANS_MATRIX_COMPOSITE, VTXSHADER_TRANS_IDENTITY);
	shader->trackMatrix(4,TRANS_MATRIX_WORLD, VTXSHADER_TRANS_IDENTITY);
	shader->trackMatrix(8,TRANS_MATRIX_WORLD, VTXSHADER_TRANS_INVERSETRANSPOSE);
	shader->trackMatrix(12,TRANS_MATRIX_PROJECTION, VTXSHADER_TRANS_IDENTITY);
	shader->trackMatrix(16,TRANS_MATRIX_WORLD,VTXSHADER_TRANS_INVERSE);
	shader->setConstant(22,Vector(0,0,0,0));
	m_shaders[DMAN_VTXSHADER_BASIC] = shader;

	shader = createVertexShader("terrain.vsd");
	m_shaders[DMAN_VTXSHADER_TERRAIN] = shader;

	shader = createVertexShader("anisolight.vsd");
	m_shaders[DMAN_VTXSHADER_ANISO] = shader;

	shader = createVertexShader("speccube.vsd");
	m_shaders[DMAN_VTXSHADER_GLARECUBE] = shader;

	shader = createVertexShader("ripple.vsd");
	m_shaders[DMAN_VTXSHADER_RIPPLECUBE] = shader;

	m_rend->deleteObject(shader);
}

void DisplayManager::renderScene(void* demo) {

	// Pre-process scene
	initFrame();

	((Demo*)demo)->render();

	// Post-process scene
	//drawMouse();
}

void DisplayManager::renderEntity(RenderableEntity* obj) {

	m_rend->getCamera()->push();

	obj->getOrientation()->enable();
	obj->render();

	m_rend->getCamera()->pop();
}

void DisplayManager::initFrame() {

	RenderableEntity* pRnd;

	m_rend->clearBuffers();

	if (m_camera->getCamera()->isLockedTrackball())
		m_camera->getCamera()->updateTrackball();

	m_rend->getCamera()->identity();
	m_camera->view();

	if (m_camera->QueryInterface(Entity::RENDERABLE,(void**)&pRnd))
		renderEntity(pRnd);
}

void DisplayManager::setViewer(ViewableEntity* m_viewer) {
	m_camera = m_viewer;
}

ViewableEntity* DisplayManager::getViewer() {
	return m_camera;
}

Texture* DisplayManager::create2DMipmap(const char* filename) {

	Texture* tex;

	RND_NEW2DMIPTEXTURE(tex,m_rend,filename)

	return tex;
}

Texture* DisplayManager::create2DTexture(const char* filename) {

	Texture* tex;
	
	RND_NEW2DTEXTURE(tex,m_rend,filename)

	return tex;
}

Texture* DisplayManager::createCubeMap(const char* filename) {

	Texture* tex;

	tex = (Texture*) m_rend->newObject(RND_OBJECT_TEXTURE);
	tex->create(filename,TEXTURE_TYPE_CUBEMAP);
	tex->setTexGen(TEXTURE_COORD_S | TEXTURE_COORD_T | TEXTURE_COORD_R, TEXTURE_GEN_CUBEMAP);

	return tex;
}

VertexShader* DisplayManager::createVertexShader(const char* filename) {

	VertexShader* vtx;

	RND_NEWVTXSHADER(vtx,m_rend,filename)

	return vtx;
}

PixelShader* DisplayManager::createPixelShader(const char* filename) {
	
	PixelShader* pxl;

	RND_NEWPIXELSHADER(pxl,m_rend,filename)

	return pxl;
}

Font* DisplayManager::createFont(const char* face, int size) {

	Font* font;

	font = (Font*) m_rend->newObject(RND_OBJECT_FONT);
	font->create(face,size,FONT_STYLE_BOLD,Triple(1,1,1),m_rend->getWindow());

	return font;
}

Font* DisplayManager::getStoredFont(int font) {

	return m_fonts[font];

}

VertexShader* DisplayManager::getStoredVertexShader(int shader) {

	return m_shaders[shader];
}

void DisplayManager::drawMouse() {

	Vector pos = m_inp->pollChannel(m_mouse).m_trackposition;

	m_rend->text((int)pos[0],(int)pos[1],m_fonts[DMAN_FONT_TNR20B],"M");
}