#include "camera.h"

Camera::Camera(Renderer* rend, InputManager* inp) : m_rend(rend), m_input(inp), m_tracklock(false) {
	
}

void Camera::enable() {
	m_rend->getCamera()->lookat(m_eye,m_at,m_up);
}

void Camera::lookat(const Vector& eye, const Vector& at, const Vector& up) {
	m_at = at; m_eye = eye; m_up = up;
}

void Camera::setViewPoint(const Vector& at) {
	m_eye = at;
}

Vector Camera::getEyePoint() {
	return m_eye;
}

void Camera::moveOnView(op_float mag) {
	
	Vector view = (m_at - m_eye).normalize();

	m_eye += view*mag;
	m_at += view*mag;
}

void Camera::lockToTrackball(op_inphnd mouse) {

	Vector middle_pos;
	vector<void*> params;

	if (mouse < 0) {
		m_tracklock = false;
		return;
	}

	m_mouse = mouse;
	m_tracklock = true;

	middle_pos[0] = m_rend->getWindow()->getViewer()->getViewport().width/2;
	middle_pos[1] = m_rend->getWindow()->getViewer()->getViewport().height/2;

	params.push_back(&middle_pos);
	m_input->executeOverChannel(m_mouse,INPUT_MOUSE_EXECUTE_SETPOSITION,params);
}

void Camera::updateTrackball() {
									
	op_float angleY,angleZ;						
	Vector mouse_pos,middle_pos,axis;
	vector<void*> params;

	middle_pos[0] = m_rend->getWindow()->getViewer()->getViewport().width/2;
	middle_pos[1] = m_rend->getWindow()->getViewer()->getViewport().height/2;

	mouse_pos = m_input->pollChannel(m_mouse).m_trackposition;					
	
	if(((int)mouse_pos[0] == (int)middle_pos[0]) && ((int)mouse_pos[1] == (int)middle_pos[1]))
		return;
	
	params.push_back(&middle_pos);
	m_input->executeOverChannel(m_mouse,INPUT_MOUSE_EXECUTE_SETPOSITION,params);						
	
	angleY = (op_float)((middle_pos[0]-mouse_pos[0]))/500.0f;		
	angleZ = (op_float)((middle_pos[1]-mouse_pos[1]))/500.0f;			
		
	axis = (m_eye-m_at).cross(m_up).normalize();
	
	rotate_view(angleZ,axis);
	rotate_view(angleY,Vector(0,1,0));
}

bool Camera::isLockedTrackball() {
	return m_tracklock;
}

bool Camera::QueryInterface(const EntityType type, void** pObj) {
	
	if (type == Entity::GENERIC) {
		*pObj = (Entity*)(this);
		return true;
	}
	
	return false;
}

void Camera::rotate_view(op_float angle, Vector& v) {

	Vector rot_v,view;
	op_float cosTheta,sinTheta;

	view = m_eye-m_at;		
	
	cosTheta = (op_float)cos(angle);
	sinTheta = (op_float)sin(angle);
	
	rot_v[0] = (cosTheta + (1 - cosTheta) * v[0] * v[0]) * view[0];
	rot_v[0] += ((1 - cosTheta) * v[0] * v[1] - v[2] * sinTheta) * view[1];
	rot_v[0] += ((1 - cosTheta) * v[0] * v[2] + v[1] * sinTheta) * view[2];
	
	rot_v[1] = ((1 - cosTheta) * v[0] * v[1] + v[2] * sinTheta)	* view[0];
	rot_v[1] += (cosTheta + (1 - cosTheta) * v[1] * v[1]) * view[1];
	rot_v[1] += ((1 - cosTheta) * v[1] * v[2] - v[0] * sinTheta) * view[2];
	
	rot_v[2] = ((1 - cosTheta) * v[0] * v[2] - v[1] * sinTheta)	* view[0];
	rot_v[2] += ((1 - cosTheta) * v[1] * v[2] + v[0] * sinTheta) * view[1];
	rot_v[2] += (cosTheta + (1 - cosTheta) * v[2] * v[2]) * view[2];
	
	m_at = m_eye + rot_v;
}