#!/usr/local/bin/perl5 -w

use strict;
use Symbol;
use lib '/usr/lw2j/private/Stat/Perl';
use FileHandle;

require Wrapper;
require Transform;
require Numerical;
require UniRand;
require GKQuantile;

my $fn = shift @ARGV;
my $fh = gensym();

if (defined($fn)) {
  open($fh, $fn) || die "open (r) '$fn':  $!";
} else {
  $fh = \*STDIN;
}

autoflush STDOUT 1;

my $wrapper = new Wrapper($fh);
my $dims    = $wrapper->get_dims();
my $ct      = $wrapper->get_count();
my $quantile_ct = 1000;

my $NAME_IDX  = $UniRand::MATCH_INDICES{"name"};
my $PARAM_IDX = $UniRand::MATCH_INDICES{"params"};
my $CHI_IDX   = $UniRand::MATCH_INDICES{"chi"};
my $CDF_IDX   = $UniRand::MATCH_INDICES{"cdf"};
my $BIC_IDX   = $UniRand::MATCH_INDICES{"BIC"};

my @dist_xfs = @UniRand::DIST_LIST_PLAIN;
my @other_xfs = @Transform::XF_LIST;


{
  print ";;; ==== BEGIN TRANSFORMATION LIST BLOCK ====\n";
  print ";;; Estimators in use:\n;;\t";
  print join("\n;;\t", map { $_->[0]} @dist_xfs);
  print "\n;;; XForms in use:\n;;\t";
  print join("\n;;\t", map { $_->[0]} @other_xfs);
  print "\n;;; ==== END TRANSFORMATION LIST BLOCK ====\n\n";
}

{
  my @transformers = ();

  my $i=0;
  for ($i=0; $i < $dims; $i++) {
    my @col = $wrapper->get_col($i);
    my $xf  = undef;

    $transformers[$i] = +[];

    # Make sure that it's not constant...
    my $sameval  = $col[0];
    my $constant = 1;

    constant_check:  foreach (@col) {
      if ($_ != $sameval) {
        $constant = 0;
        last constant_check;
      }
    }

    if (!$constant) {
      if ($ct > $quantile_ct) {
        # Sample.
        my $GKQ = GKQuantile::new GKQuantile();

        $GKQ->insert(@col);
        @col = $GKQ->quantize($quantile_ct);
      }

      my @best_fits = UniRand::best_match(\@col, 0.3, undef, undef, @dist_xfs);
      my $chi_flag  = 0;

      foreach $xf (@best_fits) {
        my $name      = $xf->[$NAME_IDX];
        my $cdf       = $xf->[$CDF_IDX];
        my $params    = $xf->[$PARAM_IDX];
        my $chi       = $xf->[$CHI_IDX];
        my $bic       = $xf->[$BIC_IDX];


        if (defined($cdf)) {
          # Found a fit.  Might be horrible, but it's a fit.
          push @{$transformers[$i]}, +[ $name, $cdf, $params, $chi, $bic ];

          if (defined($chi) && ($chi >= 0.01)) {
            # Good match, assuming the table hasn't been extended.
            $chi_flag = 1;
          }
        }
      }

      # Also apply the non-distribution transformations.

      if (!($chi_flag)) {
        foreach $xf (@other_xfs) {
          my $name = $xf->[0];
          my $est  = $xf->[1];

          my ($cdf, $params, $inverse) = &$est(\@col);

          if (defined($cdf)) {
            push @{$transformers[$i]}, +[ $name, $cdf, $params, undef, undef ];
          }
        }
      }
    }
  }

  print ";;; ==== BEGIN TRANSFORMATION CHOICE BLOCK ====\n";
  for ($i=0; $i < $dims; $i++) {
    my $listref = $transformers[$i];
    my $tuple = undef;

    foreach $tuple (@{$transformers[$i]}) {
      my ($name, $cdf, $params, $chi, $bic) = @$tuple;
      print ";;;  <$i, $name, ", join(", ", @$params), ">";

      if (defined($bic)) {
        print " <BIC=$bic>";
      }

      if (defined($chi)) {
        print " <chi=$chi>";
      }

      print "\n";
    }
  }
  print ";;; ==== END TRANSFORMATION CHOICE BLOCK ====\n\n";

  {
    my $j=0;
    for ($j=0; $j < $ct; $j++) {
      my @orig = $wrapper->get_obj($j);
      my @xf   = ();
      for ($i=0; $i < $dims; $i++) {
        my $listref = $transformers[$i];
        my $tuple = undef;

        foreach $tuple (@{$transformers[$i]}) {
          my ($name, $cdf, $params, $chi, $bic) = @$tuple;

          push @xf, &$cdf($orig[$i]);
        }
      }
      print join(" ", @xf), "\n";
    }
  }
}


if (defined($fn)) {
  close($fh) || die "close (r) '$fn':  $!";
}

exit 0;
