#!/usr/local/bin/perl5 -w
#
# Purpose:
#  Versatile argument parsing support.  No doubt, it's an
#  oft re-invented wheel.  Give it enough information, and
#  it'll not only parse arguments for you (invoking hooks
#  that you set), but it'll also print program invocation
#  help as need be.
##################################################################
# RCS:
#
# $Id: ParseArgs.pm,v 1.4 2003/01/28 16:34:21 lw2j Exp $
# $Log:	ParseArgs.pm,v $
# Revision 1.4  2003/01/28  16:34:21  lw2j
# Fixed indentifying, clarified what sanity-checking functions
# are supposed to return (nonzero=good).
# 
# Revision 1.3  2002/10/07  12:25:41  lw2j
# Applied saner indenter script.
#
# Revision 1.2  2002/08/12  17:55:55  lw2j
# Identified.
#
# Revision 1.1  2002/06/26  16:50:24  lw2j
# Initial revision
#
# Revision 1.5  2002/01/03  17:09:11  lw2j
# Corrected prototype for parse_args -- ($@), was ($$).
#
# Revision 1.4  2001/12/15  15:22:58  lw2j
# Removed extraneous -- and - in examples.
#
# Revision 1.3  2001/12/13  17:47:32  lw2j
# Added 'show_settings'.  Made the current column number internal to
# the print wrapper.  Added support for prefixing each line.
#
# Revision 1.2  2001/12/12  22:47:31  lw2j
# Added documentation/examples, backslash handling, --foo=bar syntax,
# validator setting as well.
#
# Revision 1.1  2001/12/12  20:09:15  lw2j
# Initial revision
#
##################################################################



##################################################################
#
# Package Header
#
##################################################################

package ParseArgs;
require Exporter;

use File::Basename;
use English;
use strict;
use FileHandle;




##########################################################################
# Design and Usage
##########################################################################
#
# The ParseArgs module tracks several pieces of data.
#
# The most important is the arg_table.  The arg_table is a reference to
# an ordered list, where each member of the list corresponds to a
# possible command-line argument (for instance, -b, --foo).
#
# Each item is actually a reference to another list.  This list has the
# following fields, in the order specified by $self->{"indices"} (used
# just in case we want to add/remove/reorder fields).
#
#     opts => The ways this option can be specified, without the -- or -.
#             If there is only one, this can be a scalar:  "f".  Otherwise,
#             it must be a list reference, e.g. +[ "f", "file" ] which
#             means "This argument can be specified as -f or --file".
#
#     desc => A scalar string description, such as "The input file."  This
#             description will be word-wrapped automatically.
#
#   argmap => If this argument takes any following values, this must be a
#             list reference, where each item corresponds to a setting,
#             in order.  If there is only one value, this may be a list
#             reference instead of a reference to a list of list
#             references.
#
#             These references have the fields described in arg_indices.
#
#     valref => A reference to the corresponding program value.
#               If this is a reference to a list, it will consume ALL
#               values up to the next - or --.
#
#     valdef => Default value for valref.  This may be undefined.
#    valdesc => Like desc, description text.
#     valopt => Whether or not this value is optional (1 = optional, 0 =
#               mandator), if the parent argument is specified.
#     valsan => Reference to a subroutine which, if this argument receives
#               this value, will be invoked for "sanity checking".  The
#               argument passed to the &$valsan subroutine will be the
#               valuie given.  If multiple values are given as a list,
#               &$valsan will be given a reference to the entire list.
#               It should return non-zero if the check passes, 0 otherwise.
#
#               Note that valsan is processed as each argument is scanned.
#               Be careful about mandating relationships between related
#               variables.
#
#   toggle => If defined, it must be a reference to a scalar, which will
#             receive a 1 if this argument is specified.  It will receive
#             a 0 if --disable-argname is specified.  --enable-argname is
#             also recognized, although it's not too useful since --argname
#             does the same thing.
#
#  default => The default value for a toggle variable.  Ignored if toggle
#             is undefined.
#
# optional => If 0, this argument is mandatory (either the normal form,
#             or, for toggles, the --disable-argname form), must be
#             explicitly provided.  1 permits the user to not specify
#             this argument.
#
#
# An example is:
#
# +[+[+[ "f", "file" ],
#      "Specify input filename",
#     +[ +[ \$infile_name, "data", "Filename containing interesting data",
#        0, \&validate_filename ]],
#     undef,
#     undef,
#     1
#    ],
##  ...
# This first entry means that there is an optional argument (the second 1)
# which may be specified as either -f or --file.  If either is given,
# the user MUST specify an additional value.  $infile_name will be changed
# to reflect this value, and then it'll be passed to &validate_filename,
# which will cause argument parsing to abort if it returns a 0.  The
# two undefs mean that there is no toggle variable associated with the
# presence or absence of --file.   If -f isn't specified at all,
# $infile_name will contain "data", assuming it wasn't overwritten
# elsewhere.
#
# ...
# +[ "risky",
#    "Set this to enable risky optimizations",
#    +[ +[ \$risky, 0, "Binary flag", 1, undef ]],
#    \$risky,
#    0,
#    1
#  ],
# ...
# This entry specifies a flag '--risky'.  It's a toggle that defaults to
# 0, and can be explicitly specified by saying --risky (1),
# --disable-risky (0), or --enable-risky (1).  In addition, it takes an
# optional value, which also sets $risky, so the person can specify
# "--risky 0" to disable it, or "--risky=1" to enable it, and so forth.
# If it accepted negative values, "--risky \\-1" should work (double
# backslash to get around the shell).  If we wanted to add a validator
# function to check whether \$risky is binary, we could.  *shrug*
#
# (Note that values are evaluated after the toggle.  So,
#   --disable-risky=1 will actually set $risky to 1.)
#
#
#
# Aside from the table, other values can be set.  These are...
#
#
#  preamble => If 'print_help()' gets invoked, either due to unparsable
#              arguments or by explicitly calling it, this text gets
#              printed after the program name but before the argument
#              list.  May be undefined.
#  epilogue => This text gets printed after the arguments are described.
#        fh => The filehandle where all printing goes to.  Defaults to
#              \*STDOUT.
# validator => If defined, this function gets called with no arguments
#              after all arguments have been parsed.  Like the sanity
#              checkers, 0 => failure.  Useful for tying loose ends and
#              checking related variables.
#    prefix => Defaults to undef.  If defined, gets prepended to every
#              line.
#    margin => Used for line wrapping.  Defaults to 70 columns.
#
#
# So one can invoke 'new ParseArgs(+[ table... ])', and then use
# set_fh(), set_preamble(), set_validator(), et al.  Then, call
# @remaining = parse_arg(@ARGV), where @remaining holds whatever
# follows.
#
##########################################################################





BEGIN {
  use Exporter ();
  use Symbol;
  use vars qw($VERSION @ISA @EXPORT @EXPORT_OK %EXPORT_TAGS);

  $VERSION     = 1.0;
  @ISA         = qw(Exporter);
  @EXPORT =  qw(
    new
    clear_table

    set_preamble
    set_epilogue
    set_validator
    set_fh
    set_prefix
    set_margin

    load_table
    print_help
    parse_args

    print_settings
  );
  %EXPORT_TAGS = ();
  @EXPORT_OK   = ();

  return 1;
}



sub new($@) {
  my $class   =  shift @_;
  my $self    = +{};
  my $lref    = undef;
  bless $self, $class;

  # Let's be lax and take either a list reference, or a list
  # itself.
  if ((scalar(@_) == 1) && ((ref $_[0]) eq "ARRAY")) {
    $lref = $_[0];
  } else {
    $lref = \@_;
  }

  $self->{"indices"} = +{
    opts => 0,
    desc => 1,
    argmap => 2,
    toggle => 3,
    default => 4,
    optional => 5,
    specified => 6
  };
  $self->{"arg_indices"} = +{
    valref => 0,
    valdef => 1,
    valdesc => 2,
    valopt => 3,
    valsan => 4,
    valspec => 5
  };


  # 'fh' is the filehandle output is sent to.
  ($self->{"fh"}) = \*STDOUT;

  $self->{"preamble"}  = undef;
  $self->{"epilogue"}  = undef;
  $self->{"validator"} = undef;
  $self->{"prefix"}    = undef;
  $self->{"margin"}    = 70;

  my $colno = 0;
  $self->{"column"}    = \$colno;

  $self->clear_table();
  if (defined($lref)) {
    $self->load_table($lref);
  }


  return $self;
}




sub clear_table($) {
  my $self = shift;

  $self->{"arg_table"}  = +[];
  $self->{"short_list"} = +{};
  $self->{"long_list"}  = +{};
}



sub set_preamble($$) {
  my $self   = shift;
  my $preamble = shift;

  $self->{"preamble"} = $preamble;
}


sub set_epilogue($$) {
  my $self   = shift;
  my $epilogue = shift;

  $self->{"epilogue"} = $epilogue;
}


sub set_filehandle($$) {
  my $self = shift;
  my $fh   = shift;

  ($self->{"fh"}) = $fh;
}


sub set_validator($$) {
  my $self = shift;
  my $validator = shift;

  ($self->{"validator"}) = $validator;
}


sub set_prefix($$) {
  my $self = shift;
  my $prefix = shift;

  $self->{"prefix"} = $prefix;
}


sub set_margin($$) {
  my $self = shift;
  my $margin = shift;

  $self->{"margin"} = $margin;
}


# The table is the table of arguments and related parameters.
#
# It should contain one list reference per parameter with the
# following fields.
#
#     opts:   list reference containing all command-line ways to specify
#             the argument.  Methods should not include -- or -; the
#             former is used for long arguments, the latter for short.
#             If only one way, it may be a scalar.
#     desc:   scalar containing description text.
#   argmap:   A list reference of list references.  Each list reference
#             corresponds to one associated value, and contains:
#                reference to corresponding program value
#                default value
#                description
#                flag indicating whether or not its optional (0=req'd)
#                sanity check function reference (can be undefined)
#                  1=acceptable, not-1 = not acceptable.
#
#             Optional argmap entries must not precede mandatory
#             entries.  List argmap items must be the LAST one.
#             May be a one-level list ref if there's only one argument.
#   toggle:   Reference to a scalar.  If defined, this value is set to
#             1 if present, and 0 if the option is specified with
#             --disable-.  If not defined, assumed to not be a toggle.
#  default:   If defined, this is the default value of toggle.
# optional:   A bit indicating whether or not this entire parameter
#             is optional.
#

sub load_table($$) {
  my $self = shift;
  my $lref = shift;
  my $itmref = undef;


  foreach $itmref (@{$lref}) {
    my ($opts, $desc, $argmap, $toggle, $default, $optional) =
    @{$itmref};
    my @copy = ();

    if ((ref $opts) eq "ARRAY") {
      push @copy, +[ @{$opts} ];
    } elsif ((ref $opts) eq "SCALAR") {
      push @copy, +[ $opts ];
    } elsif (!(ref $opts)) {
      push @copy, +[ $opts ];
    }

    push @copy, $desc;

    if (defined($argmap)) {
      if ((ref ($argmap->[0])) eq "ARRAY") {
        push @copy, +[ @{$argmap} ];
      } else {
        push @copy, +[ +[ @{$argmap} ]];
      }
    } else {
      push @copy, +[];
      $argmap = +[];
    }

    push @copy, $toggle;
    push @copy, $default;
    push @copy, $optional;
    push @copy, 0;   # Specified or not.

    push @{$self->{"arg_table"}}, \@copy;

    if (defined($toggle)) {
      $$toggle = $default;
    }

    if ((!defined($toggle)) || ($toggle)) {
      # Set defaults for options, as well, if the main argument is not
      # disabled by default.
      my $argitm = undef;

      foreach $argitm (@{$copy[$self->{'indices'}->{'argmap'}]}) {
        my $valdef = $argitm->[$self->{'arg_indices'}->{'valdef'}];
        my $isdef  = defined($valdef);
        defined($argitm) || die;


        if ((ref ($argitm->[$self->{'arg_indices'}->{'valref'}])) eq 'SCALAR') {
          ${$argitm->[$self->{'arg_indices'}->{'valref'}]} =
          $isdef ? $valdef : undef;
        } else {
          @{$argitm->[$self->{'arg_indices'}->{'valref'}]} =
          $isdef ? @$valdef : ();
        }

        $argitm->[$self->{'arg_indices'}->{'valspec'}] = 0;
      }
    }

    my $specifier = undef;
    foreach $specifier (@{$copy[$self->{'indices'}->{'opts'}]}) {
      $self->{"hash_table"}->{$specifier} = \@copy;
    }
  }
}





# Internal function.  Takes a file handle, piece of text, column
# number (ref), column limit, first-line indent, and second+-line
# indent.

sub _print_wrap($$$$) {
  my $self    = shift;
  my $text    = shift;
  my $cref    = $self->{"column"};
  my $clim    = $self->{"margin"};
  my $indent0 = shift;
  my $indent1 = shift;

  my @words = split /\s+/, $text;
  my $word  = undef;
  my $line  = 0;
  my $fh    = $self->{'fh'};
  my $prefix = $self->{'prefix'};


  while (defined($word = shift @words)) {
    if ($word eq '\\n') {
      print $fh "\n";

      $$cref = 0;
      if (defined($prefix)) {
        print $fh $prefix;
        $$cref += length $prefix;
      }

      next;
    }

    if ($word =~ /^(.*)\\n(.*)$/) {
      if (defined($2)) {
        unshift @words, $2;
      }
      unshift @words, "\\n";
      if (defined($1)) {
        unshift @words, $1;
      }
      next;
    }

    if (defined($prefix) && ($$cref == 0)) {
      print $fh $prefix;
      $$cref += length $prefix;
    }

    if (($line == 0) && ($$cref < $indent0)) {
      print $fh " " x ($indent0 - $$cref);
      $$cref = $indent0;
    } elsif (($line == 1) && ($$cref < $indent1)) {
      print $fh " " x ($indent1 - $$cref);
      $$cref = $indent1;
    }

    my $wordlen = length($word);

    if ((($line == 0) && ($$cref > $indent0)) ||
    (($line == 1) && ($$cref > $indent1))) {
      # Need to add a space for separation.
      if (($$cref + $wordlen + 1) <= $clim) {
        print $fh " $word";
        $$cref += $wordlen + 1;

        if (($word =~ /\.$/) && ($$cref < $clim)) {
          print $fh " ";
          $$cref++;
        }
        next;
      }
    } else {
      # No space.
      if (($$cref + $wordlen) <= $clim) {
        print $fh "$word";
        $$cref += $wordlen;
        if (($word =~ /\.$/) && ($$cref < $clim)) {
          print $fh " ";
          $$cref++;
        }
        next;
      }
    }

    $line = 1;
    # Did not fit; need to wrap.
    print $fh "\n";
    $$cref = 0;
    redo;
  }

  return;
}




# $self, preamble, epilogue, plus file handle, and line-wrap limit.
sub print_help($) {
  my $self     = shift;
  my $name     = basename($PROGRAM_NAME);
  my $fh       = $self->{"fh"};
  my $has_toggles = 0;

  if (!defined($fh)) {
    $fh = \*STDOUT;
  }

  $self->_print_wrap( "$name -- \\n", 0, 0);

  if (!defined($self->{"arg_table"})) {
    $self->print_wrap( "Argument table not loaded.\\n", 0, 0);
    return;
  }

  my $preamble = $self->{"preamble"};
  my $epilogue = $self->{"epilogue"};

  if (defined($preamble)) {
    $self->_print_wrap( $preamble, 0, 0);
    $self->_print_wrap( "\\n", 0, 0);
  } else {
    $self->_print_wrap( "\\n \\n", 0, 0);
  }

  my $optref = undef;
  my $colnum = 15;  # Arbitrary

  foreach $optref (@{$self->{"arg_table"}}) {
    my @variations = @{$optref->[$self->{'indices'}->{'opts'}]};
    my $primary    = shift @variations;
    my $desc       = $optref->[$self->{'indices'}->{'desc'}];
    my $columns    = 0;

    $self->_print_wrap( "\\n", 0, 0);

    if (length($primary) == 1) {
      $primary = "-$primary";  # Short option
    } else {
      $primary = "--$primary"; # Long option
    }

    my $text = "";

    if ((length $primary) < ($colnum-3)) {
      $self->_print_wrap( " " x ($colnum - (length $primary) - 3),
      0, 0);
      $columns = $colnum;
    }

    $text = $primary . ":  [" . $desc . "] ";

    if ($optref->[$self->{'indices'}->{'optional'}] == 0) {
      $text = $text . "Must be specified. ";
    }

    # A toggle?
    if (defined($optref->[$self->{'indices'}->{'toggle'}])) {
      $text = $text .  "A binary option, defaulting to ";
      $has_toggles = 1;
      if ($optref->[$self->{'indices'}->{'default'}] == 0) {
        $text = $text . "false.";
      } else {
        $text = $text . "true.";
      }
    }

    $self->_print_wrap( $text, 0, $colnum);

    # Does it take arguments?
    if (scalar(@{$optref->[$self->{'indices'}->{'argmap'}]})) {
      my $argitem = undef;

      $self->_print_wrap( " Takes arguments.", 0, $colnum);
      $self->_print_wrap( "\\n", 0, 0);

      foreach $argitem (@{$optref->[$self->{'indices'}->{'argmap'}]}) {
        my ($valref, $valdef, $valdesc, $valopt, $valsan) = @{$argitem};

        my $text = "<$valdesc> ";

        if ($valopt == 0) {
          $text = $text . "Must be specified if parent argument is. ";
        }

        if ((ref $valref) eq 'ARRAY') {
          if ((!defined($valdef)) || (!scalar(@{$valdef}))) {
            $text = $text . "Defaults to empty list. ";
          } else {
            $text = $text . "Defaults to [";
            $text = $text . join(", ", @{$valdef}) . "]. ";
          }
        } else {
          if (!defined($valdef)) {
            $text = $text . "Defaults to undefined. ";
          } else {
            $text = $text . "Defaults to '$valdef'. ";
          }
        }

        if (defined($argitem->[$self->{'arg_indices'}->{'valsan'}])) {
          $text = $text . "Sanity checking is enabled.";
        }
        $self->_print_wrap( $text, $colnum, $colnum);
        $self->_print_wrap( "\\n", $colnum, $colnum);
      }
    } else {
      $self->_print_wrap( "No arguments.", $colnum, $colnum);
    }

    $self->_print_wrap( "\\n", 0, 0);

    if (scalar(@variations)) {
      my $text = "Also can be specified as: ";
      my $var  = undef;
      my $ct   = 0;

      foreach $var (@variations) {
        if ($ct > 0) {
          $text = $text . ", ";
        }
        $ct++;
        if (length($var) == 1) {
          $text = $text . "-$var";
        } else {
          $text = $text . "--$var";
        }
      }

      $text = $text . ".";

      $self->_print_wrap( $text, $colnum, $colnum);
      $self->_print_wrap( "\\n", $colnum, $colnum);
    }
  }

  if ($has_toggles == 1) {
    my $text = "Binary attributes can be explicitly disabled via the " .
    "--disable-varname form.  --enable-varname is superfluous, " .
    "since that is the default behavior of --varname, but is " .
    "still allowed.";
    $self->_print_wrap( $text, 0, 0);
    $self->_print_wrap( "\\n", $colnum, $colnum);
  }

  if (defined($epilogue)) {
    $self->_print_wrap( $epilogue, 0, 0);
  }

  $self->_print_wrap( "\\n \\n", $colnum, $colnum);
}








# Process an individual argument.
sub _process_argument($$$$) {
  my $self     = shift;
  my $argument = shift;
  my $arglref  = shift;
  my $type     = shift;  # Might be undef, 'enable' or 'disable'.
  my $itmref   = $self->{"hash_table"}->{$argument};
  my $fh       = $self->{"fh"};


  $itmref->[$self->{'indices'}->{'specified'}] = 1;  # 'specified'

  if ((defined($type)) && ($type eq 'disable')) {
    ${$itmref->[$self->{'indices'}->{'toggle'}]} = 0;
  } elsif (defined($itmref->[$self->{'indices'}->{'toggle'}])) {
    ${$itmref->[$self->{'indices'}->{'toggle'}]} = 1;
  }

  # Iterate through the argmap.
  my $argmap = $itmref->[$self->{'indices'}->{'argmap'}];
  my $argitm = undef;

  foreach $argitm (@{$argmap}) {
    my $sanity   = $argitm->[$self->{'arg_indices'}->{'valsan'}];  # Verifier

    if ((!scalar(@$arglref)) || ($arglref->[0] =~ /^\-/o)) {
      # End of the road.
      # Is this argitem mandatory?  If so, fail.
      if ($argitm->[$self->{'arg_indices'}->{'valopt'}] == 0) {
        # Well, there's one major exception.  If it's mandatory,
        # and the next item is of the form -\d\d\d\d\d (in other
        # words, is quite possibly a number) we'll accept it.
        # Some allowances are made for floating point numbers and
        # scientific notation.  Beyond that, the user can force
        # acceptance by using a backslash (\-0xABBEF, say).

        if (($arglref->[0] =~ /^(\-\d*\.?\d*[Ee]?\d*)$/o) &&
        ($arglref->[0] =~ /\d/o)) {
          goto not_argument;
        }

        my $colno = 0;

        $self->_print_wrap("Missing mandatory option for argument.\\n",
        $colno, 0, 0);
        $self->print_help();
        exit(-1);
      }

      # Otherwise, return.  Default values were already set during the
      # load_table subroutine.
      return;
    }

    not_argument:
    # Is it a list that we're looking for, or a scalar?
    if ((ref ($argitm->[$self->{'arg_indices'}->{'valref'}])) eq 'ARRAY') {
      # List. Snarf all values up to the next option, and return.
      my @vallist = ();

      while ((scalar(@$arglref)) && (!($arglref->[0] =~ /^\-/o))) {
        my $val = shift @$arglref;

        # Treat backslash as an escape.  Supported so that a person
        # can specify \--foobar, for instance, and it won't be
        # confused as meaning --foobar as a command-line argument.
        $val =~ s/\\(.)/$1/g;

        # Special value.
        if (($val =~ /^[Nn][Uu][Ll][Ll]$/o) ||
        ($val =~ /^[Uu][Nn][Dd][Ee][Ff]$/o)) {
          $val = undef;
        }
        $argitm->[$self->{'arg_indices'}->{'valspec'}] = 1;
        push @vallist, $val;
      }

      @{$argitm->[$self->{'arg_indices'}->{'valref'}]} = @vallist;


      if (defined($sanity)) {
        if (!(&$sanity(+[ @vallist ]))) {
          print $fh "Value fails sanity check.\n";
          $self->print_help();
          exit(-1);
        }
      }
      return;
    } else {
      # Scalar.
      my $val = shift @$arglref;

      # Special value.
      if (($val =~ /^[Nn][Uu][Ll][Ll]$/o) ||
      ($val =~ /^[Uu][Nn][Dd][Ee][Ff]$/o)) {
        $val = undef;
      }

      # Handle backslashes.
      $val =~ s/\\(.)/$1/g;

      if (defined($sanity)) {
        if (!(&$sanity($val))) {
          print $fh "Value fails sanity check.\n";
          $self->print_help();
          exit(-1);
        }
      }

      $argitm->[$self->{'arg_indices'}->{'valspec'}] = 1;
      ${$argitm->[$self->{'arg_indices'}->{'valref'}]} = $val;
    }
  }
}




# Give it the @ARGV list.  Returns the list of remaining
# arguments, if any.

sub parse_args($@) {
  my $self        = shift;
  my @arglist     = @_;
  my $table_ref   = $self->{"arg_table"};
  my $fh          = $self->{"fh"};
  my $columns     = 0;


  defined($table_ref) || die;

  # Reset the 'specified' flags to 0.
  map { $_->[$self->{'indices'}->{'specified'}] = 0 } @{$table_ref};

  my $param = undef;
  my $idx   = 0;

  processing:  while (defined($param = shift @arglist)) {
    if ($param eq "--") {
      last processing;
    }

    if (!($param =~ /^\-/o)) {
      push @arglist, $param;
      last processing;
    }

    if ($param =~ /^(\-\-?[^=]+)=([^=]+)/o) {
      # Allow the --foo=bar form.

      unshift @arglist, $2;
      unshift @arglist, $1;
      next;
    }

    # Short argument.
    if ($param =~ /^\-[^\-]/o) {
      if (length($param) > 2) {
        # -ab is an illegal short argument; if it were a single arg
        # it should be --ab.  Let's expand it (convert to -a -b).

        my @vals = split //, $param;
        shift @vals;   # Get rid of the leading value.
        map { unshift @arglist, "-$_" } (reverse @vals);
        next processing;
      } else {
        # Short argument?
        my $character = substr $param,1,1;
        if (exists($self->{"hash_table"}->{$character})) {
          # We may need to modify the rest of the list.
          $self->_process_argument($character, \@arglist, undef);
          next;
        } else {
          # Unknown short argument.
          $self->_print_wrap( "\\nUnknown short argument:  -$character.\\n", 0, 0);
          $self->print_help();
          exit(-1);
        }
      }
    } else {
      # Long argument.
      my $long_arg = substr $param,2;

      # Should already have eliminated --.
      defined($long_arg) || die "Programming error.";

      if (exists($self->{"hash_table"}->{$long_arg})) {
        $self->_process_argument($long_arg, \@arglist, undef);
        next;
      }

      # Check for an --enable or --disable form pointing to a
      # corresponding toggle.
      my $type = undef;

      if ($long_arg =~ /^enable\-(.+)$/o) {
        $type = "enable";
        $long_arg = $1;
      } elsif ($long_arg =~ /^disable\-(.+)$/o) {
        $type = "disable";
        $long_arg = $1;
      }

      if (defined($type)) {
        if (exists($self->{"hash_table"}->{$long_arg})) {
          if (defined($self->{"hash_table"}->{$long_arg}->[$self->{'indices'}->{'toggle'}])) {
            $self->_process_argument($long_arg, \@arglist, $type);
            next;
          }
        }
      }
    }

    # Should have hit 'next' by now if argument was successfully
    # identified and processed.
    $self->_print_wrap( "\\nFailed to identify or process argument.\\n", 0, 0);
    $self->print_help();
    exit(-1);
  }

  # Need to make sure that top-level mandatory parameters were specified.
  {
    my $optref = undef;

    foreach $optref (@{$self->{'arg_table'}}) {
      if ((!($optref->[$self->{'indices'}->{'optional'}])) && # mandatory
      (!($optref->[$self->{'indices'}->{'specified'}])))  # unspecified
      {
        $self->_print_wrap( "\\nMissing mandatory argument.\\n", 0, 0);
        $self->print_help();
        exit(-1);
      }
    }
  }

  if (defined($self->{"validator"})) {
    my $validator = $self->{"validator"};

    if (!(&$validator())) {
      $self->_print_wrap( "\\nValidator returns an error.\\n", 0, 0);
      $self->print_help();
      exit(-1);
    }
  }
  return @arglist;
}




# Print the current value for each parameter and its options.
# $self
sub print_settings($) {
  my $self     = shift;
  my $name     = basename($PROGRAM_NAME);
  my $fh       = $self->{"fh"};
  my $has_toggles = 0;

  if (!defined($fh)) {
    $fh = \*STDOUT;
  }

  $self->_print_wrap( "\\nParameter settings:\\n", 0, 0);

  if (!defined($self->{"arg_table"})) {
    $self->print_wrap( "\\nArgument table not loaded.", 0, 0);
    return;
  }

  my $optref = undef;
  my $colnum = 11;  # Arbitrary

  foreach $optref (@{$self->{"arg_table"}}) {
    my @variations = @{$optref->[$self->{'indices'}->{'opts'}]};
    my $primary    = shift @variations;
    my $desc       = $optref->[$self->{'indices'}->{'desc'}];

    if (length($primary) == 1) {
      $primary = "-$primary";  # Short option
    } else {
      $primary = "--$primary"; # Long option
    }

    my $text = "";

    $self->_print_wrap( "\\n", 0, 0);
    if ((length $primary) < ($colnum-3)) {
      $self->_print_wrap( " " x ($colnum - (length $primary) - 3),
      0, 0);
    }

    $text = $primary . ":  [" . $desc . "] ";

    # A toggle?
    if (defined($optref->[$self->{'indices'}->{'toggle'}])) {
      $text = $text . "A binary option, ";
      $has_toggles = 1;

      if (($optref->[$self->{'indices'}->{'specified'}])) {
        $text = $text . "specified to be ";
        $text = $text . (($optref->[$self->{'indices'}->{'toggle'}]) ?
        "true." : " false.");
      } else {
        $text = $text . "unspecified and therefore defaulting to ";

        if ($optref->[$self->{'indices'}->{'default'}] == 0) {
          $text = $text . "false.";
        } else {
          $text = $text . "true.";
        }
      }
    }

    $self->_print_wrap( $text, 0, $colnum);

    # Does it take arguments?
    if (scalar(@{$optref->[$self->{'indices'}->{'argmap'}]})) {
      my $argitm = undef;

      $self->_print_wrap( "\\n", 0, 0);

      foreach $argitm (@{$optref->[$self->{'indices'}->{'argmap'}]}) {
        my ($valref, $valdef, $valdesc, $valopt, $valsan) = @{$argitm};
        my $text = "<$valdesc> ";

        if ((ref $valref) eq "ARRAY") {
          if (!($argitm->[$self->{'arg_indices'}->{'valspec'}])) {
            if ((!defined($valdef)) || (!scalar(@{$valdef}))) {
              $text = $text . "Unspecified, so using default of empty list. ";
            } else {
              $text = $text . "Unspecified, so using default of [";
              $text = $text . join(", ", @{$valdef}) . "]. ";
            }
          } else {
            if ((!defined($valref)) || (!scalar(@{$valref}))) {
              $text = $text . "Specified to be empty list. ";
            } else {
              $text = $text . "Specified to be [";
              $text = $text . join(", ", @{$valref}) . "]. ";
            }
          }
        } else {
          if (!($argitm->[$self->{'arg_indices'}->{'valspec'}])) {
            if (!defined($valdef)) {
              $text = $text . "Unspecified, so using the default of undefined. ";
            } else {
              $text = $text . "Unspecified, so using default of  '$valdef'. ";
            }
          } else {
            if (!defined($valref)) {
              $text = $text . "Specified as undefined. ";
            } else {
              $text = $text . "Specified to be '$$valref'. ";
            }
          }
        }

        $self->_print_wrap( $text, $colnum, $colnum);
      }
    }

    $self->_print_wrap( "\\n", 0, 0);
  }

  $self->_print_wrap( "\\n", $colnum, $colnum);
}



return 1;
