: # Use perl...
eval 'exec perl5 -w -S $0 "$@"'
if 0;

#!/usr/bin/perl5 -w
#
# Create a PNG graph of the pair-count data, with the
# best-fit line superimposed.
#
# [ NOTE:  PNG now is only the default... it doesn't
#   have to be.  Substitute whatever Gnuplot terminal
#   type you want via -f.
#
#   argv() support has been added; if named 'to_foo.pl',
#   the default terminal type becomes 'foo'.
#
#   Another new addition is precision specification on
#   the axes, and for the line parameters now included
#   with the title.  The user can specify additional
#   text to add to the title as well.
# ]
#
# All input is derived from STDIN.
#
# The first three lines must each contain one number --
# the slope, the y-intercept, and the correlation.
#
# (Note -- the correlation isn't integral to the plot.
#  It's added because a) it might be good to show on the
#  plot, and b) because it's outputted by the robust
#  linefitter.)
#
# Every subsequent line must contain exactly two
# numbers -- an x coordinate and a y coordinate.
# Both are expected to have already been subjected
# to log_2.
#
# This script doesn't care whether or not the pairs
# have already been trimmed.  They should have been
# for the robust fitter, but one can plot the raw
# pair-count log/log data if desired.
#
# The output can be redirected to a file or standard
# output, or both, as desired.
#
#######################################################
# $Id: to_png.pl,v 1.5 2002/01/03 10:54:12 lw2j Exp $
# $Log:	to_png.pl,v $
# Revision 1.5  2002/01/03  10:54:12  lw2j
# Untabified.
#
# Revision 1.4  2001/08/22  17:21:29  lw2j
# Untabified (tab=8).
#
# Revision 1.3  2001/07/09  11:04:26  lw2j
# Multiple changes.
#    - Now argv() sensitive if $0 ~ to_blah.pl
#    - Adds "y=ax+b" specification to title
#    - User can add additional text to title
#    - User can specify precision for x- and y- axes independently,
#      as well as for the line parameters and correlation
#      coefficient.
#
#######################################################

use strict;

##### <change_me>
#
# Change this to where you placed the modules.
#
use lib '/afs/cs.cmu.edu/user/lw2j/shared/FracDim/FracDim';
##### </change_me>

use Symbol;
use FileHandle;
use IPC::Open3;
use File::Basename;

require mktemp;


{
    my ($param_ref, $fname) = parse_arguments();

    my $a = <STDIN>;
    my $b = <STDIN>;
    my $r = <STDIN>;

    defined($r) || die;
    chomp($a);
    chomp($b);
    chomp($r);

    $a =~ s/^\s+//o;
    $b =~ s/^\s+//o;
    $r =~ s/^\s+//o;


# We use a temporary file because Gnuplot doesn't take data
# from STDIN as far as I can tell.

    my ($pair_fh, $pair_fn) = mktemp::mktemp();

# It's log data, but we'll use gnuplot's scale to make it
# clear, so we undo the logs.
#

    read_data:  {
        my $line = <STDIN>;

        defined($line) || last read_data;
        chomp($line);
        defined($line) || last read_data;

        $line =~ s/^\s+//o;

        my ($log_radius, $log_count) = split /\s+/, $line;

        defined($log_count) || last read_data;

        print $pair_fh (2 ** ($log_radius)), " ", (2 ** ($log_count)), "\n";

        redo read_data;
    }

    close($pair_fh) || die "Failed to write pair-count tempfile:  $!";

    my $gnuplot_fh_i = FileHandle->new;   # Input
    my $gnuplot_fh_o = FileHandle->new;   # Output
    my $gnuplot_fh_e = FileHandle->new;   # Error

    autoflush $gnuplot_fh_i 1;
    autoflush $gnuplot_fh_o 1;
    autoflush $gnuplot_fh_e 1;


    my $gnuplot_fn = 'gnuplot';

    if (exists($param_ref->{'g'})) {
        $gnuplot_fn = $param_ref->{'g'};
    }

    # Error gets redirected to $gnuplot_fh_o.
    if (!open3($gnuplot_fh_i, $gnuplot_fh_o, $gnuplot_fh_e,
               $gnuplot_fn)) {
        die "Fatal error opening(3) '$gnuplot_fn':  $!";
    }

    # Delete file if it exists.
    if (defined($fname)) {
        unlink($fname);
    }

    my $format_type = 'png';

    {
        my $progname = basename($0);

            if ($progname =~ /^to_([^.]+)\.pl$/) {
                $format_type = $1;
            }
    }

    if (exists($param_ref->{'f'})) {
        $format_type = $param_ref->{'f'};
    }

    print $gnuplot_fh_i "set terminal $format_type\n";


    if (defined($fname)) {
        print $gnuplot_fh_i "set output \"$fname\"\n";
    } else {
        print $gnuplot_fh_i "set output\n";
    }

    my $pf_formatter = "%.4f";

    if (exists($param_ref->{'p'})) {
        $pf_formatter = $param_ref->{'p'};
    }

    # Here, _t stands for truncated (since they probably are).
    my $a_t = sprintf($pf_formatter, $a);
    my $b_t = sprintf($pf_formatter, $b);
    my $r_t = sprintf($pf_formatter, $r);

    my $title = "Pair-Count Plot and Best-Fit Line\\ny=${a_t}*x+${b_t} (Corr = ${r_t})";

    if (exists($param_ref->{'n'})) {
        $title = $title . "\\n" . ($param_ref->{'n'});
    }

    print $gnuplot_fh_i "set title \"$title\"\n";
    print $gnuplot_fh_i "set xlabel \"Radius (logarithmic)\"\n";
    print $gnuplot_fh_i "set ylabel \"Pairs (logarithmic)\"\n";

    # Figure out and apply axis formatting strings.
    my $x_format = "%.3g";
    my $y_format = "%.4g";

    if (exists($param_ref->{'x'})) { $x_format = $param_ref->{'x'}; };
    if (exists($param_ref->{'y'})) { $y_format = $param_ref->{'y'}; };

    print $gnuplot_fh_i "set format x \"$x_format\"\n";
    print $gnuplot_fh_i "set format y \"$y_format\"\n";

    print $gnuplot_fh_i "set logscale xy 2\n";
    print $gnuplot_fh_i "set key under\n";
    print $gnuplot_fh_i "plot \"$pair_fn\" title \"Pair-Count Data\" with linespoints, (2**(($a*log(x)/log(2))+$b)) title \"Linear approximation\"\n";


    # Both $fname *and* STDOUT.
    if (defined($fname) && exists($param_ref->{'w'})) {
        print $gnuplot_fh_i "set output\n";
        print $gnuplot_fh_i "replot\n";
    }

    close($gnuplot_fh_i) || die "Unexpected error closing stdin to '$gnuplot_fn':  $!";

    my @gnuplot_output = <$gnuplot_fh_o>;
    my @gnuplot_error  = <$gnuplot_fh_e>;

    close($gnuplot_fh_o) || die "Unexpected error closing stdout from '$gnuplot_fn':  $!";
    close($gnuplot_fh_e) || die "Unexpected error closing stderr from '$gnuplot_fn':  $!";



    # Check for certain errors.
    if (grep(/invalid command/, @gnuplot_error)) {
        die "'$gnuplot_fn' reports an invalid command.\nThis error condition should not happen.\n";
    }

    if (grep(/unknown or ambiguous terminal type/, @gnuplot_error)) {
        die "'$gnuplot_fn' does not understand terminal type '$format_type'.\n";
    }

    if (grep(/^\s*^\s*$/, @gnuplot_error)) {
        die "'$gnuplot_fn' is indicating an unexpected error type.\n";
    }

    # If it t'was sent to STDOUT, we need to pipe that -- and only that.
    #
    # We note that Gnuplot appears to send its interactive foo to
    # standard *error*.  Hence, we should be able to output just what's
    # on $gnuplot_fh_o.
    if ((!defined($fname)) || (exists($param_ref->{'w'}))) {
        print @gnuplot_output;
    }
}


exit 0;



#####################
sub print_usage_and_exit() {
    use File::Basename;

    my $progname = basename($0);

    print "$progname -- Take line-fitting data and pair-count data\n";
    print "             and use Gnuplot to generate a graph.\n";
    print "\n";
    print "Usage:\n";
    print "\n";
    print "$progname [-g gnuplot_path]  [-f format]      [-n name]\n";
    print "          [-p printf_format] [[-w] filename]\n";
    print "\n";
    print " -g          If Gnuplot is not in your path, you can use\n";
    print "             this option to indicate where -- for \n";
    print "             instance \"-g ~/bin/gnuplot\".\n";
    print " -f          Use to specify a different image format.\n";
    print "             The default is \"png\".  The argument is\n";
    print "             passed as-is to Gnuplot.\n";
    print " -n          Use this to specify a string to add to the title,\n";
    print "             such as a reminder about the source of the data.\n";
    print "             This string will be on a separate line.\n";
    print " -p          Specifies printf()-style formatting string for\n";
    print "             slope, y-intercept, and correlation.  Default is\n";
    print "             \"%.4f\n";
    print " -x          Specifies printf()-style formatting string for\n";
    print "             X axis (log radius).  Defaults to \"%.3g\".\n";
    print " -y          Specifies printf()-style formatting string for\n";
    print "             Y axis (log pairs).   Defaults to \"%.4g\".\n";
    print " -w          Write to both 'filename' *and* standard\n";
    print "             output.  Ignored if no filename is given.\n";
    print "\n";
    print "The filename is the name of the output file to be written\n";
    print "by Gnuplot.  If no filename is specified, its output will\n";
    print "be sent directly to standard output.\n";
    exit 0;
}



# Returns a reference to a hash table of parameters,
# and a scalar containing the filename.
#
# Always uses @ARGV.
#
sub parse_arguments() {
    my $arg;
    my $fname  = undef;
    my %params = ();

    while (defined($arg = $ARGV[0]) && ($arg =~ /^-/)) {
        $arg = shift @ARGV;

        if ($arg eq '-w') {
            $params{'w'}  = 1;
        } elsif (($arg eq '-g') ||
                 ($arg eq '-f') ||
                 ($arg eq '-n') ||
                 ($arg eq '-p') ||
                 ($arg eq '-x') ||
                 ($arg eq '-y')) {
            $arg =~ s/^-//;
            my $val = shift @ARGV;

            defined($val) || die "Failed to supply value for -$arg";

            if ($val ne 'undef') {
                $params{$arg} = $val;
            } else {
                $params{$arg} = undef;
            }
        } elsif ($arg eq '--') {
            last;
        } else {
            print_usage_and_exit();
        }
    }

    $fname   = shift @ARGV;

    return (\%params, $fname);
}
