#!/usr/local/bin/perl5 -w
#
# Purpose:
#
#   Provide a module for, given two references to arrays of
#   x and y points, tries to eliminate the mostly flat
#   regions at either extreme.  It's very crude, but fairly
#   fast.
#
#   It returns two references to NEW arrays containing the
#   hopefully-trimmed points.
#
#   There's a basic assumption that there ARE flat regions at
#   either end, because it'll trim the two extreme points
#   unless there aren't very many to begin with.
#
##########################################################################
# $Id: Trim_Flat.pm,v 1.5 2002/01/03 10:54:12 lw2j Exp $
# $Log:	Trim_Flat.pm,v $
# Revision 1.5  2002/01/03  10:54:12  lw2j
# Untabified.
# 
# Revision 1.4  2001/08/22  22:45:23  lw2j
# Fixed empty return in Trim_Flat.pm (with 4 or fewer points, it didn't
# return references).
#
# Revision 1.3  2001/08/22  17:21:29  lw2j
# Untabified (tab=8).
#
# Revision 1.2  2001/08/13  16:42:37  lw2j
# Fixed nonsense code in the "Five points is dangerously low" part.
#   -- should have used shift instead of unshift
#   -- should have used $new_ct, consistently
# This should remove some more fd=0 bugs.
#
# Revision 1.1  2001/05/29  18:04:21  lw2j
# Initial revision
#
#
##########################################################################
#
# Package Header
#
##########################################################################


package Trim_Flat;
require Exporter;

use strict;

##########################################################################

BEGIN {
    use Exporter ();
    use Symbol;
    use vars qw($VERSION @ISA @EXPORT @EXPORT_OK %EXPORT_TAGS);

    $VERSION     = 1.0;
    @ISA         = qw(Exporter);
    @EXPORT =  qw(
                  trim_flat
                  );
    %EXPORT_TAGS = ();
    @EXPORT_OK   = ();

    return 1;
}



# Given references to two parallel arrays (x and y
# coordinates), figure out 'trimmed' versions that
# hopefully do not have flat regions at either end.
#
# Returns references to NEW arrays.

sub trim_flat($$) {
    my $old_x_ref = shift;
    my $old_y_ref = shift;
    my @old_x     = @{$old_x_ref};
    my @old_y     = @{$old_y_ref};
    my $diff  = 0.01;

    my $min   = $old_y[0];
    my $max   = $old_y[0];
    my $ct    = scalar @old_x;

    defined($min) || die;

    ((scalar(@old_y)) == $ct) || die;

    my $i = 0;

    if ($ct < 5) {
        # don't bother...
        return ($old_x_ref, $old_y_ref);
    }

    for ($i=1; $i < $ct; $i++) {
        my $y_val = $old_y[$i];
        $min = ($min > $y_val) ? $y_val : $min;
        $max = ($max < $y_val) ? $y_val : $max;
    }

    # Fudge factor/tolerance.
    $min = $min+$diff;
    $max = $max-$diff;

    my @new_x = ();
    my @new_y = ();


    # How many points would qualify?
    my $new_ct   = grep { ($_ >= $min) && ($_ <= $max) } @old_y;

    if ($new_ct > 5) {
        for ($i=0; $i < $ct; $i++) {
            if ($old_y[$i] < $min) {
                next;
            }

            if ($old_y[$i] > $max) {
                next;
            }

            push @new_x, $old_x[$i];
            push @new_y, $old_y[$i];
        }
    } else {
        # Five points is dangerously low.  Let's remove
        # ONLY perfectly flat regions at either end.

        my $new_ct = $ct;

        @new_x = @old_x;
        @new_y = @old_y;

        {
            while (($new_ct > 2) && ($new_y[0] == $new_y[1])) {
                shift @new_x;
                shift @new_y;
                $new_ct--;
            }

            while (($new_ct > 2) && ($new_y[$new_ct-1] == $new_y[$new_ct-2])) {
                pop @new_x;
                pop @new_y;
                $new_ct--;
            }
        }
    }

    return (\@new_x, \@new_y);
}
