: # Find Perl
eval 'exec perl5 -w -S $0 "$@"'
if 0;

# This is a simple demo script, and adds zero bits of
# information beyond the README and other files.  Since
# it serves no purpose but to give people a way to test
# in one command, even w/o reading anything else, this
# script has a few additional dependencies.
#
#
# - This file may be rife with Unixisms.
# - Gnuplot should be in the user's path.
# - Ghostview is required for display

# This is more of a generic post-install script than
# anything you'd want to take apart and include in your
# own code.


# Just making the include explicit.
use lib '.";

use strict;
use English;
use Symbol;
use FileHandle;

use     IPC::Open2;
require Benchmark;
require DiskFracDim;
require Wrapper;
use     mktemp;

# Certain phases (namely, the FracDim computation itself)
# may take a considerable amount of time.  Don't buffer
# STDOUT, because otherwise the delay may be considerable.
#
autoflush STDOUT 1;


# Phase 0 -- Removing obsolete files.
#
# Users of previous versions installing to the same
# directory may have obsolete files that have been
# completely eliminated rather than overwritten.
#
# An example is DiskFracTest.pl, which has been
# replaced by fd.pl.  fd.pl produces syntactically
# different output and also adds the slope, which
# isn't necessarily the fractal dimension (it
# depends on $q$).  These changes would break scripts
# parsing DiskFracTest.pl's output so there's little
# reason to not make a clean break.

{
    my @obsolete_known = qw( DiskFracTest.pl );
    my @obsolete_found = ();

    my $fname = ();

    foreach $fname (@obsolete_known) {
        if (-e $fname) {
            push @obsolete_found, $fname;
        }
    }

    if (scalar(@obsolete_found)) {
        print "The following file(s) from previous versions\n";
        print "are now deprecated:\n";
        print "\n";

        foreach $fname (@obsolete_found) {
            print "\t$fname\n";
        }

        print "For each, you may \n";
        print "(D)elete\n";
        print "(I)gnore\n";
        print "(M)ove to a directory called OLD\n";
        print "\n";
        print "Typing an STAR -- as in D* -- will cause that\n";
        print "choice to apply to ALL subsequent deprecated\n";
        print "files.\n";

        my $choice_star = undef;

        {
            foreach $fname (@obsolete_found) {
                my $choice_made = undef;

                if (defined($choice_star)) {
                    print "'$fname' -- [$choice_star]\n";
                    $choice_made = $choice_star;
                } else {
                    get_choice: {
                        print "'$fname' -- [D]elete [I]gnore [M]ove:  \n";

                        $choice_made = <STDIN>;
                        defined($choice_made) || redo get_choice;
                        chomp($choice_made);
                        $choice_made =~ s/^\s+//;
                        $choice_made =~ tr/a-z/A-Z/;

                        if ($choice_made =~ /^(.)\*$/) {
                            $choice_made = $1;
                            $choice_star = $1;
                        }

                        if (!($choice_made =~ /^[DIM]$/)) {
                            redo get_choice;
                        }
                    }
                }

                if ($choice_made eq 'D') {
                    die unless unlink($fname);
                } elsif ($choice_made eq 'I') {
                    # Ignoring
                } elsif ($choice_made eq 'M') {
                    if (!(-d 'OLD')) {
                        print "Creating directory 'OLD'...\n";
                        die unless mkdir "OLD", 0700;
                    }

                    die unless rename $fname, "OLD/$fname";
                }
            }
        }
    }
}

# Phase 1 -- Permissions.
#
# Since we need to modify a couple o' 'use lib' statements,
# we want to make sure that two of the files are writeable.
# If they aren't, toggle those bits on.
#
# This phase knows nothing of ACLs, just traditional Unix
# fast-filesystem bits.
#
# We also need to set the executable bit on all included .pl files.
{
    # 0100 = user can execute
    # 0200 = user can write
    # 0300 = both
    #
    # We fix ourselves in case we weren't shipped as writeable
    # but a user managed to invoke it via, say, ". RUNME.pl".
    # Just for future convenience.
    #


    print "*** Checking file permissions ***\n";

    if (!(-w ".")) {
        die "Fatal error -- cannot write to current directory.";
    }

    my @fix_files = (
                     +[ "DiskFracDim.pm",   0200 ],
                     +[ "Pair_Count.pm",    0200 ],
                     +[ "FixPlace.pl",      0100 ],
                     +[ "LineGen.pl",       0100 ],
                     +[ "Pair_Count.pl",    0300 ],
                     +[ "RUNME.pl",         0100 ],
                     +[ "Robust_LSFit.pl",  0300 ],
                     +[ "TriGen.pl",        0100 ],
                     +[ "Trim_Flat.pl",     0300 ],
                     +[ "fd.pl",            0300 ],
                     +[ "to_eps.pl",        0300 ],
                     +[ "to_png.pl",        0300 ]
                     );

    my $listref = undef;
    foreach $listref (@fix_files) {
        my $file = $listref->[0];
        my $goal = $listref->[1];

        print "Checking permissions on $file ($goal)... ";

        if (!(-e $file)) {
            die "Fatal error -- expected file $file does not exist.";
        }

        # Reuse stat info from the last test
        my ($dev,   $ino,   $mode,  $nlink,
            $uid,   $gid,   $rdev,  $size,
            $atime, $mtime, $ctime, $blksize,
            $blocks) = stat($file);

        if ($uid ne $EUID) {
            die "Fatal error -- '$file' owned by uid $uid.  EUID = $EUID.";
        }

        if (($mode & $goal) == $goal) {
            print "ok\n";
            next;
        } else {
            print "off... ";

            $mode |= $goal;
            if (!(chmod $mode, $file)) {
                die "Fatal error:  chmod() failed on $file:  $!";
            }

            print "(fixed)\n";
        }
    }

}



# Phase 2 -- Fixing library calls.
#
# Now, we invoke the FixPlace script.
#
#
{
    print "*** Running FixPlace.pl ***\n";

    # This really should NOT fail, if we made it past the
    # previous phase...
    if (system("./FixPlace.pl")) {
        die "Error running FixPlace.pl:  $!";
    }
}



# Phase 3 -- Generating test data.
#
# Generating a Sierpinski data set, of 5,000 points
# only (for speed reasons.  Given that this script
# might be the first thing run, it behooves us to
# not use a terribly slow case.
{
    print "*** Generating Sierpinski.5000 ***\n";

    if (system("./TriGen.pl 5000 > Sierpinski.5000")) {
        die "Error running TriGen.pl:  $!";
    }
}


# Phase 4 -- Invoke FracDim code.
#
# We grab the log-log results directly from the structure.
# Outside scripts might simply want to grab stdout and
# parse it (see the format used in 'print_loglog()'),
# or alternately add an explicit method for taking the
# pairs from the DiskFracDim object.


# This will be a temporary file to feed into gnuplot as
# data.
my $pair_fn = undef;
my $fd      = undef;
my $slope   = undef;
my $y_int   = undef;
my $corr    = undef;

{

    # Estimate was 45 seconds before some tweaks
    # implementing an earlier cutoff.

    print "*** Invoking FracDim *** \n";
    print "\n";
    print "WARNING -- This phase may take a considerable\n";
    print "amount of time.  On a 400MHz Pentium II, this\n";
    print "took approximately 40 seconds.\n";
    print "\n";
    print "Due to optimizations that do not apply to the\n";
    print "Sierpinski set, you may find that the speed\n";
    print "varies considerably depending on the data.\n";
    print "\n";

    my $sier_fh = gensym();
    my $wrapper = undef;
    my $fracobj = undef;
    my %params  = ();

    # We need a filehandle to pass to the wrapper.

    open($sier_fh, "Sierpinski.5000") ||
        die "Strange, fatal error opening Sierpinski.5000:  $!";

    $wrapper = new Wrapper($sier_fh);

    $params{'data_o'}   = $wrapper;

    my $t0 = new Benchmark;
    $fracobj = DiskFracDim::new DiskFracDim(%params);
    ($fd, $y_int, $corr, $slope) = $fracobj->fracdim();
    my $t1 = new Benchmark;
    my $td = Benchmark::timestr(Benchmark::timediff($t1, $t0));

    print "Time elapsed:  $td\n";


    print "Fractal dimension reported:  $fd (theoretical ~ 1.58)\n";
    print "(line-fitting) Correlation:  $corr\n";

    close($sier_fh) ||
        die "Unexpected fatal error closing Sierpinski.5000:  $!";


    # Allow for some precision issues due to architectural
    # differences, but .001 should probably be a safe level of
    # accuracy to demand.
    if ((int($fd * 1000)/1000) != 1.627) {
        die "Fatal error -- expected the fd value to be ~1.627.";
    }

    my @log_r = @{$fracobj->{'log_radius_trimmed'}};
    my @log_s = @{$fracobj->{'log_count_trimmed'}};

    # Store this for Gnuplot to use.
    my $pair_fh = undef;
    my $ct = scalar @log_r;
    my $i  = undef;

    ($ct == (scalar @log_s)) ||
        die "Severe, unexpected logical error in DiskFracDim!";
    ($pair_fh, $pair_fn) = mktemp();

    # It's log data, but we'll use gnuplot's scale to make it
    # clear, so we undo the logs.
    for ($i=0; $i < $ct; $i++) {
        print $pair_fh (2 ** ($log_r[$i])), " ", (2 ** ($log_s[$i])), "\n";
    }

    close($pair_fh) || die "Failed to write pair-count tempfile:  $!";
}



# Phase 5 -- Gnuplot.
#
# We use the data obtained from the previous section to plot.
# Note that the coordinates have already been trimmed to remove
# 'flat' regions at either extreme.

{
    my $gnuplot_fh = gensym();

    print "*** Using GNUPLOT for log-log graph ***\n";

    if (!open($gnuplot_fh, "|gnuplot")) {
        die "Fatal error -- gnuplot not in path?  $!";
    }

    unlink("sierpinski.eps");

    print $gnuplot_fh "set terminal postscript eps\n";
    print $gnuplot_fh "set output \"sierpinski.eps\"\n";
    print $gnuplot_fh "set title \"Pair-count plot for Sierpinski.5000\"\n";
    print $gnuplot_fh "set xlabel \"Radius (logarithmic)\"\n";
    print $gnuplot_fh "set ylabel \"Pairs (logarithmic)\"\n";
    print $gnuplot_fh "set logscale xy 2\n";
    print $gnuplot_fh "set key under\n";
    print $gnuplot_fh "plot \"$pair_fn\" title \"Sierpinski\" with linespoints\n";
    print $gnuplot_fh "replot (2**(($fd*log(x)/log(2))+$y_int)) title \"Linear approximation\"\n";
    print $gnuplot_fh "set terminal postscript eps\n";
    print $gnuplot_fh "set output \"sierpinski.eps\"\n";
    print $gnuplot_fh "replot\n";
    close($gnuplot_fh) || die "Unexpected error -- failed to end gnuplot:  $!";
}


# Phase 6 -- Display

{
    print "*** Displaying resulting sierpinski.eps ***\n";

    system("gv sierpinski.eps") && die "Unexpected ghostview return value";
}



# Phase 7 -- Generating test data.
#
# Generating a self-similar line (with gaps).
#
# Warning -- it's 50-dimensional and thus this takes ~4MB space or so.
#
{
    print "*** Generating Line.5000 ***\n";

    if (system("./LineGen.pl 5000 > Line.5000")) {
        die "Error running LineGen.pl:  $!";
    }
}



# Phase 8 -- Invoke FracDim code with q=1 (special case).
#
# Very similar to Phase 4, but it tests code that only applies
# to q=1, and it uses a different data set...


# We'll save these two arrays to redirect to "to_eps.pl".
my @rf_buf = ();
my @trim_buf = ();

{

    print "*** Invoking FracDim for Q=1 *** \n";
    print "\n";

    print "WARNING -- This phase may also take a considerable\n";
    print "amount of time -- perhaps 290 seconds on a 400MHz\n";
    print "Pentium II computer.\n";
    print "\n";
    print "This test invokes code that only applies for\n";
    print "the special case of the exponent being 1.\n";
    print "\n";
    print "This test bypasses the DiskFracDim.pm interface\n";
    print "and instead uses scripts more directly.\n";
    print "\n";


    # Opening and sending output to memory, rather than
    # using temporary files.  My mktemp() routine verifies
    # by opening the file, but the scripts send to STDOUT
    # rather than to a file handle.  So if an OS complained
    # about more than one user writing to the handle, I'd
    # have to buffer the data in memory, close the file
    # handle (thus allowing another user to open to that
    # same name), re-open it, and write the data.  So I'd
    # end up buffering the data anyway.
    #
    # Basically, I'd need an atomic way of verifying file
    # non-existence, then redirecting output to it.  And
    # I don't know of a portable way to do that.   *shrug*
    #
    # This script is trying to make very few assumptions
    # about the local environment and how safe it is to
    # create temporary files and so forth.  It's unavoidable
    # for the box-counting (in general), but we can avoid
    # it here.

    my $t0 = new Benchmark;

    my $pc_sym = gensym();
    my @pc_buf = ();

    open($pc_sym, "./Pair_Count.pl -q 1 Line.5000 |") || die
        "Failed to invoke Pair_Count.pl on Line.5000: ";
    @pc_buf = <$pc_sym>;
    close($pc_sym) || die "Failed to close Pair_Count.pl on Line.5000: ";

    my $trim_sym_in = gensym();
    my $trim_sym_out = gensym();

    open2($trim_sym_out, $trim_sym_in, "./Trim_Flat.pl");
    print $trim_sym_in @pc_buf;
    close($trim_sym_in) || die;
    @trim_buf = <$trim_sym_out>;
    close($trim_sym_out) || die;

    my $rf_sym_in  = gensym();
    my $rf_sym_out = gensym();

    open2($rf_sym_out, $rf_sym_in, "./Robust_LSFit.pl");
    print $rf_sym_in @trim_buf;
    close($rf_sym_in) || die;
    @rf_buf = <$rf_sym_out>;
    close($rf_sym_out) || die;

    ($slope, $y_int, $corr) = @rf_buf;

    chomp($slope);
    chomp($y_int);
    chomp($corr);

    # q=2, so division by ($q-1) is a NOP
    my $fd = $slope;

    my $t1 = new Benchmark;
    my $td = Benchmark::timestr(Benchmark::timediff($t1, $t0));

    print "Time elapsed:  $td\n";
    print "Fractal dimension reported:  $fd (theoretical ~ 0.5)\n";
    print "(line-fitting) Correlation:  $corr\n";


    # Allow for some precision issues due to architectural
    # differences, but .001 should probably be a safe level of
    # accuracy to demand.

    # Expectation changed as a result of occupancy checking.
    if ((int($fd * 1000)/1000) != 0.501) {
        die "Fatal error -- expected the fd value to be ~0.501.";
    }
}



# Phase 9 -- Gnuplot.
#
# We use the data obtained from the previous section to plot.
# Note that the coordinates have already been trimmed to remove
# 'flat' regions at either extreme.

{
    # Using a different method -- to_eps.pl.
    # It creates a 'temp.eps'.

    my $to_eps_fh = gensym;

    open($to_eps_fh, "| ./to_eps.pl") || die;
    print $to_eps_fh @rf_buf;
    print $to_eps_fh @trim_buf;
    close($to_eps_fh) || die;
}


# Phase 10 -- Display

{
    print "*** Displaying resulting line.eps ***\n";

    system("gv temp.eps") && die "Unexpected ghostview return value";
}


# Phase 11 -- Final comments

{
    print "\n";
    print "Your installation appears to be in functioning\n";
    print "properly.  See the README for more information on\n";
    print "using this package.\n";
}
