/* File:     ExtHash.h
 * Purpose:  Spawned off of DBLayer.h, to supply an interface
 *           for an in-memory extensible-hash implementation.
 *
 *           This is currently experimental, and lacks some
 *           useful features such as on-disk support.  The 
 *           advantages over Berkeley DB are:
 *
 *           - Minimalist; should probably have lower CPU
 *             overhead since it supports only the necessary
 *             features.
 *           - Easier to debug, theoretically, for the same
 *             reason.
 *           - Included with this package, which may be nice
 *             if you don't have a pre-built Berkeley DB w/
 *             C++ API ready, and don't have the space, time
 *             or inclination to complete one.
 *
 *           Memory consumption is rather high right now --
 *           considerably more than with Berkeley DB. 
 *           Reducing the page size may help.
 *
 *
 * RCS:
 ************************************************************
 * $Id: ExtHash.h,v 1.1 2001/09/03 19:36:11 lw2j Exp $
 * $Log:	ExtHash.h,v $
 * Revision 1.1  2001/09/03  19:36:11  lw2j
 * Initial revision
 * 
 * Revision 1.2  2001/08/22  17:20:18  lw2j
 * Untabified.
 * 
 * Revision 1.1  2001/08/22  01:47:55  lw2j
 * Initial revision
 * 
 *
 ************************************************************
 */


#ifndef EXTHASH_H
#define EXTHASH_H

#if (USE_EXTHASH==1)

#include <stdio.h>
#include <stdlib.h>
#include "DBLayer.h"

/* forward references */
struct top_level_block_t;
struct second_level_block_t;
struct overflow_block_t;



class ExtHash:public DBLayer {
 public:
  /* Buckets are divided into a main secondary-level page,
   * with overflow pages as need be.  The PAGE_SIZE term
   * helps to determine when we need to split a bucket --
   * the larger the page size, the fewer splits and rehashes,
   * but the greater possible length of a sequential search
   * within a page.
   */
  static const unsigned int PAGE_SIZE=2048;

  /* Dimensionality is used so we know how large the keys
   * are.
   *
   * The 'in_memory' parameter is currently ignored.
   */

  ExtHash(unsigned int dims=0, int in_memory=1);
  virtual ~ExtHash();

  /* Returns 0 if not there. */
  virtual unsigned int fetch(const double *key_dbls);


  /* If it's not there, store and return a 1.
   * If it exists already, increment and return the NEW
   * value.
   * On failure, return a 0.
   */
  virtual unsigned int fetch_plusplus(const double *key_dbls);


  /* Returns 0 on failure, otherwise returns the count --
   * in this application, there's really no reason to 
   * store a 0 (they're all counts, and 0 is the default
   * if not there), so that makes a decent error 
   * indicator, especially combined with errno. 
   */
  virtual unsigned int store(const double *key_dbls, unsigned
                     int value);


  /* The following are for sequential access.  These
   * routines will return -1 on error, 0 on success,
   * and 1 if it's reached a boundary.
   */

  /* Reset cursor to the first element, and retrieve
   * the pointed-to key and its corresponding data.
   *
   * The user is responsible for allocating the space
   * used by the key -- ExtHash will copy it into 
   * there. The value will get the count, 'natch.
   *
   * Adding new items while iterating via a cursor
   * is a bad idea...
   */
   
  virtual int cursor_first(double *key_dbls, unsigned int &value);
  
  /* The same memory caveats apply.  This behaves
   * like cursor_first, except that it does not
   * reposition the cursor.
   */
  virtual int cursor_retrieve(double *key_dbls, unsigned int &value);
  

  /* This advances the cursor and then retrieves
   * the pointed-to object.. */
  virtual int cursor_next(double *key_dbls, unsigned int &value);

 protected:
  unsigned char          bits;   /* # of bits that matter     */
  second_level_block_t **table;  /* will be realloc'd at will */
  second_level_block_t  *head;   /* in linked-list of buckets */
  second_level_block_t  *tail;   /* ditto */

  second_level_block_t*  curs_sec;  /* which bucket          */
  overflow_block_t*      curs_ovr;  /* which overflow page   */
  unsigned int           curs_idx;  /* which entry in page   */

  unsigned int dimensionality;
};

#endif  /* USE_EXTHASH */
#endif  /* EXTHASH_H */
