/* File:     BerkeleyLayer.h
 * Purpose:  To provide a simple layer to the Berkeley
 *           database for storing/retrieving counts.
 *           Once was DBLayer.h.
 *
 *           Since these files are meant to be temporary,
 *           it shouldn't be necessary to ensure portability
 *           via fixed types (int32's, for instance, versus
 *           generic ints) and network-byte order.
 *
 * RCS:
 ************************************************************
 * $Id: BerkeleyLayer.h,v 1.1 2001/09/03 19:36:11 lw2j Exp $
 * $Log:	BerkeleyLayer.h,v $
 * Revision 1.1  2001/09/03  19:36:11  lw2j
 * Initial revision
 * 
 * Revision 1.2  2001/08/22  17:20:18  lw2j
 * Untabified.
 * 
 * Revision 1.1  2001/08/22  01:47:55  lw2j
 * Initial revision
 * 
 *
 ************************************************************
 */


#ifndef BERKELEY_LAYER_H
#define BERKELEY_LAYER_H

#if (USE_BERKELEY == 1)

#include <stdio.h>
#include <stdlib.h>
#include <db_cxx.h>
#include "DBLayer.h"

class BerkeleyLayer:public DBLayer {
 public:
  /* Dimensionality is used so we know how large the keys
   * are.
   */

  BerkeleyLayer(unsigned int dims=0, int in_memory=0);
  virtual ~BerkeleyLayer();


  static const int filename_length = 64;

  /* Returns 0 if not there. */
  virtual unsigned int fetch(const double *key_dbls);


  /* If it's not there, store and return a 1.
   * If it exists already, increment and return the NEW
   * value.
   * On failure, return a 0.
   */
  virtual unsigned int fetch_plusplus(const double *key_dbls);


  /* Returns 0 on failure, otherwise returns the count --
   * in this application, there's really no reason to 
   * store a 0 (they're all counts, and 0 is the default
   * if not there), so that makes a decent error 
   * indicator, especially combined with errno. 
   */
  virtual unsigned int store(const double *key_dbls, unsigned
                     int value);


  /* The following are for sequential access.  These
   * routines will return -1 on error, 0 on success,
   * and 1 if it's reached a boundary.
   */

  /* Reset cursor to the first element, and retrieve
   * the pointed-to key and its corresponding data.
   *
   * The user is responsible for allocating the space
   * used by the key -- BerkeleyLayer will copy it into 
   * there. The value will get the count, 'natch.
   */
   
  virtual int cursor_first(double *key_dbls, unsigned int &value) {
    return cursor_method(key_dbls, value, DB_FIRST);
  }

  /* The same memory caveats apply.  This behaves
   * like cursor_first, except that it does not
   * reposition the cursor.
   */
  virtual int cursor_retrieve(double *key_dbls, unsigned int &value) {
    return cursor_method(key_dbls, value, DB_CURRENT);
  }


  /* This advances the cursor and then retrieves
   * the pointed-to object.. */
  virtual int cursor_next(double *key_dbls, unsigned int &value) {
    return cursor_method(key_dbls, value, DB_NEXT);
  }

 protected:
  /* The three public cursor methods use this. */
  virtual int cursor_method(double *key_dbls, unsigned int &value,
                    u_int32_t flag);

  char  filename[BerkeleyLayer::filename_length];
  Db   *db;
  Dbc  *cursor;
  unsigned int dimensionality;
};

#endif  /* USE_BERKELEY     */
#endif  /* BERKELEY_LAYER_H */
