/* File:     BerkeleyLayer.cc
 * Purpose:  To provide a simple layer to the Berkeley
 *           database for storing/retrieving counts.
 *
 *           Was previously named 'DBLayer.cc'.
 *
 * RCS:
 ************************************************************
 * $Id: BerkeleyLayer.cc,v 1.1 2001/09/03 19:36:11 lw2j Exp $
 * $Log:	BerkeleyLayer.cc,v $
// Revision 1.1  2001/09/03  19:36:11  lw2j
// Initial revision
// 
// Revision 1.3  2001/08/22  17:20:18  lw2j
// Untabified.
// 
// Revision 1.2  2001/08/22  04:27:01  lw2j
// Added the (necessary!) ulen fields, and fixed up the fetching/storing
// and iteration code.
// 
 * Revision 1.1  2001/08/22 01:47:55  lw2j
 * Initial revision
 *
 *
 ************************************************************
 */

#if (USE_BERKELEY == 1)
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <limits.h>
#include <assert.h>
#include <unistd.h>
#include <errno.h>

#include "BerkeleyLayer.h"

/* Type should be either DB_BTREE or DB_HASH.  I really, 
 * really don't recommend using RECNO here.
 *
 * Dimensionality is used so we know how large the keys
 * are.
 */

BerkeleyLayer::BerkeleyLayer(unsigned int dims, 
                 int in_memory):db(NULL), cursor(NULL) {
  
  /* Don't be silly. */

  assert(dims > 0);

  dimensionality = dims;

  filename[0] = '\0';

  if (in_memory) {
    db = new Db(0,0);
    assert(!(db->open(NULL, NULL, DB_BTREE, 0, 0)));
  } else {
    int  fno = 0;

    sprintf(filename, "BerkeleyLayer_%.12d_XXXXXX", getpid());
    fno = mkstemp(filename);

    assert(fno != -1);
    assert(!close(fno));

    /* It's likely at this stage that 'filename' is a 
     * safe one to use; we'll just have to remember to
     * unlink it later.
     */


    db = new Db(0,0);
    assert(!(db->open(filename, NULL, DB_BTREE, DB_CREATE | DB_TRUNCATE,
                    0600)));
  }

  /* Get a cursor. */
  assert(!(db->cursor(NULL, &cursor, 0)));
}



BerkeleyLayer::~BerkeleyLayer(void) {
  assert(!(db->close(0)));

  delete db;

  if (filename[0]) {
    /* Clean up the temporary file. */
    assert(!unlink(filename));
  }
}



/* Returns 0 if not there. */
unsigned int BerkeleyLayer::fetch(const double *key_dbls) {
  Dbt key(0,0);
  Dbt data(0,0);

  unsigned int value = 0;
  int retval = 0;

  if ((!db) || (!key_dbls) || (!dimensionality)) {
    errno = EINVAL;
    return 0;
  }
  
  key.set_data((void*) key_dbls);
  key.set_size(sizeof(double) * dimensionality);
  key.set_flags(DB_DBT_USERMEM);
  key.set_ulen(sizeof(double) * dimensionality);

  data.set_data((void*) (&value));
  data.set_size(sizeof(unsigned int));
  data.set_flags(DB_DBT_USERMEM);
  data.set_ulen(sizeof(unsigned int));

  if (!((retval=db->get(NULL, &key, &data, 0)))) {
    return value;
  }

  if (retval == DB_NOTFOUND) {
    return 0;
  } else {
    /* Assumption violated:  under normal operation,
     * the entry not being there should be the only
     * failure mode.  Assert(0) and exit, because
     * the odds are very strong there's a bug.
     */
    assert(0);
  }
}



/* If it's not there, store and return a 1.
 * If it exists already, increment and return the new
 * value.
 * On failure, return a 0.
 */
unsigned int BerkeleyLayer::fetch_plusplus(const double *key_dbls) {
  Dbt key(0,0);
  Dbt data(0,0);

  unsigned int value = 0;
  int retval = 0;

  if ((!db) || (!key_dbls) || (!dimensionality)) {
    errno = EINVAL;
    return 0;
  }

  key.set_data((void*) key_dbls);
  key.set_size(sizeof(double) * dimensionality);
  key.set_flags(DB_DBT_USERMEM);
  key.set_ulen(sizeof(double) * dimensionality);

  data.set_data((void*) (&value));
  data.set_size(sizeof(unsigned int));
  data.set_flags(DB_DBT_USERMEM);
  data.set_ulen(sizeof(unsigned int));

  if (!(retval=(db->get(NULL, &key, &data, 0)))) {
    value = value + 1;
    
    assert(!((retval=db->put(NULL, &key, &data, 0))));
    return value;
  } 
  
  if (retval == DB_NOTFOUND) {
    /* new entry:  store a 1 */
    value = 1;

    assert(!((retval=db->put(NULL, &key, &data, 0))));
    return value;
  } else {
    /* Assumption violated:  under normal operation,
     * the entry not being there should be the only
     * failure mode.  Assert(0) and exit, because
     * the odds are very strong there's a bug.
     */
    assert(0);
  }
}



/* Returns 0 on failure, otherwise returns the count --
 * in this application, there's really no reason to 
 * store a 0 (they're all counts, and 0 is the default
 * if not there), so that makes a decent error 
 * indicator, especially combined with errno. 
 */
unsigned int BerkeleyLayer::store(const double *key_dbls, 
                            unsigned int value) {

  Dbt key(0,0);
  Dbt data(0,0);

  int retval = 0;

  if ((!db) || (!key_dbls) || (!dimensionality)) {
    errno = EINVAL;
    return 0;
  }

  key.set_data((void*) key_dbls);
  key.set_size(sizeof(double) * dimensionality);
  key.set_flags(DB_DBT_USERMEM);
  key.set_ulen(sizeof(double) * dimensionality);

  data.set_data((void*) (&value));
  data.set_size(sizeof(unsigned int));
  data.set_flags(DB_DBT_USERMEM);
  data.set_ulen(sizeof(unsigned int));

  assert(!((retval=db->put(NULL, &key, &data, 0))));
  
  return value;
}



/* This provides a common method for accessing the db_cursor
 * interface.
 */
int BerkeleyLayer::cursor_method(double *key_dbls, unsigned int &value, 
                           u_int32_t flag) {
  Dbt key(0,0);
  Dbt data(0,0);

  int retval = 0;

  assert(cursor);
  if ((!db) || (!key_dbls) || (!dimensionality)) {
    errno = EINVAL;
    return 0;
  }

  key.set_data((void*) key_dbls);
  key.set_size(sizeof(double) * dimensionality);
  key.set_flags(DB_DBT_USERMEM);
  key.set_ulen(sizeof(double) * dimensionality);

  data.set_data((void*) (&value));
  data.set_size(sizeof(unsigned int));
  data.set_flags(DB_DBT_USERMEM);
  data.set_ulen(sizeof(unsigned int));
    
  retval=cursor->get(&key, &data, flag);

  if (!retval) {
    /* success */
    return 0;
  }

  if (retval == DB_NOTFOUND) {
    /* boundary of sorts -- there's nobody there */
    return 1;
  }

  /* Error besides an empty database.  Hrm? */
  assert(0);
}

#endif /* USE_BERKELEY */
