#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/param.h>
#include <assert.h>

#include "config_unix.h"
#include "config_win32.h"

#include "util.h"
#include "config.h"
#include "equation.h"

#include <Util/inetMisc.h>
#include <Util/seqBuffer.h>

#include "tfrcReceiver.h"

// #define DEBUG1
/* #define DEBUG_EXTENDED */

int TfrcReceiver::GetRemoteProcess(int *addr, int *port) {
  if (this->raddr == 0) {
    assert(this->rport == 0);
    return -1;
  }

  assert(this->rport != 0);
  *addr = this->raddr;
  *port = this->rport;
  return 0;
}


/* use mult to deweight old loss intervals */
double TfrcReceiver::lossfraction() { 
  double pplavg1 = 0.0, pplavg2 = 0.0;
  double weightsum1 = 0.0, weightsum2 = 0.0;
  int size1 = 0, size2 = 0, ppl1, ppl2;
  int num_intervals = 0;
  int i;
  
  /* loss avg. without latest interval */
  for (i = 0; i < this->weights; ++i) {
    ppl2 = this->ppl[(this->ppl_cur - i + this->weights)%(this->weights + 1)];
    if (ppl2 > 0) {
      /* pplavg2 += ppl2 * this->weight[i] * mult[i]; */
      /* weightsum2 += this->weight[i] * mult[i]; */
      pplavg2 += ppl2 * this->weight[i];
      weightsum2 += this->weight[i];
      ++num_intervals;
    }
  }
  
  /* determine deweighting factor */
  if (num_intervals > 0 && weightsum2 > 0 && pplavg2 > 0) {
    pplavg2 /= weightsum2;
    this->new_mult = min((DW_FACTOR * pplavg2)/this->ppl[this->ppl_cur], 1.0);
  } else {
    this->new_mult = 1.0;
  }
  
  /* reset for second computation */
  pplavg2 = 0.0;
  weightsum2 = 0.0;
  
  /* compute lossrate with/without packets after last loss
   * -->only take these packets into account if that decreases the loss rate */
  for (i = 0; i < this->weights; ++i) {
    ppl1 = this->ppl[(this->ppl_cur - i + this->weights+1)%(this->weights+1)];
    ppl2 = this->ppl[(this->ppl_cur - i + this->weights) % (this->weights+1)];
    /* no deweighting of current interval */
    if (ppl1 > 0) {
      if (i > 0) {
	/* double m1 = max(mult[i] * this->new_mult, MIN_MULT); */
	double m1 = max(this->new_mult, MIN_MULT);
	weightsum1 += this->weight[i] * m1;
	pplavg1 += ppl1 * this->weight[i] * m1;
      } else {
	weightsum1 += this->weight[i];
	pplavg1 += ppl1 * this->weight[i];
      }
      size1 += ppl1;
    }
    if (ppl2 > 0) {
      if (i > 0) {
	/* double m2 = max(mult[i + 1] * this->new_mult, MIN_MULT); */
	double m2 = max(this->new_mult, MIN_MULT);
	weightsum2 += this->weight[i] * m2;
	pplavg2 += ppl2 * this->weight[i] * m2;
      } else {
	weightsum2 += this->weight[i];
	pplavg2 += ppl2 * this->weight[i];
      }
      size2 += ppl2;
    }
  }
  
  assert(weightsum1 > 0);
  assert(weightsum2 > 0);
  assert(pplavg1 > 0);
  assert(pplavg2 > 0);
  
  pplavg1 /= weightsum1;
  pplavg2 /= weightsum2;
  
  if (pplavg1 > pplavg2) {
    this->ppl_end = this->high_offs;
    this->ppl_start = this->ppl_end - size1 + 1;
  } else {
    this->ppl_end = this->high_offs - this->ppl[this->ppl_cur];
    this->ppl_start = this->ppl_end - size2 + 1;
  }
  
  return min(1.0 / pplavg1, 1.0 / pplavg2);
}



void TfrcReceiver::fake_history(int ppl_est) {
  if (this->ppl[0] < ppl_est)
    this->ppl[0] = ppl_est;
}



void TfrcReceiver::set_weights() {
  if (this->const_weights) {
    for (int i = 0; i < this->weights; ++i)
      this->weight[i] = 1.00;
  } else {
    double delta = 1.0 / ((this->weights + 1) / 2 + 1);
    for (int i = 0; i < this->weights / 2; ++i)
      this->weight[i] = 1.00;
    for (int i = this->weights / 2; i < this->weights; ++i)
      this->weight[i] = this->weight[i-1] - delta;
  }

  if (verbosity > 1) {
    printf("\nweights ");
    for (int i = 0; i < this->weights; ++i)
      printf("%3.2f ", this->weight[i]);
  }
}


void TfrcReceiver::log_loss(int64 arrival) {
  // cerr << "\nloss observed";

  /* account for missing packets */
  this->ppl[this->ppl_cur] += this->mseq_offs - this->high_offs - 1; 
        
  /* log params */
  this->ppl_hist[this->num_loss - 1] = this->ppl[this->ppl_cur];
  this->ts_hist[this->num_loss - 1] = arrival / MS_TO_US;
  this->seq_hist[this->num_loss - 1] = this->high_offs + 1;
                                        
  /* forget about packets missing inbetween */
  this->high_offs = this->mseq_offs; 

  /* shift and update mult array and reset new_mult */
  /* no longer necessary because of history faking
     for (i = weights; i > 0; --i) {
     mult[i] = max(mult[i - 1] * new_mult, MIN_MULT);
     }
     mult[0] = 1.0;
  */
  if (this->new_mult < 1.0) {
    /* fake history */
    double l_fake = lossfraction();
    this->new_mult = 1.0; /* reset deweighting factor */
    for (int i = 0; i <= MAX_WEIGHTS; ++i) {
      if (this->ppl[i] > 0) /* only fake intervals that were already used */
	this->ppl[i] = (int) (1.0 / l_fake + 0.5);
    }
  }
  
  this->ppl_cur = (this->ppl_cur + 1) % (this->weights + 1);
  this->ppl[this->ppl_cur] = 0;
}


int TfrcReceiver::update_history(u_int32 mseq, int64 arrival, double rtt, 
				 u_int32 tzero, double b_act, int pktsize) {
  
  /* init this->hist.seq_offset with first mseq received */
  if (this->hist.seq_offset == 0xFFFFFFFF) {
    this->hist.seq_offset = mseq;
  }

  /* ignore packets with seq number lower than that of first packet 
   * (out of order) 
   */
  if (mseq < this->hist.seq_offset) {
    return 1;
  }

  this->mseq_offs = mseq - this->hist.seq_offset;
  
  
  /* ignore reordered packet (after TD ACK) or duplicate packet */
  if (this->mseq_offs <= this->high_offs) {
    return 1;
  }


  if (! ((arrival - this->hist.lastloss > rtt * MS_TO_US) 
      && (this->round > this->hist.lastround) 
      && (this->mseq_offs > this->high_offs + 1))) {

    /* packet in order or less than one RTT from previous loss */
    this->high_offs = this->mseq_offs;
    ++(this->ppl[this->ppl_cur]); /* add packet to last interval */
    return 1;
  }


  /* check if missing packet(s) already arrived
   * and the reordered packets can now be inserted in the history 
   */
  if ((this->hist.num_reordered > 0) 
      && (this->hist.seq_reordered[0] == this->high_offs + 1)) {
    if ((this->hist.num_reordered == 2) 
	&& (this->hist.seq_reordered[1] == this->high_offs + 2)) {
      this->high_offs += 2;
      this->hist.num_reordered = 0;
    } else {
      this->hist.seq_reordered[0] = this->hist.seq_reordered[1];
      --(this->hist.num_reordered);
    }
  }
    
  /* this->high_offs might have changed, check again for missing packet */
  if (this->mseq_offs > this->high_offs + 1) {
    /* (emulate triple duplicate ACK) */
    if (this->hist.num_reordered < 2) { 
      if (this->hist.num_reordered == 1) {
	if (this->hist.seq_reordered[0] > this->mseq_offs) {
	  this->hist.seq_reordered[1] = this->hist.seq_reordered[0];
	  this->hist.seq_reordered[0] = this->mseq_offs;
	} else {
	  this->hist.seq_reordered[1] = this->mseq_offs;
	}
      } else {
	this->hist.seq_reordered[0] = this->mseq_offs;
      }
      ++(this->hist.num_reordered);
    } else {
      /* loss event */
      if((this->hist.seq_reordered[0] - this->high_offs == 2) 
	 && (this->hist.first_loss_ignored == 0)) {
	/* ignore first SINGLE loss */
	this->high_offs = this->mseq_offs;/* forget about missing packet */
	this->hist.num_reordered = 0;
      } else {
	++this->num_loss;

	if (this->num_loss >= HIST_SIZE) {
	  cerr << "\nloss history buffer overflow";
	  return 0;
	}
	
	if (this->hist.fake_hist == 0) {
	  /* fake history (once) -> half send rate if that results in
	     a smaller initial loss rate, otherwise it was probably a 
	     random lost packet */
	  fake_history((int) (1.0 / lossfraction_eq(b_act / 2.0, rtt, 
						    tzero, pktsize)));
	  this->hist.fake_hist = 1;
	}
	log_loss(arrival);
	this->hist.num_reordered = 0;
	this->hist.lastloss = arrival;
	this->hist.lastround = this->round;
      }
      this->hist.first_loss_ignored = 1;
    }
  }
  ++(this->ppl[this->ppl_cur]); /* add packet to last interval */

  return 1;
}


/* return # of bytes (excluding TFRC header), or -1 if connection is
 * believed to be terminated, */
int TfrcReceiver::recv_single(char *buf, int buflen) {
  char recv_buffer[MAX_UDP_PKTSIZE];
  struct pkt_dtg *packet = (struct pkt_dtg *)recv_buffer;
  
  /* packets ready */
  struct sockaddr_in sin;
  socklen_t len = sizeof(sin);
  int datalen;

  
  if (isFirst) {
    isFirst = FALSE;
    memcpy(&sin, &firstSA, len);
    datalen = firstBufLen;
    memcpy(recv_buffer, firstBuf, datalen);
    
    if (isConnected) {
      cerr << "tfrcReceiver: "
	   << getInetAddrName(GetMyAddr())
	   << " received data passed by dataMgrTfrc\n";
    }
  } else {
    datalen = Recvfrom2(this->fd, recv_buffer, MAX_UDP_PKTSIZE, 0, 
			(struct sockaddr *)&sin, &len);
    
    if (datalen < 0) {
      switch(errno) {
      case ECONNREFUSED: /* connection refused (ICMP port unreachable) */
	if (verbosity > 0) {
	  printf("\nTFRC RCV %s/%d connection refused (ICMP)",
		 getInetAddrName(this->raddr), this->rport);
	}
	return -1;
	break;
      default:
	assert("unhandled error code");
	break;
      }
    }
  }

 /* set the remote addr/port, if it has not been set previously */
  if (! isConnected) {
    if (this->raddr != 0) {
      if (this->raddr != htonl(sin.sin_addr.s_addr) || 
	  this->rport != htons(sin.sin_port)) {
	printf("\nTFRC RCV WRN %s <- %s/%d: received bad pkt %s/%d (%d) 1",
	       getInetAddrName(GetMyAddr()),
	       getInetAddrName(this->raddr), this->rport,
	       getInetAddrName(htonl(sin.sin_addr.s_addr)), 
	       htons(sin.sin_port), datalen);
	
	fprintf(stderr, "\nTFRC RCV WRN %s -> %s/%d: rcvd bad pkt %s/%d (%d) 1",
		getInetAddrName(GetMyAddr()),
		getInetAddrName(this->raddr), this->rport,
		getInetAddrName(htonl(sin.sin_addr.s_addr)), 
		htons(sin.sin_port), datalen);

	return 0;
      }
    } else {
      this->raddr = htonl(sin.sin_addr.s_addr);
      this->rport = htons(sin.sin_port);
    }
    if (verbosity > 0) {
      printf("\nTFRC RCV %s/%u connected", getInetAddrName(this->raddr), 
	     this->rport);
    }
    
    isConnected = TRUE;
  } else {
    if (this->raddr != htonl(sin.sin_addr.s_addr) || 
	this->rport != htons(sin.sin_port)) {
      printf("\nTFRC RCV WRN %s <- %s/%d: received bad pkt %s/%d (%d) 2",
	     getInetAddrName(GetMyAddr()),
	     getInetAddrName(this->raddr), this->rport,
	     getInetAddrName(htonl(sin.sin_addr.s_addr)), 
	     htons(sin.sin_port), datalen);
      
      fprintf(stderr, "\nTFRC RCV WRN %s -> %s/%d: rcvd bad pkt %s/%d (%d) 2",
	      getInetAddrName(GetMyAddr()),
	      getInetAddrName(this->raddr), this->rport,
	      getInetAddrName(htonl(sin.sin_addr.s_addr)), 
	      htons(sin.sin_port), datalen);

      return 0;
    }
  }
  

  /* emulate transient behavior: drop recvd pkts probabilistically */
  if (ETBRecvDrop(htonl(sin.sin_addr.s_addr))) {
    return 0;  
  }

  this->data_per_report += datalen;
  
  int64 arrival = get_time();
  u_int32 mseq    = ntohl(packet->mseq);
  double b_act   = ntohl(packet->bitrate);
  u_int32 rtt_cur = ntohl(packet->rtt);
  
  u_int32 mode = ntohl(packet->mode); /* wait until RTT decreases->M_NORMAL */
  
  if (this->rtt_ema > 0.0)
    this->rtt_ema = (1 - RTT_DECAY) * this->rtt_ema + RTT_DECAY * rtt_cur;
  else
    this->rtt_ema = rtt_cur;
  
  u_int32 tzero = (u_int32)max(4 * max(this->rtt_ema, rtt_cur), 500);
  this->rtt_cur_old = rtt_cur;
  
  /* adjust round number only if sender has responded to the first loss event
   * and kicked out of SLOWSTART
   */
  if (!(mode == M_SLOWSTART && this->num_loss > 0)) {
    this->round = ntohl(packet->round);
  }
  
  if (update_history(mseq, arrival, rtt_cur, tzero, b_act, datalen) == 0) {
    assert("error in update_history" == NULL);
  }      
  

  /* track loss -- yhchu */
  // if (seqBuffer == NULL) { seqBuffer=new SeqBuffer(); }
  if (seqBufferSes == NULL) { seqBufferSes=new SeqBuffer(); }

  // seqBuffer->RecvDataPkt(mseq);
  switch(seqBufferSes->RecvDataPkt(mseq)) {
  case BUF_OK:
  case BUF_REORDER:
  case BUF_DUP:
  case BUF_LATE:
    break;
  default:
    cerr << "unknown DataPktType";
    assert(0);
    break;
  }

   /* this includes TFRC packet overhead (sizeof(struct pkt_dtg)) */
  this->reportBW->Update(datalen);


  /* send REPORTS_PER_RTT reports per rtt (when at least one packet
   * was received since last report), send report for every packet
   * in M_INITRTT mode and when we encountered loss and are still in
   * M_SLOWSTART 
   */
  double  l_act = 0.0;
  double  b_exp = 0;   /* expected bitrate */
  double b_rep = 0;    /* rcvd bitrate within one RTT (bytes/sec) */
  if (((arrival - this->last_report) / MS_TO_US >= 
       (this->rtt_ema / REPORTS_PER_RTT)) 
      || (mode == M_INITRTT) 
      || (mode == M_SLOWSTART && this->num_loss > 0)) {
    
    /* update reported bitrate every report */
    b_rep = this->data_per_report / 
      ((double)(arrival - this->last_report)/MICROSEC);
    this->data_per_report = 0;
    
    /* calculate expected bitrate */
    if (this->num_loss > 0) {
      l_act = lossfraction();
      b_exp = bitrate_eq(l_act, this->rtt_ema, tzero, datalen);
    } else {
      l_act = 0.0;
      b_exp = 0.0;
    }
    
    /* report lossrate etc. */
    char send_buffer[REP_SIZE];
    struct rep_dtg *rep    = (struct rep_dtg *)send_buffer;

    rep->mseq     = packet->mseq;
    rep->dseq     = packet->dseq;
    rep->ts       = packet->ts;
    rep->b_rep    = htonl((u_int32) b_rep);  /* truncated */
    rep->b_exp    = htonl((u_int32) b_exp);

    //    rep->loss_cur = htonl((u_int32) (seqBuffer->ComputeLoss()*1000000.0));  /* yhchu */
    //rep->loss_ses = htonl((u_int32) (seqBufferSes->ComputeLoss()*1000000.0));  /* yhchu */

    /* yhchu */
    rep->rtt_smooth = htonl((u_int32) (this->rtt_ema * 100.0));
    rep->loss_smooth = htonl((u_int32) (l_act*1000000.0));  /* smoothed loss */

    int bw = reportBW->Report();
    if (bw == EstimateBandwidth::NO_REPORT_AVAILABLE) {
      rep->bw_smooth = htonl(0);  /* the first second */
    } else {
      rep->bw_smooth = htonl((u_int32)bw);
    }
    rep->loss_ses = htonl((u_int32)(seqBufferSes->ComputeLoss()*1000000.0));  
    rep->bw_ses = htonl((u_int32)reportBW->ReportSessionBw());  
    
    assert(this->raddr != 0 && this->rport != 0);
    struct sockaddr_in sin = GetSockaddr(this->raddr, this->rport);

    /* do nothing if Sendto2 fails */
    Sendto2(this->fd, (const char *)rep, REP_SIZE, 0, 
	    (const struct sockaddr *)&sin, sizeof(sin), "tfrcReceiver");
    
    this->last_report = arrival;

    // seqBuffer->Reset();  /* reset the loss rate computation */
  }

  /* copy of the data content */
  int ret = datalen - sizeof(struct pkt_dtg);
  assert(ret <= buflen && ret > 0);
  memcpy(buf, recv_buffer+(sizeof(struct pkt_dtg)), ret);
  
  return ret;
}


void TfrcReceiver::Init() {
  for (int i = 0; i < HIST_SIZE; ++i) {
    this->ppl_hist[i] = 0;
    this->ts_hist[i] = 0;
    this->seq_hist[i] = 0;
  }

  for (int i = 0; i <= MAX_WEIGHTS; ++i) {
    this->ppl[i] = 0;
    /* mult[i] = 1.0; */
  }

  this->mseq_offs = 0;
  this->high_offs = -1;
  this->ppl_start = this->ppl_end = 0;
  this->ppl_cur = 0;
  this->weights = 8;       /* tunable var */
  this->round = 0;
  this->num_loss = 0;
  this->const_weights = 0;  /* tunable var */
  this->new_mult = 1.0;
  
  this->hist.seq_offset = 0xFFFFFFFF;
  this->hist.num_reordered = 0;
  this->hist.fake_hist = 0;
  this->hist.first_loss_ignored = 0;
  this->hist.lastloss = 0;
  this->hist.lastround = 0;

  /* recv_single */
  this->data_per_report = 0;
  this->last_report = 0;
  this->rtt_cur_old = 0;                     /* rtt_cur of previous packet */
  this->rtt_ema = 0.0;

  set_weights();
  
  this->timer = new TfrcTimer((int64)TFRC_RCV_MAX_IDLE * (int64)MICROSEC, 
			      (int64)TFRC_RCV_MAX_INIT * (int64)MICROSEC);

  this->isConnected = FALSE;
  this->isFirst = FALSE;
  this->firstBufLen = 0;

  this->seqBuffer = NULL;
  this->seqBufferSes = NULL;

  this->reportBW = NULL;
  SetReportBW();
}


TfrcReceiver::TfrcReceiver(char *buf, int buflen, 
			   struct sockaddr *sa, int socklen) {
  Init();
  
  this->fd = Socket(AF_INET, SOCK_DGRAM, 0);
  SetsockoptReuseAddrPort(this->fd);
  SetsockNonBlocking(this->fd);
  
  this->raddr = htonl(((struct sockaddr_in *)sa)->sin_addr.s_addr);
  this->rport = htons(((struct sockaddr_in *)sa)->sin_port);

  InjectData(buf, buflen, sa, socklen);
}


TfrcReceiver::TfrcReceiver(int port) {
  Init();
  
  this->fd = Socket(AF_INET, SOCK_DGRAM, 0);
  SetsockoptReuseAddrPort(this->fd);
  SetsockNonBlocking(this->fd);

  assert(port != 0);
  this->laddr = INADDR_ANY;  /* any addr */
  this->lport = port;           /* any port */

  this->raddr = 0;
  this->rport = 0;

  struct sockaddr_in sin = GetSockaddr(this->laddr, this->lport);
  Bind(this->fd, (struct sockaddr *)&sin, sizeof(sin));
}


void TfrcReceiver::SetIdleTimer(int64 maxIdleTime, int64 maxInitIdleTime) {
  if (this->timer != NULL) {   delete this->timer;  }

  this->timer = new TfrcTimer(maxIdleTime, maxInitIdleTime);
}


void TfrcReceiver::InjectData(char *buf, int buflen, 
			      struct sockaddr *sa, int socklen) {
  assert(buflen <= MAX_UDP_PKTSIZE);
  this->firstBufLen = buflen;
  memcpy(this->firstBuf, buf, this->firstBufLen);
  
  assert(socklen == sizeof(struct sockaddr_in));
  memcpy(&this->firstSA, sa, socklen);

  isFirst = TRUE;
}

int TfrcReceiver::SetFD(fd_set *rs, fd_set *) {
  FD_SET(this->fd, rs);

  if (isFirst) {
    return 0;
  }
  return (this->timer->Expire()/1000);
}


/* return -1 if connection is believed to be terminated, 0 otherwise
 */
int TfrcReceiver::Process(fd_set *rs, fd_set *, char *buf, int buflen) {
  if (isFirst) {
    this->timer->Update();
    SetReportBW();
    return recv_single(buf, buflen);
  }

  if (FD_ISSET(this->fd, rs)) {
    FD_CLR(this->fd, rs);
    this->timer->Update();
    
    return recv_single(buf, buflen);
  }

  if (this->timer->Expire() == 0) {
    if (verbosity > 0) 
      printf("\nTFRC RCV %s/%d timer expired", getInetAddrName(raddr), rport);
    return -1;
  } else {
    return 0;
  }
}

void TfrcReceiver::SetReportBW() {
  if (this->reportBW != NULL) {
    delete this->reportBW;
  }

  /* set offset */
  this->reportBW = new EstimateBandwidth(1, 1);
}


int TfrcReceiver::GetRemoteAddr() {
  return raddr;
}


int TfrcReceiver::GetRemotePort() {
  return rport;
}

void TfrcReceiver::DebugExit() {
  if (verbosity > 0) {
    printf("\nreceiver finished, writing history files");
    
    printf("\nlosshistory");
    for (int i = 0; i < this->num_loss; ++i) {
      printf("\nHIST %f %i %i", (float) this->ts_hist[i] / MILLISEC, 
	     this->ppl_hist[i], this->seq_hist[i]);
    }
    printf("\nweights");
    for (int i = 0; i < this->weights; ++i)
      printf("%f ", this->weight[i]);
  }
}


TfrcReceiver::~TfrcReceiver() {
  Close();
  delete reportBW;
  delete timer;
}


int TfrcReceiver::Close() {
  return close(this->fd);
}
