#include <assert.h>
#include <stdlib.h>
#include <stdarg.h> /* va_list in FreeBSD and SUN */
#include <string.h>
#include <errno.h>
#include <strings.h>
#include <unistd.h>
#include <iostream.h>
#include <stdio.h>

#include <Gossip/global.h>
#include <Util/seqBuffer.h>

#include "inetProxy.h"
#include "streamSelector.h"
#include "rtp.h"

StreamSelector::StreamSelector(int srcAddr, 
			       int *sessionPorts, 
			       int sessionSize) {
  this->srcAddr = srcAddr;
  assert(sessionSize > 0 && sessionSize <= MAX_PORTS_PROXY_LISTENS);
  
  this->sessionSize = sessionSize;
  for (int i=0; i<sessionSize; i++) {
    sessions[i].cntlPort = RtpCntlPort(sessionPorts[i]);
    sessions[i].dataPort = RtpDataPort(sessionPorts[i]);
    sessions[i].seqBuf = new SeqBuffer();
    sessions[i].loss = -1;
    sessions[i].numPktDrop = 0;
  }

  startTime = 0;
  curQuality = 0;
}


StreamSelector::~StreamSelector() {
  for (int i=0; i<sessionSize; i++) {
    delete sessions[i].seqBuf;
  }
}


/* modify session[i]->loss periodically
 * - ignore first SS_IGNORE seconds of data
 * - re-compute loss rate every SS_INTERVAL seconds 
 * - exponential smoothing 
 */
void StreamSelector::ComputeLoss(int id, int seq) {
  long curTime = GetCurrTime();
  
  if ((curTime - startTime) < SS_IGNORE_MS) { 
    return;
  }
  
  if (nextTimeComputeLoss == 0) {
    nextTimeComputeLoss = curTime + SS_INTERVAL_MS;
  }

  // update packets
  sessions[id].seqBuf->RecvDataPkt(seq);
  
  if (curTime < nextTimeComputeLoss) {
    return;
  }

  if (verbosity > -1) {
    printf("\nStreamSelector %s", GetNameByAddr(srcAddr));
  }

  for (int i=0; i<sessionSize; i++) {
    float loss = sessions[i].seqBuf->ComputeLoss();

    if (sessions[i].loss == -1) {
      sessions[i].loss = loss;
    } else {
      sessions[i].loss = loss * SS_SMOOTH_EXP + 
	sessions[i].loss * (1.0 - SS_SMOOTH_EXP);
    }

    if (verbosity > -1) {
      printf(" %d/%d/%.2f/%.2f/%d/%d", i, sessions[i].dataPort, loss,
	     sessions[i].loss,
	     sessions[i].seqBuf->GetNumPktsRcvd(),
	     sessions[i].numPktDrop);
    }

    sessions[i].seqBuf->Reset();
    sessions[i].numPktDrop = 0;
  }

  sessions[id].seqBuf->RecvDataPkt(seq);
  nextTimeComputeLoss = curTime + SS_INTERVAL_MS;
  
  /* recompute level */
  AdjustQuality();
  
  if (verbosity > -1) {
    printf(" Quality %d", curQuality);
  }
}

// return port # to forward (i.e. sessions[i].cntl/data port
// to if data should be passed up to upper layer
// else return FALSE
/* design question: XXX
 * 1. should we have our own seq # in our header, or should we assume
 * the existence of RTP and tap into their header?
 * 2. how do I define packet loss rate? (avg over interval, smoothing...)
 */
int StreamSelector::Recv(int port, int seq) {
  long curTime = GetCurrTime();
  
  // printf("\nRECV %d %d", port, seq);

  /* get the session id */
  int id; for (id = 0; id <sessionSize; id++) {
    if (port == sessions[id].cntlPort ||
	port == sessions[id].dataPort) {
      break;
    }
  }

  /* session not found */
  if (id == sessionSize) {
    return port;
  }

  if (IsRtpCntlPort(port)) {
    return sessions[0].cntlPort;
  } else {
    /* initialize start time */
    if (startTime == 0) {
      startTime = curTime;
    }
    
    /* recompute loss rate */
    ComputeLoss(id, seq);
    
    if (id == curQuality) {
      return sessions[0].dataPort;
    } else {
      sessions[id].numPktDrop++;
      return FALSE;
    }
  }
}


void StreamSelector::AdjustQuality() {
  /* if observing good performance, try to move up quality */
  if (sessions[curQuality].loss <= SELECT_LOSS_LOW) {
    if ((curQuality < (sessionSize-1)) &&
	(sessions[curQuality+1].loss <= SELECT_LOSS_LOW)) {
      curQuality++;
    }
  }
  
  /* if observing bad performance, try to move down quality */
  if (sessions[curQuality].loss >= SELECT_LOSS_HIGH) {
    // do not change if curQuality is already the lowest
    if (curQuality > 0) {
      curQuality--;
    }
  }
  
  assert(curQuality >= 0 && curQuality < sessionSize);
}
