#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <strings.h>
#include <unistd.h>
#include <iostream.h>
#include <stdio.h>
#include <netdb.h>

#include <arpa/inet.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <netinet/in.h>	/* sockaddr_in{} and other Internet defns */

#include <Gossip/global.h>

#include <Util/packetBuffer.h>  // where MAX_UDP_PKTSIZE defined
#include <Util/inetTypes.h>
#include <Util/inetMisc.h>
#include <Util/misc.h>
#include <Util/estimateBandwidth.h>

#include <Internet/inetGossipHost.h>


#include "inetProxy.h"
#include "listenPort.h"
#include "clientMgr.h"

ListenPort::ListenPort(int addr, int port, int priority,
		       InetGossipHost *gossipHost) {
  this->addr = addr;
  this->port = port;
  this->priority = priority;
  this->gossipHost = gossipHost;
  this->pktSeqNum = 0;

  isMulticast = IsMulticastAddr(addr);
  bwRcvdFromApp = NULL;

  recvFd = Socket(AF_INET, SOCK_DGRAM, 0);
  SetsockoptReuseAddrPort(recvFd);
  SetsockNonBlocking(recvFd);

  /* It is VERY important to set recv bufsize (and perhaps send buf) properly.
   * In my windows machine (dahon.cmcl), if both vic and narada
   * are running at the same time, the system is doing context switching
   * between these two machines.  If the buffer size is not large enough,
   * data may be dropped before Narada has a chance to read from the
   * rcv buffer.
   * -- yhchu
   */
  SetsockoptBufSize(recvFd, 65535);
  
  // bind
  struct sockaddr_in sock;
  bzero((char *)(&sock), sizeof(sock));
  sock.sin_family = AF_INET;
  sock.sin_addr.s_addr = htonl(addr);
  sock.sin_port = htons(port);
  Bind(recvFd, (struct sockaddr *)&sock, sizeof(sock));

  /* multicast? join the group */
  if (isMulticast) {
    struct ip_mreq mr;
    mr.imr_multiaddr.s_addr = htonl(addr);
    mr.imr_interface.s_addr = htons(INADDR_ANY);
    
    Setsockopt(recvFd, IPPROTO_IP, IP_ADD_MEMBERSHIP, 
	       (char *)&mr, sizeof(mr));
  }
}


ListenPort::~ListenPort() {
  if (bwRcvdFromApp != NULL) {
    delete bwRcvdFromApp;
  }
}


int ListenPort::SetFD(fd_set *rs, fd_set *) {
  FD_SET(recvFd, rs);
  return 1000;
}

/* read data from application */
int ListenPort::ReadFromNetwork(fd_set *rs, fd_set *) {
  
  if (! FD_ISSET(recvFd, rs)) {
    return -1;
  }
    
#define MAX_NUM_DATA_READ  10000
  int numDataRead = 0;
  
  while (FuncRecvFromApp() >= 0) {
    if (numDataRead++ <= MAX_NUM_DATA_READ) continue;
    
    MyWarning("Sending rate too fast to handle");
    break;
  }
  return 0;
}


/* Return bufLen if succeed, or -1 if no data to read
 */
int ListenPort::FuncRecvFromApp() {
  
  /* receive data from application */
  char encapBuf[MAX_UDP_PKTSIZE];
  char *buf = encapBuf+sizeof(ApplicationIDType);
  ApplicationIDType *encapBufHdr = (ApplicationIDType *)encapBuf;

  struct sockaddr_in sock;
  socklen_t socklen = sizeof(sock);
  int bufLen = Recvfrom2(recvFd, buf,
			 MAX_UDP_PKTSIZE - sizeof(ApplicationIDType), 0, 
			 (struct sockaddr *)&sock, &socklen);

  int pktSrcAddr = htonl(sock.sin_addr.s_addr);
  int pktSrcPort = htons(sock.sin_port);

  /* XXX: HACK: in windows, when vic is communicating to 127.0.0.1,
   * the pktSrcAddr perceived by the proxy is always loopback.  Here
   * I rewrite the pktSrcAddr
   */
  if (INADDR_LOOPBACK == pktSrcAddr) {
    pktSrcAddr = GetMyAddr();
  }

  if (bufLen < 0) {
    //printf("\nProxy: RecvFromApp: %s/%d (%d)",
    //   GetNameByAddr(pktSrcAddr), pktSrcPort, bufLen);
    return -1;
  }

  if (verbosity > 1) {
    printf("\nProxy: RecvFromApp: %s/%d (%d)",
	   GetNameByAddr(pktSrcAddr), pktSrcPort, bufLen);
  }

  if (bufLen == 0) {
    MyWarning("\nWRN: Proxy: RecvFromApp: %s/%d (%d)",
	      GetNameByAddr(pktSrcAddr), pktSrcPort, bufLen);
    return 0;
  }

  if (isMulticast) {
    /* multicast */

    // check loopback packets
    if (clientMgrPtr->IsLoopbackPacket(pktSrcAddr, pktSrcPort)) {
      if (verbosity > 1) {
	printf("\n Filtering self-packet off !!");
      }
      
      return 0;
    }

    // addr: mcast group addr
    clientMgrPtr->SendData(port, (const char *)buf, bufLen, addr);
  } else {

    clientMgrPtr->UpdateUnicastClientAndChannel(pktSrcAddr, port);

    clientMgrPtr->SendData(port, (const char *)buf, bufLen, pktSrcAddr);
  }

  // send to the overlay
  encapBufHdr->port = htons(port);
  encapBufHdr->srcAddr = htonl(pktSrcAddr);
  encapBufHdr->seqNum = htonl(pktSeqNum++);
  gossipHost->SendDataForApp((const char*)encapBuf, 
			     bufLen + sizeof(ApplicationIDType),
                             priority);
  
  /* yhchu: track bw */
  if(verbosity > -1) {
    if (bwRcvdFromApp == NULL) {
      bwRcvdFromApp = new EstimateBandwidth(10, 5);
    }

    struct EBReportList *reportList = bwRcvdFromApp->UpdateAndReport(bufLen);
    
    while (reportList != NULL) {
      printf("\n%ld:BwRcvdFromApp %d %d %s/%d", 
	     reportList->time, reportList->seq, reportList->byteCnt,
	     GetNameByAddr(addr), port);
      struct EBReportList *next = reportList->next;
      free(reportList);
      reportList = next;
    }
  }
  
  return bufLen;
}


int ListenPort::GetPort() {
  return port;
}
