#include <assert.h>
#include <stdlib.h>
#include <stdarg.h> /* va_list in FreeBSD and SUN */
#include <string.h>
#include <errno.h>
#include <strings.h>
#include <unistd.h>
#include <iostream.h>
#include <stdio.h>
#include <netdb.h>

#include <arpa/inet.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/resource.h>  /* SUN rlimit */
#include <netinet/in.h>	/* sockaddr_in{} and other Internet defns */

#include <Gossip/global.h>
#include <Gossip/gossipAgent.h>  // GOSSIP_PORT
#include <Gossip/query.h>

#include <Util/inetTypes.h>
#include <Util/inetMisc.h>
#include <Util/misc.h>
#include <Internet/inetGossipHost.h>
#include <Internet/dataMgr.h>
#include <Internet/dataMgrUdp.h>
#include <Internet/dataMgrTcp.h>
#include <Internet/dataMgrTfrc.h>
#include <Internet/correlatedCongestion.h>

/* specific to inetDriver */
#include <Internet/applicationAgent.h>

/* global in inetDriver */
InetGossipHost *gossipHost;
ApplicationAgent *applicationAgent;
DataMgr *dataMgrPtr;
CorrelatedCongestion *correlatedCongestionPtr;

const int INFINITE_TIME=100000;
int KILL_PORT=3333;
void Usage();

/* ETB declarations */
void ETBParse(char *optarg);
void ETBAction(float timeSinceStart);
int IsETBEnabled();

void OutOfMemory(){
  cerr << "\n" << getInetAddrName(GetMyAddr());
  MyError("Ran out of memory!");
}

Query *GetQueryAgent(){
  return(gossipHost->GetQueryAgent());
}

void MyError(const char *format, ...) {
  va_list ap;

  fprintf(stderr, "\nERR - existing: %s: ", getInetAddrName(GetMyAddr()));
  va_start(ap, format);
  vfprintf(stderr, format, ap);
  fflush(stderr);

  printf("\nERR - exiting: %s: ", getInetAddrName(GetMyAddr()));
  va_start(ap, format);
  vfprintf(stdout, format, ap);
  fflush(stdout);

  assert(0);
  exit(1);
}


void MyWarning(const char *format, ...) {
  va_list ap;

  fprintf(stderr, "\nWRN: %s: ", getInetAddrName(GetMyAddr()));
  va_start(ap, format);
  vfprintf(stderr, format, ap);
  fflush(stderr);

  printf("\nWRN: %s: ", getInetAddrName(GetMyAddr()));
  va_start(ap, format);
  vfprintf(stdout, format, ap);
  fflush(stdout);
}


void FuncSendToNetwork(int toAddr, int toPort, const char *buf, int bufLen,
		       PacketType packetType, int priority) {
  dataMgrPtr->SendToNetwork(toAddr, toPort, buf, bufLen, packetType, priority);

}


void FuncRecvDataForApp(const char *buf, int bufLen) {
  applicationAgent->Recv(buf, bufLen);
}



void DoExit() { 
  /* process about to exit, print status */
  printf("\nSTOP INTERNET RUN");
  if (applicationAgent != NULL) {
    applicationAgent->DoDebug();
  }
  
  /* XXX will cause some extra data to be printed */
  gossipHost->DoDebug();
  delete gossipHost;
  cout.flush();
  exit(0);
}


void InitiateLeave(int stayTimeOnLeave){
  gossipHost->LeaveGroup(stayTimeOnLeave);
}

int IsReadyToLeave(){
  return(gossipHost->IsReadyToLeave());
}

void SleepUntilJoinTime(int joinTimeInSec){
  if(joinTimeInSec > 0){
    fd_set fd;
    struct timeval tv;
    int retVal;
    
    cout << " Sleeping for " << joinTimeInSec << " seconds \n";
    FD_ZERO(&fd);
    tv.tv_sec = joinTimeInSec;
    tv.tv_usec = 0;
    
    retVal=select(0, NULL, NULL, NULL, &tv);
    if(retVal < 0){
      MyError("Select: Error");
    }
  }
}

/*
 * translate DNS names of active members to IP addresses
 * if a hostname cannot be resolved, it will be taken out from the member list
 *
 * returns the adjusted number of active members
 */
int CreateMemberAddrList(int numActiveMembers,
			 char **memberNameList,
			 int *memberAddrListPtr) {

  int cnt = 0;
  
  for (int i=0; i< numActiveMembers; i++) {
    int remote_addr;

    if ((remote_addr = getInetAddrByName(memberNameList[i])) == -1) {
      MyWarning("hostname %s not found", memberNameList[i]);
      continue;
    }

    memberAddrListPtr[cnt] = remote_addr;
    if(verbosity > 0){
      printf("\nKNOWN_MEMBER %s %d",getInetAddrName(remote_addr),remote_addr);
    }

    cnt++;
  }
  
  return cnt;
}

void CheckParamsSourceFlag(int sourceFlg,
			   int dataPeriod,
			   int dataSize){

  if(sourceFlg){
    assert(
	   (dataPeriod != 0) && 
	   (dataSize != 0) 
	   );
  }
  else{
    assert(
	   (dataPeriod == 0) && 
	   (dataSize == 0)
	   );
  }
}


void CreateNewDataMgr(char *dataMgrType, int wndsize){
  if (strcmp(dataMgrType, "udp") == 0) {
    dataMgrPtr = new DataMgrUdp(GOSSIP_PORT);
  } else if (strcmp(dataMgrType, "tcp") == 0) {
    dataMgrPtr = new DataMgrTcp(GOSSIP_PORT, wndsize);
  } else if (strcmp(dataMgrType, "tfrc") == 0) {
    dataMgrPtr = new DataMgrTfrc(GOSSIP_PORT);
  } else {
    printf("unknown transport option -T %s, expect -T {udp | tcp}\n\n", 
	   dataMgrType);
    Usage();
    exit(-1);
  }
}


void MiscInit(){
  /* test timezone */
  /* Also set random seed */
  struct timeval tv;
  struct timezone tz;
  
  
  if (verbosity > 1) {
    printf("\nMy host addr is %s, host name %s",
	   getInetAddrIP(GetMyAddr()), getInetAddrName(GetMyAddr()));
  }

  gettimeofday(&tv, &tz);
  srand(tv.tv_sec);

  if(verbosity > 0){
    printf("\nHOSTNAME %s", getInetAddrName(GetMyAddr()));
  }
}
  
void CheckParamsJoinLeaveTime(int joinTimeInSec,
			      int leaveTimeInSec){

  if(IsETBEnabled()){
    assert(joinTimeInSec == 0);
    assert(leaveTimeInSec == INFINITE_TIME);
  }

  if(joinTimeInSec < 0){
    MyError("JoinTime must be >= 0");
  }

  if(leaveTimeInSec < 0){
    MyError("LeaveTime must be >= 0");
  }
  
  if(joinTimeInSec > leaveTimeInSec){
    MyError("Join time must be smaller than leave time!!");
  }
}


void DoKill(int numActiveMembers, char **memberNameList) {
  struct sockaddr_in srcSock;
  const int on = 1;
  
  int net_fd = Socket(AF_INET, SOCK_DGRAM, 0);
  bzero((char *)&srcSock, sizeof(srcSock));
  srcSock.sin_family = AF_INET;
  srcSock.sin_addr.s_addr = htonl(INADDR_ANY);
  srcSock.sin_port = htons(KILL_PORT);
  Setsockopt(net_fd, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on));
  Bind(net_fd, (struct sockaddr *)&srcSock, sizeof(srcSock));
    
  /* send kill UDP datagram 10 times */
  for (int j=0; j<10; j++) {
    for (int i=0; i< numActiveMembers; i++) {
      struct sockaddr_in dstSock;
      char buf[] = "KILL";
      int remote_addr;
      
      if ((remote_addr = getInetAddrByName(memberNameList[i])) == -1) {
	printf("\nhostname %s not found", memberNameList[i]);
	MyError("\nhostname not found");
	continue;
      }
      
      dstSock.sin_family = AF_INET;
      dstSock.sin_addr.s_addr = htonl(remote_addr);
      dstSock.sin_port = htons(GOSSIP_PORT);
      
      sendto(net_fd, (const char *)&buf, sizeof(buf), 0, 
	     (struct sockaddr *)&dstSock, sizeof(dstSock));

      if (j == 0)
	printf("\nKill Narada daemon on host %s", memberNameList[i]);
    }
  }
  printf("\n");
}


void Usage() {
  fprintf(stderr, "-v [verbosity level], def=1\n");
  fprintf(stderr, "-N [host IP name or address] (if default to 127.0.0.1)\n");

  fprintf(stderr, "-V enable per-packet verbosity\n");
  fprintf(stderr, "-D use static propagation delay instead of dynamic latency\n");
  fprintf(stderr, "-M [Protocol Metric] def=BW_LATENCY\n");
  fprintf(stderr, "-B [Probe Method] def=RTT\n");
  fprintf(stderr, "-P [gossip_port], def=5566 \n");
  fprintf(stderr, "-r [max source rate in Kbps], def=1200Kbps \n");

  fprintf(stderr, "-m [active member IP addr]+ (may set multiple -m flags)\n");
  fprintf(stderr, "-l [lower degree bound], def=3\n");
  fprintf(stderr, "-u [upper degree bound], def=6\n");

  fprintf(stderr, "-x [wait time till sending 1st pkt], def=0\n");
  fprintf(stderr, "-p [data period in ms], def=0\n");
  fprintf(stderr, "-s [data size in bytes], def=0\n");
  fprintf(stderr, "-E [source step length in sec], def=120 (make source rate a step function)\n");

  fprintf(stderr, "-J [join time in sec], def=0\n");
  fprintf(stderr, "-L [join time in sec], def=INFINITE\n");
  fprintf(stderr, "-R [stay time on leave in sec], def=60 (collaborative leave mode)\n");
  

  fprintf(stderr, "-T [udp|tcp|tfrc] transport layer option for data forwarding, def=tcp\n");
  fprintf(stderr, "-w [TCP window size], def=65535 bytes\n");

  fprintf(stderr, "-K kill remote Narada daemons \n");
  fprintf(stderr, "-Q [what port kill message will be sent from], def=3333\n");
  fprintf(stderr, "-S [running time in minutes], def=20\n");

  fprintf(stderr, "-X [paramters for ETB]\n");

  fprintf(stderr, "-H [performance history input filename]\n");
  fprintf(stderr, "-W [performance history output filename]\n");

  fprintf(stderr, "-h print out help\n");

  fprintf(stderr, "\n");
  fprintf(stderr, "example: for a source of (1KByte * 10 pkt/sec = 80Kbps)\n");
  char *commonStr = "-S 5";
  fprintf(stderr, "(source) ./inet %s -p 100 -s 1024\n", commonStr);
  fprintf(stderr, "(receiver) ./inet %s -m openarch.cmcl.cs.cmu.edu\n", commonStr);
}


int main(int argc, char *argv[]) {

  /***** Setting up default values *******/

  /*** Individual Gossip Host Params *****/
  // argc is the upper bound on how many members in the argument list 
  char **memberNameList = new (char *)[argc];
  int numActiveMembers = 0;
  int lowerDegreeBound = 3;
  int upperDegreeBound = lowerDegreeBound * 2;
  
  /*** Parameters regarding source characteristics *******/
  int sourceFlg = 0;
  int startSendInSec = 0;  // when to start sending data 
  int dataPeriod = 0;
  int dataSize = 0;
  int stepFunctionSourceFlg=0; /** XXX Not debugged ***/
  int stepLengthInSec=120;


  /**** Parameters pertaining to dynamic membership ******/
  int joinTimeInSec=0;
  int leaveTimeInSec=INFINITE_TIME;
  int stayTimeOnLeaveInSec=60;

  /**** Parameters pertaining to Internet use ******/
  char *dataMgrType = "tcp";
  int wndsize = 65535;

  /**** Parameters pertaining to stopping Inetdriver ******/
  int stopTimeInMin = 20;  // default 20 min 
  int doKill = FALSE;

  if (argc <= 1) {
    fprintf(stderr, "If you want to invoke inet without arg, use ./inet -z\n");
    fprintf(stderr, "\n");
    Usage();
    exit(-1);
  }

  /* parse arguments */
  {
    char optstr[]="v:N:VDM:P:r:m:l:u:x:p:s:E:J:L:R:T:w:KQ:S:X:zhH:W:B:t:";
    char c;
    
    while((c=getopt(argc,argv,optstr)) != -1){
      switch(c){
	
	/******** Parameters for Util Library ***********/
      case 'v':
	verbosity=atoi(optarg);
	break;
	
      case 'N':
	MyHostName = optarg; 
	break;
	
	/*********  Parameters for Gossip Library *********/
	
      case 'V':
	perPacketVerbosity=1;
	break;
	
      case 'D':
	PROPAGATION_DELAY_FLAG=1;
	break;
	
      case 'M':
	ParseProtocolMetric(optarg);
	break;

      case 'B':
	ParseProbeMethod(optarg);
	break;

      case 'P':
	GOSSIP_PORT=atoi(optarg);
	break;
	
      case 'r':
	MAX_SOURCE_RATE=atoi(optarg);
	break;

	/***** Parameters to tune individual Gossip Host ******/

      case 'm':
	memberNameList[numActiveMembers] = optarg;
	numActiveMembers++;
	break;

      case 'l':
	lowerDegreeBound=atoi(optarg);
	break;

      case 'u':
	upperDegreeBound=atoi(optarg);
	break;

	/****** Parameters to control source characteristics ******/

      case 'x':
	startSendInSec = atoi(optarg);
	break;

      case 'p':
	dataPeriod=atoi(optarg);
	sourceFlg=1;
	break;

      case 's':
	dataSize=atoi(optarg);
	sourceFlg=1;
	break;

      case 'E':
	stepFunctionSourceFlg=1;
	stepLengthInSec=atoi(optarg);
	break;

	/******* Parameters to control dynamic join/leave *******/
	
      case 'J':
	joinTimeInSec=atoi(optarg);
	break;
      case 'L':
	leaveTimeInSec=atoi(optarg);
	break;
      case 'R':
	stayTimeOnLeaveInSec=atoi(optarg);
	break;

	/******* Parameters to configure Internet usage *****/
      case 'T':
	dataMgrType = optarg;
	break;
	
      case 'w':  /* update wndsize */
	wndsize = atoi(optarg); 
	break;

	/****** Parameters for InetDriver ******/
      case 'K':
	doKill = TRUE;
	break;
	
      case 'Q':
	KILL_PORT=atoi(optarg);
	break;

      case 'S':
	stopTimeInMin=atoi(optarg); /* min->ms */
	break;


	/****** Parameter for ETB test ******/
	
      case 'X': 
	ETBParse(optarg);
	break;
	
	
	/****** Parameters for usage *********/
      case 'z':
	break;
      case 'h':
	Usage();
	exit(-1);

	/****** Performance History *******/
      case 'H': 
	PerformanceHistoryReadFile = optarg;
	break;
      case 'W':
	PerformanceHistoryWriteFile = optarg;
	break;

	/****** Time Sync ********/
      case 't':
	{
	  char timeSyncServerStr[100];
	  strcpy(timeSyncServerStr,optarg);
	  timeSyncServerAddr=getInetAddrByName(timeSyncServerStr);
	}
	break;
		
      default:
	Usage();
	MyError("Invalid Option!!");
	exit(-1);
      }
    }
  }

  
  CheckParamsJoinLeaveTime(joinTimeInSec,leaveTimeInSec);
  CheckParamsSourceFlag(sourceFlg,dataPeriod,dataSize);

  /* start clock tick */
  SetRealCurrTime();
  long startTimeInMs = GetRealCurrTime();
  
  if (doKill) {
    DoKill(numActiveMembers, memberNameList);
    DoExit();
  }
  
  MiscInit(); //initializes random seed, checks time zone, some printfs
  CreateNewDataMgr(dataMgrType,wndsize); // creates appropriate dataMgr

  /* start a gossipHost */
  gossipHost = new InetGossipHost(GetMyAddr(), 
				  FuncSendToNetwork,
				  FuncRecvDataForApp);

  correlatedCongestionPtr = new CorrelatedCongestion(GOSSIP_PORT+2);



  {
    /*** JOIN: Sequence of actions taken till member joins.******/


    SleepUntilJoinTime(joinTimeInSec);
    
    int *memberAddrListPtr = new int[numActiveMembers];
    numActiveMembers = CreateMemberAddrList(numActiveMembers,
					    memberNameList,
					    memberAddrListPtr);

    gossipHost->JoinGroup(numActiveMembers, 
			  memberAddrListPtr,
			  lowerDegreeBound, 
			  upperDegreeBound,
			  sourceFlg);
    applicationAgent = new ApplicationAgent(gossipHost, sourceFlg, 
					    dataPeriod, dataSize,
					    stepFunctionSourceFlg,
					    stepLengthInSec * 1000);

  }
  
  int hasStartSend = FALSE;
  int fd_wid = GetFDWID();
  int leaveMode=0;
  fd_set rs, ws;
  for (;;) {

    float timeSinceStart = (GetRealCurrTime() - startTimeInMs)/1000.0;
    
    /***********
	      Terminate if time is up, 
    ***************/
    if (timeSinceStart > (stopTimeInMin * 60)) {
      DoExit();
    }

    /***********
		Create application agent, and start transmission as required
    *************/

    if (sourceFlg && 
	(timeSinceStart > startSendInSec) && 
	(applicationAgent != NULL) && 
	(! hasStartSend)
	) {
      applicationAgent->StartSend();
      hasStartSend = TRUE;
    }
    
    /**************
		NOTE: Currently two ways for leave are supported.
		Either ETB, or join/leave. So, there is some redundancy
		that can be cleaned up in the future.
    ****************/

    if ((!leaveMode) && (timeSinceStart > leaveTimeInSec)) {
      gossipHost->LeaveGroup(stayTimeOnLeaveInSec * 1000);
      leaveMode=1;

      cout << "\n" << GetRealCurrTime() << ":LEAVE " 
	   << GetNameByAddr(GetMyAddr()) << " COOPERATIVE IN "
	   << stayTimeOnLeaveInSec;
    }
    
    if(leaveMode && (gossipHost->IsReadyToLeave())){
      cout << "\n" << GetRealCurrTime() << ":LEAVE " 
	   << GetNameByAddr(GetMyAddr()) << " COOPERATIVE NOW "
	   << stayTimeOnLeaveInSec;
      break; 
    }
    

    /*********
	   Do complicated ETB stuff
    **********/

    ETBAction(timeSinceStart);

    /****************
	    The core action:
	    Set all descriptors that need to be waited on
	    and find the next timer that goes off.
    ******************/



    FD_ZERO(&rs); 
    FD_ZERO(&ws);

    /**** pop the next event from the queue, if needed */
    long wait = gossipHost->PopNextEvent();
    wait = MIN(wait, dataMgrPtr->SetFD(&rs, &ws));
    wait = MIN(wait, correlatedCongestionPtr->SetFD(&rs, &ws));

    struct timeval wait_time = MsecToTimeval(wait);

    /* set wait_time to NULL if wait indefinitely */
    if (select(fd_wid, &rs, &ws, (fd_set *) NULL, &wait_time) < 0) {
      switch(errno) {
      case EINTR:
	break;  
      default:
	perror("select");
	assert(0);
      }
    }
    
    correlatedCongestionPtr->ReadFromNetwork(&rs, &ws);

    
    while (TRUE) {
      int fromAddr, fromPort, bufLen;
      char buf[MAX_PAYLOAD_SIZE];
      
      if ((bufLen=dataMgrPtr->ReadFromNetwork(&rs, &ws, buf, MAX_PAYLOAD_SIZE, 
					      &fromAddr, &fromPort)) < 0) {
	break;
      }

      /* XXX a hack to intercept and process the KILL message */
      if (fromPort == KILL_PORT) {
	printf("\n Stopping because of Kill signal");
	fprintf(stderr, "\nWRN: %s stopped because of Kill signal",
		getInetAddrName(GetMyAddr()));
	fflush(stdout);
        DoExit();
      }
      if(verbosity > 1){
	printf("\nNetwork: read %d bytes from %s/%d network",
	       bufLen, getInetAddrName(fromAddr), fromPort);
      }
      
      gossipHost->RecvFromNetwork(buf, bufLen, fromAddr, fromPort);
    }
  }
  cerr << "CAME HERE!!!!!!";
  DoExit();
}
 
