#include <stdio.h>
#include <assert.h>
#include <iostream.h>
#include <iomanip.h> 
#include <stdlib.h>
#include <math.h>
#include <sys/time.h>
#include <unistd.h>

#include "vrtEntry.h"
#include "sourceEntry.h"
#include "childList.h"
#include "physicalDelay.h"
#include "physicalBW.h"
#include "global.h"
#include "gossipPayLoad.h"
#include "METRICS/routingMetric.h"
#include "METRICS/bwLatencyMetric.h"
#include "METRICS/latencyMetric.h"
#include "METRICS/bwMetric.h"
#include "path.h"


VRTEntryType::VRTEntryType(int addrIn, 
			   int myAddrFlg
			   ){
  
  Init(addrIn);

  if(myAddrFlg){

    /**********
      If its an entry for myself, the delay to me 
      is 0, and the bandwidth is great
    ***********/

    routingMetric->SetBest();
    linkMetric->SetBest();
    path->SetZeroPath();
  }
}
  
  
VRTEntryType::VRTEntryType(int addrIn, 
			   int lastControlSeqNumIn, 
			   long timeIn
			   ){    
  Init(addrIn);
  lastControlSeqNum=lastControlSeqNumIn;
  time=timeIn;
}



void VRTEntryType::Init(int addrIn){
  routingMetric=CreateNewRoutingMetric();
  linkMetric=CreateNewRoutingMetric();
  physicalDelay = new PhysicalDelay();
  physicalBW = new PhysicalBW();
  path=new MyPath();

  addr=addrIn;

  lastControlSeqNum=NO_CONTROL_INFORMATION;
  time=NO_CONTROL_INFORMATION;

  numRTProbesSent=0;
  numRTProbesRcvd=0;
  lastRTProbeSentTime=-1;
  lastRTProbeRcvdTime=-1;
  lastRTProbe=NULL;
  
  deadFlg=0;
  childListPtr=new ChildList(addr);
  
  sourceEntryPtr=NULL;

  /**********
   Initialize Metrics. 
    As we don't know delay and bandwidth of the physical link,
   initialize these to very bad - high delay, low bandwidth.
   This ensures that in early situations before information
   of physical links is available routing can still be done.
  ************/

  routingMetric->SetNoRoute();
  linkMetric->PhysicalToRouting(this);

  bwTestFailureFlag=0;
  timeOfLastBWTestFailure=-1;
  
} 

VRTEntryType::~VRTEntryType(){
  delete childListPtr;
  delete routingMetric;
  delete linkMetric;
  delete physicalDelay;
  delete physicalBW;
  delete path;

  if(sourceEntryPtr != NULL){
    delete sourceEntryPtr;
    sourceEntryPtr=NULL;
  }
  if(lastRTProbe != NULL){
    delete lastRTProbe;
  }

}


/*********
   Bunch of asserts that ensures we are in one of the following states:
     {<rm=No_Route,cl=null,pathLen=NO_ROUTE>, 
      <rm=TR, rb=TR;, pathLen != NO_ROUTE > and
      <rm = Legal; pathLen != NO_ROUTE>}

   Let metricFromCandidateToAddr = mc
       pathLenFromCandidate =plC,

   Also, bunch of asserts to ensure that:
   {<mc=NO_ROUTE,  plC = NO_ROUTE>,
    <mc =TR, plC != NO_ROUTE>,
    <mc - Legal, plC != NO_ROUTE>
    }

    Finally asserts to ensure metricToCandidate is not NO_ROUTE!
****************/

int VRTEntryType::SanityChecks(RoutingMetric *metricToCandidate,
			       RoutingMetric *metricFromCandidateToAddr,
			       MyPath *pathFromCandidateToAddr
			       ){

  int failedAssert=0;

  if(
     !(
       (
	(routingMetric->IsNoRoute()) && 
	(childListPtr->GetNumChildren() == 0) &&
	(path->IsNoRoute()) 
	)
       ||
       (
	(routingMetric->IsTempRoute()) &&
	(!path->IsNoRoute()) 
	)
       ||
       (
	(routingMetric->IsValidRoute()) &&
	(!path->IsNoRoute()) 
	)
       )
     ){
    failedAssert=1;
  }
  
  if(
     !(
       (
	(metricFromCandidateToAddr->IsNoRoute()) &&
	(pathFromCandidateToAddr->IsNoRoute())
	)
       
       ||
       
       (
	(metricFromCandidateToAddr->IsTempRoute()) &&
	(!pathFromCandidateToAddr->IsNoRoute())
	)
       
       ||
       
       (
	(metricFromCandidateToAddr->IsValidRoute()) &&
	(!pathFromCandidateToAddr->IsNoRoute())
	)
       )
     ){
    failedAssert=2;
  }

  if(
     !(
       (metricToCandidate->IsValidRoute()) ||
       (metricToCandidate->IsTempRoute())
       )
     ){
    failedAssert=3;
     
  }

  return(failedAssert);
}


int VRTEntryType::UpdateRoutingInformation(
				   int candidateNextHop,
				   int tempNbrFlag,
				   RoutingMetric *metricToCandidate,
				   RoutingMetric *metricFromCandidateToAddr,
				   int myAddr,
				   MyPath *pathFromCandidateToAddr
				   ){	 

  int oldNH;
  int oldNHStatus=GetNextHop(&oldNH);
  RoutingMetric *oldRoutingMetric=routingMetric->Duplicate();

  int returnStatus=UpdateRoutingInformationActual(
					  candidateNextHop,
					  tempNbrFlag,
					  metricToCandidate,
					  metricFromCandidateToAddr,
					  myAddr,
					  pathFromCandidateToAddr
					  );

  int newNH;
  int newNHStatus=GetNextHop(&newNH);

  int changeFlag=1;

  if(oldNHStatus == newNHStatus){
    if(newNHStatus != 0){
      changeFlag=0;
    }
    else if(newNHStatus == 0){
      if(
	 (oldNH == newNH) &&
	 (routingMetric->Compare(oldRoutingMetric) == 0)
	 ){
	changeFlag=0;
      }
    }
  }

  if(changeFlag){
    PrintRouteChange();
  }
  
  delete oldRoutingMetric;
  return(returnStatus);
}

int VRTEntryType::UpdateRoutingInformationActual(
			 int candidateNextHop,
			 int tempNbrFlag,
			 RoutingMetric *metricToCandidate,
			 RoutingMetric *metricFromCandidateToAddr,
			 int myAddr,
			 MyPath *pathFromCandidateToAddr
			 ){
  
  int sanityCheckCode= SanityChecks(metricToCandidate,
				    metricFromCandidateToAddr,
				    pathFromCandidateToAddr);

  if(sanityCheckCode > 0){
    cout << "Sanity Checks:: Failed assert with code "
	 << sanityCheckCode
	 << " update from "
	 << GetNameByAddr(candidateNextHop);
    MyError("Sanity Check Failure");
  }
      
  
  assert(myAddr != addr);

  int IAmInCandidatePathFlg=pathFromCandidateToAddr->IsInPath(myAddr);
  int candidateIsInMyPathFlg=path->IsInPath(candidateNextHop);
  
  if (
      candidateIsInMyPathFlg &&
      IAmInCandidatePathFlg
      ){
    
    // Transient routing loop!! Remove entry for this guy.
    int nextHop;
    GetNextHop(&nextHop);
    int ret=SetNoRouteEntry(nextHop);
    assert(ret == 1);

    // Send him my control update so he can fix his table quickly.
    return(1);
  }
  
  int candidateUsesMeAsNextHopFlg= 
    (pathFromCandidateToAddr->IsNextHop(myAddr));
  
  if(candidateUsesMeAsNextHopFlg){
    assert(!IsNextHop(candidateNextHop)); // we just checked for loops
    
    if(!(routingMetric->IsNoRoute())){ 
      childListPtr->Insert(candidateNextHop); // Insert as a child.
      return(0);
    }
    else{
      // Candidate uses me as next hop when I have no route!
      // Send him my control update so he can fix his table.
      return(1);
    }
  }
  
  
  // my nbr does not use me as next hop     
  assert(!candidateUsesMeAsNextHopFlg);  
  childListPtr->Delete(candidateNextHop);
  
  if(tempNbrFlag){
    metricToCandidate->SetTempRoute();
  }
  
  if(routingMetric->IsNoRoute()){
    if(metricFromCandidateToAddr->IsNoRoute()){
      return(0);
    }
    else{
      RoutingMetric *metricViaCandidate=
	metricFromCandidateToAddr->Merge(metricToCandidate);
      
      if(!IAmInCandidatePathFlg){
	UpdateRouteEntry(candidateNextHop, 
			 metricViaCandidate,
			 pathFromCandidateToAddr
			 );
      }
      delete metricViaCandidate;
      // Send triggered update as I now have a route, 
      // when I previously did not
      //return(2); // Sanjay:Sigcomm2002
      
      return(1);
    }
  }
  else{
    // routingMetric is TR or Regular.
    if(metricFromCandidateToAddr->IsNoRoute()){
      // Set no route if candidate is already my next hop
      if(SetNoRouteEntry(candidateNextHop)){
	//My route is affected, as this guy is my next hop
	// return(2); Sanjay:Sigcomm2002
	return(0);
      }
      else{
	// My route is not affected as this guy not my next hop
	return(0);
      }
    }
    else { 
      int changedParent=0;
      
      RoutingMetric *metricViaCandidate=
	metricFromCandidateToAddr->Merge(metricToCandidate);
      
      if(!IsNextHop(candidateNextHop)){
	int isImproved=routingMetric->Compare(metricViaCandidate);
	
	if(isImproved > 0){
	  if(!IAmInCandidatePathFlg){
	    UpdateRouteEntry(candidateNextHop, 
			     metricViaCandidate, 
			     pathFromCandidateToAddr
			     );
	    changedParent=1;
	  }
	}
      }
      
      else{
	// Already my next hop. Must update no matter what!
	assert(!IAmInCandidatePathFlg); // we have already checked for it
	UpdateRouteEntry(candidateNextHop, 
			 metricViaCandidate,
			 pathFromCandidateToAddr
			 );
      }
      delete metricViaCandidate;
      return(changedParent);
    }
  }
  assert(0);
  return(0);
}

int VRTEntryType::IsValidRoute(){
  if(routingMetric->IsValidRoute()){
    return(1);
  }
  else{
    return(0);
  }
}

int VRTEntryType::IsNoRoute(){
  if(routingMetric->IsNoRoute()){
    return(1);
  }
  else{
    return(0);
  }
}

void VRTEntryType::UpdateSelfRoutingInformation(int nbr,
						int nbrUsesMeAsNextHopFlg){
  if(nbrUsesMeAsNextHopFlg)
    childListPtr->Insert(nbr);
  else
    childListPtr->Delete(nbr);
}


int VRTEntryType::UpdateOnNbrDeletion(int nbrDeleted){
  if(IsNextHop(nbrDeleted)){
    int ret=SetNoRouteEntry(nbrDeleted);
    assert(ret == 1);
    PrintRouteChange();
    return(1);
  }
  else{
    childListPtr->Delete(nbrDeleted);
    return(0);
  }
}

int VRTEntryType::UpdateOnNbrTemp(int nbrBecomeTemp){
  if(IsNextHop(nbrBecomeTemp)){
    if(IsValidRoute()){
      routingMetric->SetTempRoute();
      PrintRouteChange();
    }
    return(1);
  }
  else{
    return(0);
  }
}


void VRTEntryType::UpdateOnControlTimeout(){
  if(!IsNoRoute()){
    routingMetric->SetNoRoute();
    path->SetNoRoute();
    childListPtr->Destroy();
    PrintRouteChange();
  }
}


int VRTEntryType::SetNoRouteEntry(int candidateNextHop){
  // Is this guy already my next hop?

  if (IsNextHop(candidateNextHop)) {
    routingMetric->SetNoRoute();
    path->SetNoRoute();
    childListPtr->Destroy();  
    return(1);
  }
  else{
    return(0);
  }
}

/*****************
 Following function called only for an address that is not my own.
 It assumes all required checks have already been conducted
******************/
void VRTEntryType::UpdateRouteEntry(int candidateNextHop,
				    RoutingMetric *metricViaCandidate,
				    MyPath *pathViaCandidate
				    ){
  routingMetric->SetMetric(metricViaCandidate);
  path->Merge(candidateNextHop,pathViaCandidate);
}



/*******
 Functions pertaining to dead/live status
*********/
int VRTEntryType::IsDead(){
  if(deadFlg) return 1;
  else return 0;
}

void VRTEntryType::MarkDead(){
  if(GetCurrTime() < 100000){
    assert(0);
  }
  deadFlg=1;
}

void VRTEntryType::MarkAlive(){
  assert(0);
  deadFlg=0;
}

/********************
 Functions pertaining to control information manipulation
*********************/
int VRTEntryType::UpdateControlInformation(int lastControlSeqNumUpdate,
					   long timeOfUpdate){
  if(
     (lastControlSeqNumUpdate != NO_CONTROL_INFORMATION) &&
     (
      (lastControlSeqNum == NO_CONTROL_INFORMATION) || 
      (lastControlSeqNum < lastControlSeqNumUpdate)
      )
     ){
    lastControlSeqNum=lastControlSeqNumUpdate;
    time=timeOfUpdate;
    return(1);
  }
  return(0);
}


void VRTEntryType::IncrementMyControlSeqNum(){
  
  if(lastControlSeqNum == NO_CONTROL_INFORMATION){
    struct timeval tv;
    gettimeofday(&tv,NULL);
    lastControlSeqNum=tv.tv_sec;
    assert(lastControlSeqNum > 0);
  }
  else{
    lastControlSeqNum++;
  }
  
  time=GetCurrTime();
}

  
int VRTEntryType::GetLastControlSeqNum(){
  return(lastControlSeqNum);
}


long VRTEntryType::GetTimeOfLastUpdate(){
  return(time);
}

/*************
 Miscellaneous functions
***************/

RoutingMetric *VRTEntryType::GetRoutingMetric(){
  return(routingMetric->Duplicate());
}

RoutingMetric *VRTEntryType::GetLinkMetric(){
  return(linkMetric->Duplicate());
} 


int VRTEntryType::GetAddr(){
  return(addr);
}


int VRTEntryType::IsNextHop(int addr){
  return(path->IsNextHop(addr));
}

int VRTEntryType::GetNextHop(int *nextHop){
  return(path->GetNextHop(nextHop));
}

/*********** 
     The next few functions are delegated to the PhysicalDelay class
****************/

int VRTEntryType::GetPhysicalDelay(){
  if(PROPAGATION_DELAY_FLAG){
    return(physicalDelay->GetPropagationDelay());
  }
  else{
    return(physicalDelay->GetDynamicLatency());
  }
}

long VRTEntryType::GetTimeSinceLastDelayProbe(){
  return(physicalDelay->GetTimeSinceLastDelayProbe());
}

int VRTEntryType::GetNumDelayProbes(){
  return(physicalDelay->GetNumDelayProbes());
}

void VRTEntryType::IntegrateDelayResult(int roundTripDelay){

  //assert(roundTripDelay >= 0);
  if (roundTripDelay < 0) {
    MyWarning("vrtEntry: IntegrateDelayResult to %s is negative %d",
	      GetNameByAddr(GetAddr()), roundTripDelay);
    return;
  }
 
  physicalDelay->IntegrateDelayResult(roundTripDelay);
  linkMetric->PhysicalToRouting(this);

  if(IsNextHop(addr)){
    //This guy must be a neighbor; We might want to recheck this later
    if(routingMetric->IsValidRoute()){
      routingMetric->SetMetric(linkMetric);
    }
  }
  
  if(
     (verbosity > 1) ||
     ((verbosity > 0) && (GetNumDelayProbes() % 10 == 0)) ||
     ((verbosity > 0) && (GetNumDelayProbes() < 10))
     ){
    
    cout.setf(ios::fixed,ios::floatfield);

    cout << "\n" 
	 << GetCurrTime() 
	 << ":New physical link estimate "
	 << "(D:" << roundTripDelay/2 << ") to " 
	 << GetNameByAddr(addr) 
	 << " is delay:"  << setprecision(2) 
	 << physicalDelay->GetSmoothLatency() << ":"
	 << physicalDelay->GetDynamicLatency() << ":"
	 << physicalDelay->GetPropagationDelay() << " ";

    linkMetric->Print();
    physicalDelay->Print();
  }
}



/*******************
 Functions dealing with physicalBW
*********************/

int VRTEntryType::GetPhysicalBW(){
  return(physicalBW->GetBW());
}

long VRTEntryType::GetTimeSinceLastBWProbe(){
  return(physicalBW->GetTimeSinceLastBWProbe());
}

int VRTEntryType::GetNumBWProbes(){
  return(physicalBW->GetNumBWProbes());
}

int VRTEntryType::GetNumBWResultProbes(){
  return(physicalBW->GetNumBWResultProbes());
}

void VRTEntryType::IntegrateBWResult(long estBW, BWResultType type){
  
  physicalBW->IntegrateBWResult(estBW,type);
  linkMetric->PhysicalToRouting(this);
    
  if(IsNextHop(addr)){
    if(routingMetric->IsValidRoute()){
      routingMetric->SetMetric(linkMetric);
    }
  }
    
  if(verbosity > 0){
    cout.setf(ios::fixed,ios::floatfield);
    cout << "\n" 
	 << GetCurrTime()
	 << ":New physical link estimate (B:"
	 << estBW << ":" << (int) type
	 << ") to " 
	 << GetNameByAddr(addr) 
	 << " is smoothBW:" << GetPhysicalBW() << " ";
    linkMetric->Print();
  }
  return;
}
     



/***********
 Functions pertaining to childList
************/
int VRTEntryType::GetNumChildren(){
  return(childListPtr->GetNumChildren());
}

int VRTEntryType::IsInChildList(int addr){
  return(childListPtr->IsInChildList(addr));
}

void VRTEntryType::GetChildList(int *childAddrList){
  childListPtr->GetChildList(childAddrList);
}


/**************
Functions pertaining to sourceEntry
******************/

int VRTEntryType::IsSource() {
  return (sourceEntryPtr != NULL);
}

void VRTEntryType::SetSource() {
  assert(sourceEntryPtr == NULL);
  sourceEntryPtr = new SourceEntry(addr);
}

SourceEntry *VRTEntryType::GetSourceEntry() {
  assert(sourceEntryPtr != NULL);
  return sourceEntryPtr;
}


/***********
 Functions pertaining to path to destination
************/
MyPath *VRTEntryType::GetPath(){
  return(path->Duplicate());
}


/*************
 Functions pertaining to Routing Table Probes
**************/
int VRTEntryType::GetTimeSinceLastRTProbeSent(){
  if(numRTProbesSent == 0){
    return(-1);
  }
  else{
    return(GetCurrTime() - lastRTProbeSentTime);
  }
}

int VRTEntryType::GetTimeSinceLastRTProbeRcvd(){
  if(numRTProbesRcvd == 0){
    return(-1);
  }
  else{
    return(GetCurrTime() - lastRTProbeRcvdTime);
  }
}

int VRTEntryType::GetNumRTProbesSent(){
  return(numRTProbesSent);
}

int VRTEntryType::GetNumRTProbesRcvd(){
  return(numRTProbesRcvd);
}


void VRTEntryType::UpdateOnRTProbeSent(){
  lastRTProbeSentTime=GetCurrTime();
  numRTProbesSent++;  
}

void VRTEntryType::UpdateOnRTProbeRcvd(ProbeResponseMsg *msgPtr){
  if(lastRTProbe != NULL){
    delete lastRTProbe;
  }
  lastRTProbe=msgPtr->Duplicate();
  lastRTProbeRcvdTime=GetCurrTime();
  numRTProbesRcvd++;
}

ProbeResponseMsg *VRTEntryType::GetLastRTProbe(){
  return(lastRTProbe);
}

void VRTEntryType::SetBWTestFailure(){
  bwTestFailureFlag=1;
  timeOfLastBWTestFailure=GetCurrTime();
}

int VRTEntryType::GetTimeSinceLastBWTestFailure(){
  if(!bwTestFailureFlag){
    return(-1);
  }
  else{
    return(GetCurrTime() - timeOfLastBWTestFailure);
  }
}

/*******************
void VRTEntryType::PrettyPrint(){

  cout << "\n Addr: " << addr << " Name: " << GetNameByAddr(addr);
  cout << "\n LCSN = " << lastControlSeqNum << " Time = " << time
       << " DeadFlg = " << deadFlg;
  cout << "\n NumRTProbes = " << numRTProbes 
       << " lastRTProbeTime = " << lastRTProbeTime;

  physicalDelay->Print();
  physicalBW->Print();

  cout << "\n Link Metric: ";
  linkMetric->Print();

  cout << "\n Routing Metric: ";
  routingMetric->Print();

  path->Print();
  childListPtr->Print();
  
}
*************************/

void VRTEntryType::Print(){
  cout.setf(ios::fixed,ios::floatfield);

  int nextHop=-1;
  GetNextHop(&nextHop);

  int routingDelay=0;
  int routingBW=0;
  int physicalBWAdvertised=0;

  switch(protocolMetric){
  case STATIC:
    break;
  case BW_ONLY:
    routingBW=((BWMetric *)routingMetric)->GetBW();
    physicalBWAdvertised=((BWMetric *)linkMetric)->GetBW();
    break;
  case LATENCY_ONLY:
    routingDelay=((LatencyMetric *)routingMetric)->GetLatency();
    break;
  case BW_LATENCY:
    routingDelay=((BWLatencyMetric *)routingMetric)->GetLatency();
    routingBW=((BWLatencyMetric *)routingMetric)->GetBW();
    physicalBWAdvertised=((BWLatencyMetric *)linkMetric)->GetBW();
    break;
  }

  cout << "\n Addr: " << addr << " Name " 
       << GetNameByAddr(addr) 
       << " LCSN = " << lastControlSeqNum << " Time = " << time 
       << " PDR = "  << setprecision(2) 
       << physicalDelay->GetSmoothLatency() 
       << " PD = "   << GetPhysicalDelay() << " RD = " << routingDelay 
       << " PBR = "  << GetPhysicalBW() << " PBA = " << physicalBWAdvertised
       << " RB = " << routingBW << " NH = " 
       << GetNameByAddr(nextHop) 
       << " Num Delay Probes = " << GetNumDelayProbes() 
       << " LDProbeTime = " << GetTimeSinceLastDelayProbe()
       << " Num BW Probes = " << GetNumBWProbes()
       << " LBWProbeTime = " << GetTimeSinceLastBWProbe()
       << " numChildren = " << GetNumChildren() << " DeadFlg = " << deadFlg;

  
  path->Print();
  childListPtr->Print();

}


void VRTEntryType::PrintRouteChange(){
  
  if(verbosity < 1){
    return;
  }

  int routingDelay=0;
  int routingBW=0;
  
  switch(protocolMetric){
  case STATIC:
    break;
  case BW_ONLY:
    routingBW=((BWMetric *)routingMetric)->GetBW();
    break;
  case LATENCY_ONLY:
    routingDelay=((LatencyMetric *)routingMetric)->GetLatency();
    break;
  case BW_LATENCY:
    routingDelay=((BWLatencyMetric *)routingMetric)->GetLatency();
    routingBW=((BWLatencyMetric *)routingMetric)->GetBW();
    break;
  }

  int nextHop=-1;
  GetNextHop(&nextHop);
  
  cout << "\n" << GetCurrTime() << ":"
       << "VRTEntryChange "
       << GetNameByAddr(addr)
       << " RD = " << routingDelay 
       << " RB = " << routingBW 
       << " NH = " << GetNameByAddr(nextHop);
}
