#include <iostream.h>
#include <iomanip.h>
#include <assert.h>

#include "physicalDelay.h"
#include "global.h"

PhysicalDelay::PhysicalDelay(){
  numDelayProbes=0;
  lastDelayProbeTime=-1;

  latencySmooth=-1;
  latencyMetric=-1;
  propagationDelay=-1;
}


int PhysicalDelay::GetTimeSinceLastDelayProbe(){
  if(numDelayProbes == 0){
    return(-1);
  }
  else{
    return(GetCurrTime() - lastDelayProbeTime);
  }
}

int PhysicalDelay::GetNumDelayProbes(){
  return(numDelayProbes);
}

int PhysicalDelay::GetDynamicLatency(){
  if(numDelayProbes == 0){
    return(-1);
  }
  else{
    return(latencyMetric);
  }
}

float PhysicalDelay::GetSmoothLatency(){
  if(numDelayProbes == 0){
    return(-1);
  }
  else{
    return(latencySmooth);
  }
}

int PhysicalDelay::GetPropagationDelay(){
  if(numDelayProbes == 0){
    return(-1);
  }
  else{
    return(propagationDelay);
  }
}

void PhysicalDelay::IntegrateDelayResult(int roundTripDelay){

  //assert(roundTripDelay >= 0);
  if (roundTripDelay < 0) {
    MyWarning("physicalDelay: IntegrateDelayResult: rtt is negative %d",
	      roundTripDelay);
    return;
  }

  int oneWayDelay=roundTripDelay/2;
  
  UpdatePropagationDelay(oneWayDelay);
  UpdateDynamicLatency(oneWayDelay);

  numDelayProbes++;
  lastDelayProbeTime=GetCurrTime();

}

void PhysicalDelay::UpdatePropagationDelay(int oneWayDelay){
  if(numDelayProbes == 0){
    propagationDelay=oneWayDelay;
  }
  else{
    if(propagationDelay > oneWayDelay){
      propagationDelay=oneWayDelay;
    }
  }
}

/***************
 UpdateDynamicLatency():

 Pretty hackish right now. Expect this to change a bit in the future.

 Key Idea:
 -Maintain an exponentially smoothed estimate, latencySmooth.
 -Maintain a latencyMetric that tracks latencySmooth, but is
  more stable. Do not update latencyMetric, unless latencySmooth
  changes significantly from itself.

  (might consider moving part of this code to the LatencyMetric part).
******************/ 

void PhysicalDelay::UpdateDynamicLatency(int oneWayDelay){
  
  if(numDelayProbes == 0){
    latencySmooth=oneWayDelay;
    latencyMetric=(int) latencySmooth;
    
    return;
  }
  

  if(numDelayProbes <= NUM_DELAY_WARMUP_ESTIMATES){
    if (latencySmooth > oneWayDelay){
      latencySmooth=oneWayDelay;
      latencyMetric= (int) latencySmooth;
    }
    return;
  }
  
  
  float latencyTemp;

  latencyTemp =  ALPHA_DELAY * oneWayDelay +
                 (1 - ALPHA_DELAY) * latencySmooth;
    
  // Don't be too sensitive to one large delay value
  if((latencyTemp - latencySmooth) > 1.0){
    latencySmooth=latencySmooth+1;
  }
  else{
    latencySmooth=latencyTemp;
  }
      
      
  float diff=latencyMetric - latencySmooth;
      
  if(diff < 0){
    diff *= -1;
  }

  /*********
    The code below is pretty ad-hoc, and there's lots of scope
    for improving the heuristics
  **********/

  float relDiff;

  if(latencySmooth == 0){
    relDiff=diff;
  }
  else{
    relDiff=diff/latencySmooth;
  }
  
  
  int absoluteDelayDiff;
  if(IsLocalCongestionSignal()){
    absoluteDelayDiff=ABSOLUTE_DELAY_DIFF_LC;
  }
  else{
    absoluteDelayDiff=ABSOLUTE_DELAY_DIFF;
  }

  if(
     (diff > absoluteDelayDiff) &&
     (relDiff > RELATIVE_DELAY_DIFF)
     ){
    latencyMetric = (int)latencySmooth;
  }
    
    /******* 
	     As above heuristic may introduce lasting
	     inaccuracy if delay between people on a
	     LAN had gone up in the past.
    ********/
  
  if(
     (latencySmooth < 5) &&
     (latencyMetric > 20)
     ){
    latencyMetric = (int)latencySmooth;
  }
  
}


void PhysicalDelay::Print(){

  cout.setf(ios::fixed,ios::floatfield);

  cout << "\n Physical Delay:";

  cout << " Smooth: " << setprecision(2) << latencySmooth;
  cout << " Metric: " << latencyMetric
       << " Prop: " << propagationDelay
       << " NumProbes = " << numDelayProbes 
       << " LDProbeTime = " << lastDelayProbeTime ;

}
