/**************
perfMetric.cc:

This file lists functions of PerformanceAgent that depend on the
metric that is being optimized for.
***************/

#include <stdio.h>
#include <stdlib.h>
#include <iostream.h>
#include <assert.h>

#include "connectivityAgent.h"
#include "pokeMgr.h"
#include "vrt.h"
#include "vrtEntry.h"
#include "neighborTable.h"
#include "performanceAgent.h"
#include "METRICS/routingMetric.h"
#include "METRICS/bwLatencyMetric.h"
#include "METRICS/bwMetric.h"
#include "METRICS/latencyMetric.h"

#include "TIMERS/periodicPerformanceTimer.h"

void PerformanceAgent::CreatePokeMgr(){
  switch(protocolMetric){
  case BW_ONLY:
  case BW_LATENCY:
    pokeMgrPtr = new PokeMgr(gossipAgentPtr, this);
    break;
    
  case LATENCY_ONLY:
    pokeMgrPtr=NULL;
    break;
    
  case STATIC:
    MyError("No performance agent for STATIC tree!!!");
    break;
    
  default:
    MyError("Invalid Protocol Metric Type");
  }
}


/**************************
 ScheduleBWTestToNbrs(): 
 Relevant only for metrics that consider bandwidth

 Basic Idea: Schedule bw tests to neighbors that are unprobed, or 
             for which recent bw estimates are present.
 
 To save on the number of bw probes, we use heuristics.
 -Don't probe a member whose bw is good - it just takes a bit of 
  data flow to push the estimate down. But, it is more difficult
  to push the estimate to a neighbor up from some  bad value.
- For BW_LATENCY, don't probe a neighbor with a high latency
  (for eg. this could result in US machine repeatedly
  probing Taiwan to see if its bw has improved.
****************************/


int PerformanceAgent::ShouldScheduleBWTestToNbr(SelectionTabType *selectionTab,
						 int candidateID){
  if(
     (protocolMetric != BW_ONLY) &&
     (protocolMetric != BW_LATENCY)
     ){
    return(0);
  }
  
  int scheduleBWTest=0;
  int nbrAddr=selectionTab[candidateID].addr;
  
  VRTEntryType *routeToNbr=vrtPtr->GetRoute(nbrAddr);
  if(routeToNbr == NULL){
    MyError("No route entry to neighbor!!");
  }
  
  switch(protocolMetric){
      
  case BW_ONLY:
    {
      int timeSinceLastBWProbe=routeToNbr->GetTimeSinceLastBWProbe();
      BWMetric *linkMetric=(BWMetric *) routeToNbr->GetLinkMetric();
      
      if (
	  (timeSinceLastBWProbe < 0) ||
	  (	
	   (timeSinceLastBWProbe > MAX_NBR_UNPROBED_BW_TIME) &&
	   (!linkMetric->IsBest())
	   )
	  ){
	
	/*******
	scheduleBWTest=ShouldAddNbrIfGoodBW(nbrAddr,
					    routeToNbr->GetLastRTProbe(),
					    selectionTab,
					    candidateID);
	************/

	scheduleBWTest=1; // bypass application filter
      }
      delete linkMetric;
    }
    break;
    
  case BW_LATENCY:
    
    {
      int timeSinceLastBWProbe=routeToNbr->GetTimeSinceLastBWProbe();
      BWLatencyMetric *linkMetric=
	(BWLatencyMetric *) routeToNbr->GetLinkMetric();
      
      if(
	 (timeSinceLastBWProbe < 0) ||
	 (
	  (timeSinceLastBWProbe > MAX_NBR_UNPROBED_BW_TIME) &&
	  (!linkMetric->IsBestBW()) &&
	  (linkMetric->IsReasonableDelay())
	  )
	 ){
	
	/***********
	scheduleBWTest=ShouldAddNbrIfGoodBW(nbrAddr,
					    routeToNbr->GetLastRTProbe(),
					    selectionTab,
					    candidateID);
	**************/

	scheduleBWTest=1; // bypass application filter.

      }
      delete linkMetric;
    }
    break;
    
  case STATIC:
  case LATENCY_ONLY:
    MyError(" Should not come here!!!!");
    break;
    
  default:
    MyError("Invalid Protocol Code !!!!");
  }

  selectionTab[candidateID].linkAddIfBWGood = scheduleBWTest;
  return(scheduleBWTest);
}


int PerformanceAgent::ShouldAddNbrClose(RoutingMetric *routingMetric,
					RoutingMetric *linkMetric,
					SaturationCodeType remoteSaturationCode,
					SelectionTabType *selectionTab,
					int candidateID
					){

  int isLowDelayNbr=0;
  int isHighDelayPenaltyNbr=0;;

  switch(protocolMetric){
  case LATENCY_ONLY:
    {
      LatencyMetric *latencyLinkMetric=(LatencyMetric *) linkMetric;
      LatencyMetric *latencyRoutingMetric=(LatencyMetric *) routingMetric;
      
      isLowDelayNbr=latencyLinkMetric->IsLowDelay();
      isHighDelayPenaltyNbr=
	latencyRoutingMetric->IsHighDelayPenalty(latencyLinkMetric);
      
    }
    break;
    
  case BW_LATENCY:
    {
      BWLatencyMetric *bwLatencyLinkMetric=
	(BWLatencyMetric *) linkMetric;
      BWLatencyMetric *bwLatencyRoutingMetric=
	(BWLatencyMetric *) routingMetric;
    
      isLowDelayNbr=bwLatencyLinkMetric->IsLowDelay();
      isHighDelayPenaltyNbr=
	bwLatencyRoutingMetric->IsHighDelayPenalty(bwLatencyLinkMetric);						  
      break;
    }
  case BW_ONLY:
    break;
    
  case STATIC:
    MyError("Should not come here!!!!");
    break;
    
  default:
    MyError("Should not come here!!!!");
    break;
  }

  SaturationCodeType mySaturationCode=DetermineSaturationLevel();

  if (verbosity > 0){
    cout << "\n IsLowDelayNbr: " << isLowDelayNbr
	 << " IsHighDelayPenaltyNbr: " << isHighDelayPenaltyNbr;
  }

  if(
     isLowDelayNbr &&
     isHighDelayPenaltyNbr && 
     (remoteSaturationCode != HIGH) && 
     (mySaturationCode != HIGH)
     ){
    if(selectionTab != NULL){
      selectionTab[candidateID].lowDelayHighPenaltyNbr=1;
    }
    return(1);
  }
  else{
    if(selectionTab != NULL){
      selectionTab[candidateID].lowDelayHighPenaltyNbr=0;
    }
    return(0);
  }
}

int PerformanceAgent::IsRecentBWEstimate(int timeSinceLastBWProbe,
		                         int nbrFlag){
  if(timeSinceLastBWProbe < 0) return(0);
  if(nbrFlag == 0){
    if(timeSinceLastBWProbe > MEMBER_BWINFO_TIMEOUT) return(0);
  }
  else{
    if(timeSinceLastBWProbe > MAX_NBR_UNPROBED_BW_TIME) return(0);
  }
  return(1);
}

int PerformanceAgent::ShouldAddNbr(int fromAddr, 
				   ProbeResponseMsg *msgPtr,
				   SelectionTabType *selectionTab,
				   int candidateID) {

  if(nbrTabPtr->IsNbr(fromAddr)){
    return(0);
  }

  VRTEntryType *route=vrtPtr->GetRoute(fromAddr);
  if(route == NULL) MyError("No Route to a probed node");
  
  int timeSinceLastDelayProbe=route->GetTimeSinceLastDelayProbe();
  int timeSinceLastBWProbe=route->GetTimeSinceLastBWProbe();

  /** this message should have included a delay estimate **/
  assert(timeSinceLastDelayProbe >= 0); 
  
  RoutingMetric *linkMetric=route->GetLinkMetric();
  RoutingMetric *routingMetric=route->GetRoutingMetric();

  int shouldAddNbr=0;
  int shouldAddNbrClose=0;
  int shouldAddNbrUtility=0;

  switch(protocolMetric){
  case LATENCY_ONLY:
    shouldAddNbrClose=ShouldAddNbrClose(
					routingMetric,
					linkMetric,
					msgPtr->GetSaturationCode(),
					selectionTab,
					candidateID
					);
    
    shouldAddNbrUtility=ShouldAddNbrUtility(fromAddr,
					    msgPtr,
					    linkMetric,
					    selectionTab,
					    candidateID);
    
    shouldAddNbr = (
		    (shouldAddNbrClose) ||
		    (shouldAddNbrUtility)
		    );
    
    break;
    
  case BW_ONLY:
    if (IsRecentBWEstimate(timeSinceLastBWProbe,0)){
      shouldAddNbrUtility=ShouldAddNbrUtility(fromAddr,
					      msgPtr,
					      linkMetric,
					      selectionTab,
					      candidateID);
    }
    shouldAddNbr=(shouldAddNbrUtility);
    
    break;
    
  case BW_LATENCY:
    
    if (
	(IsRecentBWEstimate(timeSinceLastBWProbe,0)) ||
	(IsLocalCongestionSignal())
	){
      
      shouldAddNbrClose=
	ShouldAddNbrClose(
			  routingMetric,
			  linkMetric,
			  msgPtr->GetSaturationCode(),
			  selectionTab,
			  candidateID
			  );
      
      shouldAddNbrUtility=ShouldAddNbrUtility(fromAddr,
					      msgPtr,
					      linkMetric,
					      selectionTab,
					      candidateID);

    } 

    shouldAddNbr = (
		    (shouldAddNbrClose) ||
		    (shouldAddNbrUtility)
		    );
    break;
    
  case STATIC:
    MyError("STATIC should not come here !!!!");
    break;
    
  default:
    MyError("INVALID protocol code !!");
  }
  
  delete routingMetric;
  delete linkMetric;
  return(shouldAddNbr);
}

int PerformanceAgent::ShouldAddNbrIfGoodBW(int fromAddr, 
					   ProbeResponseMsg *msgPtr,
					   SelectionTabType *selectionTab,
					   int candidateID){
  
  VRTEntryType *route=vrtPtr->GetRoute(fromAddr);
  if(route == NULL) MyError("No Route to a probed node");
  int timeSinceLastBWProbe=route->GetTimeSinceLastBWProbe();

  int shouldScheduleBWTest=0;

  RoutingMetric *routingMetric=route->GetRoutingMetric();
  RoutingMetric *linkMetric=route->GetLinkMetric();

  switch(protocolMetric){
  case LATENCY_ONLY:
    break;
    
  case BW_ONLY:
    if( 
       (!IsRecentBWEstimate(timeSinceLastBWProbe,
			    nbrTabPtr->IsNbr(fromAddr)
			    )) &&
       (!IsLocalCongestionSignal())
       ){
      
      linkMetric->SetBest();
      shouldScheduleBWTest=ShouldAddNbrUtility(fromAddr,
					       msgPtr,
					       linkMetric,
					       selectionTab,
					       candidateID);
    }
    break;  
    
  case BW_LATENCY:
    if( 
       (!IsRecentBWEstimate(timeSinceLastBWProbe,
			    nbrTabPtr->IsNbr(fromAddr)
			    )) &&
       (!IsLocalCongestionSignal())
       ){
      int shouldScheduleBWTestClose=
	ShouldAddNbrClose(routingMetric,
			  linkMetric,
			  msgPtr->GetSaturationCode(),
			  selectionTab,
			  candidateID
			  );
      
      
      ((BWLatencyMetric *) linkMetric)->SetBestBW();
      int shouldScheduleBWTestUtility=ShouldAddNbrUtility(fromAddr,
							  msgPtr,
							  linkMetric,
							  selectionTab,
							  candidateID);
      
      shouldScheduleBWTest=(
			    (shouldScheduleBWTestClose) ||
			    (shouldScheduleBWTestUtility)
			    );
    }
    
    break;
    
  case STATIC:
    MyError("STATIC should not come here !!!!");
    break;
    
  default:
    MyError("INVALID protocol code !!");
  }
  
  delete routingMetric;
  delete linkMetric;

  return(shouldScheduleBWTest);
}

int PerformanceAgent::IsCloseNbr(RoutingMetric *linkMetric){

  int isLowDelayNbr=0;

  switch(protocolMetric){
  case LATENCY_ONLY:
    {
      LatencyMetric *latencyLinkMetric=(LatencyMetric *) linkMetric;
      isLowDelayNbr=latencyLinkMetric->IsLowDelay();
    }
    break;
    
  case BW_LATENCY:
    {
      BWLatencyMetric *bwLatencyLinkMetric=(BWLatencyMetric *) linkMetric;
      isLowDelayNbr=bwLatencyLinkMetric->IsLowDelay();
    }
    break;

  case BW_ONLY:
    isLowDelayNbr=0;
    break;

  case STATIC:
    MyError("Should not come here!!!!");
    break;
    
  default:
    MyError("Should not come here!!!!");

    break;
  }

  return(isLowDelayNbr);
}
