/**************************
 PerformanceAgent.cc: This agent seeks to improve the performance component
 of Narada, by adding and dropping links. 

 It has functions that:
 -Monitor existing links, and drop them if they are not needed.
 -Probe new links, and add them if they improve quality.
 -Schedule tests to determine physical properties of links 
 (particularly bandwidth).

 This agent depends on the exact metric that Narada tries to optimize.
 It could have been written as a base class, with a derived class for
 metric specific dependencies. However, as this seems relatively small,
 we put all metric specific code in a companion file 
 performanceAgentMetric.cc
*******************************/

#include <stdio.h>
#include <stdlib.h>
#include <iostream.h>
#include <assert.h>

#include "global.h"
#include "gossipPayLoad.h"
#include "vrt.h"
#include "vrtEntry.h"
#include "neighborTable.h"
#include "gossipAgent.h"
#include "connectivityAgent.h"
#include "performanceAgent.h"
#include "pokeMgr.h"
#include "METRICS/routingMetric.h"

#include "TIMERS/periodicPerformanceTimer.h"
#include "TIMERS/periodicRTProbeTimer.h"

PerformanceAgent::PerformanceAgent(GossipAgent *gossipAgentPtrIn, 
				   ConnectivityAgent *connectivityAgentPtrIn, 
				   VirtualRoutingTable *vrtPtrIn, 
				   NbrTable *nbrTabPtrIn){
  gossipAgentPtr=gossipAgentPtrIn;
  connectivityAgentPtr=connectivityAgentPtrIn;
  vrtPtr=vrtPtrIn;
  nbrTabPtr=nbrTabPtrIn;

  CreatePokeMgr();

  periodicPerformanceTimer= new PeriodicPerformanceTimer(this);
  periodicRTProbeTimer = new PeriodicRTProbeTimer(this);
  
  performanceAgentSeqNum=1;

  expectedResponseSeqNum = -1;
  expectedResponseFromAddr = -1;
}

void PerformanceAgent::StartAgent(){
  this->PeriodicPerformanceCycle();
  this->PeriodicRTProbeCycle();
}

PerformanceAgent::~PerformanceAgent(){
  delete periodicPerformanceTimer;
  delete periodicRTProbeTimer;
  
  if(pokeMgrPtr != NULL){
    delete pokeMgrPtr;
  }
}


void PerformanceAgent::PeriodicRTProbeCycle(){
  
  for(int i=0; i < NUM_PROBES_IN_CYCLE; i++){
    int addrToProbe=ProbeSelect();



    if(addrToProbe != -1){ 

      VRTEntryType *route=vrtPtr->GetRoute(addrToProbe);
      route->UpdateOnRTProbeSent();

      ProbeRequestMsg *msg=new ProbeRequestMsg(
					       performanceAgentSeqNum,
					       GetCurrTime()
					       );
      gossipAgentPtr->SendToNetwork(addrToProbe,
				    GOSSIP_PORT,
				    PROBE_REQUEST,
				    msg);
      performanceAgentSeqNum++;
    }
  }
  
  periodicRTProbeTimer->SetTimer(PERIODIC_RT_PROBE_TIMER);
}


/*******************
 PeriodicPerformanceCycle():  This function is the core engine of the
 performance agent. It executes periodically and does the following:

 -Evaluates quality of links to neighbors and determines if any 
  must be dropped
 -Selects and sends a probe request to a non-neighbor, to determine if
  adding a link to that member is useful. 
***********************/


void PerformanceAgent::PeriodicPerformanceCycle(){

  if(!connectivityAgentPtr->IsPartitionMode()){
    if(DetermineSaturationLevel() != LOW){
      EvaluateAndDropNbrs(); 
    }
  }
  
  ShouldAddNewNbr();

  periodicPerformanceTimer->SetTimer(PERIODIC_PERFORMANCE_TIMER);
}



int PerformanceAgent::ProbeSelect(){
  
  int addrToProbe = -1;
  int maxTimeSinceLastRTProbe=-1;

  for(VRTEntryType *route = vrtPtr->GetNextRouteInit(); 
      route != NULL; 
      route=vrtPtr->GetNextRoute()){
    
    // don't probe dead person, myself or a neighbor. 
    if (route->IsDead()) continue;
    if (route->GetAddr() == GetMyAddr()) continue;
    
    int timeSinceLastRTProbeSent = route->GetTimeSinceLastRTProbeSent();
    if(timeSinceLastRTProbeSent < 0){
      return(route->GetAddr());
    }
    
    if(timeSinceLastRTProbeSent > maxTimeSinceLastRTProbe){
      addrToProbe=route->GetAddr();
      maxTimeSinceLastRTProbe=timeSinceLastRTProbeSent;
    }
  }
  
  return(addrToProbe);
  
  MyError("Something wrong!!!");
  return(-1);
}

int PerformanceAgent::SelectMemberWithKnownBW(int numCandidates,
					      SelectionTabType *selectionTab){

  int maxUtilityID=-1;
  float maxUtility=-1;

  for(int i=0; i < numCandidates; i++){
    
    if(selectionTab[i].linkAdd){
      assert(!selectionTab[i].nbrFlag);
      if(selectionTab[i].lowDelayHighPenaltyNbr){
	
	if(verbosity > 0){
	  PrintSelectionTab(numCandidates, selectionTab, i);
	  printf ("\nChoose Known BW Link: Delay %s ", 
		  GetNameByAddr(selectionTab[i].addr));
	}
	
	return(selectionTab[i].addr);
	
      }
    
      assert(selectionTab[i].utilityGoodFlag);
      
      if(selectionTab[i].utility > maxUtility){
	maxUtilityID=i;
	maxUtility=selectionTab[i].utility;
      }
    }
  }


  if(maxUtilityID < 0){
    return(-1);
  }
  else{
    if(verbosity > 0){
      PrintSelectionTab(numCandidates, selectionTab, maxUtilityID);
      printf ("\nChoose Known BW Link: utility : %s ", 
	      GetNameByAddr(selectionTab[maxUtilityID].addr));
    }
    return(selectionTab[maxUtilityID].addr);
  }
}


void PerformanceAgent::PrintSelectionTab(int numCandidates,
					 SelectionTabType *selectionTab,
					 int selectedID){

  for(int i=0; i < numCandidates; i++){
    VRTEntryType *route 
      = vrtPtr->GetRoute(selectionTab[i].addr);
    
    printf ("\n%s nbr:%d add:%d test:%d near:%d util:%d %.2f %d %d %d",
	    GetNameByAddr(selectionTab[i].addr),
	    selectionTab[i].nbrFlag,
	    selectionTab[i].linkAdd,
	    selectionTab[i].linkAddIfBWGood,
	    selectionTab[i].lowDelayHighPenaltyNbr,
	    selectionTab[i].utilityGoodFlag,
	    selectionTab[i].utility,
	    route->GetPhysicalDelay(),
	    route->GetPhysicalBW(),
	    route->GetTimeSinceLastRTProbeRcvd()
	    );
  }

  if(selectedID == -1){
    return;
  }
  VRTEntryType *route=vrtPtr->GetRoute(selectionTab[selectedID].addr);
  if(route == NULL){
    MyError("No route for selected node !!!");
  }
  ProbeResponseMsg *msgPtr=route->GetLastRTProbe();
  if(msgPtr == NULL){
    MyError("No cached message for selected node !!!");
  }
  msgPtr->Print();
}

void PerformanceAgent::PrintShortList(int numShortListed, 
				      int *shortListArr ){

  printf ("\n Num ShortListed: %d",numShortListed);
  for(int i=0; i < numShortListed; i++){
    VRTEntryType *route = 
      vrtPtr->GetRoute(shortListArr[i]);
    if(route == NULL){
      MyError("No route to short listed guy!");
    }
    
    printf ("\nShortListed Unknown BW Link: %s Nbr:%d Delay: %d", 
	    GetNameByAddr(shortListArr[i]),
	    nbrTabPtr->IsNbr(shortListArr[i]),
	    route->GetPhysicalDelay()
	    );
  }
}


int PerformanceAgent::ShortListMembersWithUnknownBW(
				int numCandidates,
				SelectionTabType *selectionTab,
				ShortListMethod shortListMethod,
				int *shortListArr){
  
  int shortListID;
  
  switch(shortListMethod){
  case DELAY_SHORTLIST:
    {
      for(shortListID=0; 
	  shortListID < NUM_MEMBERS_TO_SHORTLIST; shortListID++){
	
	int minNbrDelayID=-1;
	int minNbrDelay=10000;
	int minDelayID=-1;
	int minDelay=10000;
	
	for(int i=0; i < numCandidates; i++){

	  int memberAlreadySelected=0;
	  for(int k=0; k < shortListID; k++){
	    if(selectionTab[i].addr == shortListArr[k]){
	      memberAlreadySelected=1;
	      break;
	    }
	  }
	  if(memberAlreadySelected)
	    continue;
	    
	  if(selectionTab[i].linkAddIfBWGood){
	    
	    VRTEntryType *route 
	      = vrtPtr->GetRoute(selectionTab[i].addr);
	    
	    if(route == NULL){
	      MyError("No route entry to entry in selection tab!!");
	    }
	    
	    int timeSinceBWTestFailure 
	      = route->GetTimeSinceLastBWTestFailure();
	    
	    if(
	       (timeSinceBWTestFailure > 0) &&
	       (timeSinceBWTestFailure < 30000)
	       ){
	      continue;
	    }
	    
	    int phyDelay=route->GetPhysicalDelay();
	    
	    if(phyDelay < 0){
	      MyError("No physical delay estimate to probed node!!");
	    }
	    
	    if(selectionTab[i].nbrFlag){
	      if(phyDelay < minNbrDelay){
		minNbrDelayID=i;
		minNbrDelay=phyDelay;
	      }
	    }
	    else{
	      if(phyDelay < minDelay){
		minDelayID=i;
		minDelay=phyDelay;
	      }
	    }
	  }
	}
	
	if(
	   (minNbrDelayID < 0) && 
	   (minDelayID < 0)
	   ){
	  return(shortListID);
	}
	else{
	  
	  
	  // Prevent a bad neighbor Taiwan from being chosen.
	  if(
	     (minNbrDelayID >= 0) &&
	     (
	      (minDelayID < 0) ||
	      
	      (
	       (minDelayID >= 0) &&
	       (minNbrDelay - minDelay) < 50)
	      )
	     ){
	    minDelayID=minNbrDelayID;
	  }
	  shortListArr[shortListID]=
	    selectionTab[minDelayID].addr;
	}
      }
    }
    break;

  case RANDOM_SHORTLIST:
    { /*RANDOM*/
      
      for(shortListID=0; 
	  shortListID < NUM_MEMBERS_TO_SHORTLIST; 
	  shortListID++){
	
	int numNbrs=0;
	int numNonNbrs=0;
	int nbrPresent=0;
	int numChoices=0;
	for(int i=0; i < numCandidates; i++){
	  
	  int memberAlreadySelected=0;
	  for(int k=0; k < shortListID; k++){
	    if(selectionTab[i].addr == shortListArr[k]){
	      memberAlreadySelected=1;
	      break;
	    }
	  }

	  if(memberAlreadySelected){
	    continue;
	  }
	    
	  if(selectionTab[i].linkAddIfBWGood){
	    if(selectionTab[i].nbrFlag){
	      numNbrs++;
	    }
	    else{
	      numNonNbrs++;
	    }
	  }
	}
      
      
	if(numNbrs){
	  nbrPresent=1;
	  numChoices=numNbrs;
	}
	else{
	  numChoices=numNonNbrs;
	}
      
      
	if(numChoices == 0){
	  return(shortListID);
	}
      
	int choice = rand() % numChoices;
	int j=0;
      
	for(int i=0; i < numCandidates; i++){

	  int memberAlreadySelected=0;
	  for(int k=0; k < shortListID; k++){
	    if(selectionTab[i].addr == shortListArr[k]){
	      memberAlreadySelected=1;
	      break;
	    }
	  }
	  
	  if(memberAlreadySelected){
	    continue;
	  }
	  
	  if(selectionTab[i].linkAddIfBWGood){
	    if(
	       (nbrPresent && selectionTab[i].nbrFlag) ||
	       (!nbrPresent && !selectionTab[i].nbrFlag)
	       ){
	      
	      assert(j <= choice);
	      if(j == choice){
		shortListArr[shortListID]=selectionTab[i].addr;
	      }
	      else{
		j++;
	      }
	    }
	  }
	}
      }
      break;
    }
    
  default:
    MyError("Should not come here!!!!");
  }
  
  if(shortListID != NUM_MEMBERS_TO_SHORTLIST){
    MyError("ShortListID = %d should not come here",shortListID);
  }
  return(shortListID);
}

/***************
 EvaluateAndDropNbrs():

 Drop that neighbor which has the lowest consensus cost, if that cost
 is lower than a certain threshold. Consensus Cost is the maximum
 of the cost of the link from each neighbors perspective. The cost
 of a link to $j$ from $i$'s perspective is the number of members 
 for which $j$ uses $i$ as the next hop to reach the source.

 As small exceptions, we do not drop links that have recently 
 experienced data flow.

 The protocol specific dependency comes in because if its a BW_ONLY
 protocol, then, we may even drop neighbors that have a very low delay.
******************/

void PerformanceAgent::EvaluateAndDropNbrs(){
  int numNbrs;
  int *IUseAsNextHopCount;
  int *IAmNextHopCount;

  // Only one neighbor dropped at a time.
  if(nbrTabPtr->DoExistTempNbrs()) return; 
  numNbrs=nbrTabPtr->GetNumNbrs();
  if(numNbrs <= 1) return;

  int *nbrAddrList=new int [numNbrs];
  IUseAsNextHopCount = new int [numNbrs];
  IAmNextHopCount = new int [numNbrs];

  nbrTabPtr->GetNbrList(nbrAddrList);
  
  int i;
  for(i=0; i < numNbrs;i++){
    IUseAsNextHopCount[i]=0;
    IAmNextHopCount[i]=0;
  }

  for(VRTEntryType *route=vrtPtr->GetNextRouteInit(); route != NULL; 
      route=vrtPtr->GetNextRoute()) {
    {
      int i;
      // *** Even if it is TEMPORARY_ROUTE, we give credit. ****//

      if(!route->IsValidRoute()) continue;
      if(route->GetAddr() != GetMyAddr()){
	for(i=0; (i < numNbrs) && (!route->IsNextHop(nbrAddrList[i])); i++);
	if(i >= numNbrs) MyError("Next hop not any of nbrs");
	IUseAsNextHopCount[i]++;
      }
    }

    {
      int numChildren=route->GetNumChildren();
      int *childAddrList= new int [numChildren];
      route->GetChildList(childAddrList);
      
      int i,j;
      for(i=0;i<numChildren;i++){
	for(j=0;j<numNbrs && childAddrList[i] != nbrAddrList[j];j++);
	if(j >= numNbrs) MyError("child is not present in nbrs list");
	IAmNextHopCount[j]++;
      }
      delete [] childAddrList;
    }
  }

  /*****************
     Check for which of the nbrs max{IAmNextHopCount,IUseAsNextHopCount} 
     is minimum. Also, consider only those nbrs that have been nbrs for 
     a minimum length of time
  *************/
  
  {
    
    int minUtilityNbr = -1;
    int minUtility;
    int i;
    for(i=0; (i < numNbrs) && (minUtilityNbr == -1); i++){
      int nbrDuration= GetCurrTime() - nbrTabPtr->GetTimeSinceNbr(nbrAddrList[i]);
      VRTEntryType *route=vrtPtr->GetRoute(nbrAddrList[i]);

      RoutingMetric *nbrLinkMetric=route->GetLinkMetric();

      int rateToNbr=nbrTabPtr->GetRateSentToNbr(nbrAddrList[i]);
      int rateFromNbr=nbrTabPtr->GetRateRcvdFromNbr(nbrAddrList[i]);
      
      /********* 
	   Don't drop a neighbor with data flow, 
	   that is very close, or that has been added recently
      **********/
      if(
	 (nbrDuration > MIN_DURATION_BEFORE_PRUNING_NBR) &&
	 (rateToNbr == 0) &&
	 (rateFromNbr == 0) &&
	 (!(IsCloseNbr(nbrLinkMetric)))
	 ){
	minUtilityNbr=i;
	if(IUseAsNextHopCount[i] > IAmNextHopCount[i])
	  minUtility=IUseAsNextHopCount[i];
	else minUtility=IAmNextHopCount[i];
      }
      delete nbrLinkMetric;
    }

    if(minUtilityNbr != -1) {
      for(;i < numNbrs; i++){

	int nbrDuration=
	  GetCurrTime()-nbrTabPtr->GetTimeSinceNbr(nbrAddrList[i]);

	VRTEntryType *route=vrtPtr->GetRoute(nbrAddrList[i]);
	RoutingMetric *nbrLinkMetric=route->GetLinkMetric();

	int rateToNbr=nbrTabPtr->GetRateSentToNbr(nbrAddrList[i]);
	int rateFromNbr=nbrTabPtr->GetRateRcvdFromNbr(nbrAddrList[i]);
	if(
	   (nbrDuration > MIN_DURATION_BEFORE_PRUNING_NBR) &&
	   (rateToNbr == 0) &&
	   (rateFromNbr == 0) &&
	   (!IsCloseNbr(nbrLinkMetric))
	   ){
	  int utility;
	  if(IUseAsNextHopCount[i] > IAmNextHopCount[i])
	    utility=IUseAsNextHopCount[i];
	  else utility=IAmNextHopCount[i];
	  
	  if(minUtility > utility){
	    minUtility=utility;
	    minUtilityNbr=i;
	  }
	}
	delete nbrLinkMetric;
      }
      
      float droppingThresh=CalculateDroppingThresh();
      if(minUtility < droppingThresh){
	int rateToNbr=nbrTabPtr->GetRateSentToNbr(nbrAddrList[minUtilityNbr]);
	int rateFromNbr=nbrTabPtr->GetRateRcvdFromNbr(nbrAddrList[minUtilityNbr]);
	
	if (verbosity > 0){
	  cout << "\n Trying to drop. minUtility = " << minUtility
	       << " dropping thresh = " << droppingThresh
	       << " rateToNbr = " << rateToNbr
	       << " rateFromNbr = " << rateFromNbr;

	  vrtPtr->Print();
	}
	
	nbrTabPtr->MarkNbrTemp(nbrAddrList[minUtilityNbr]); 
	IntentToCancelNbrMsg *msg=new IntentToCancelNbrMsg(0);
	gossipAgentPtr->SendToNetwork(nbrAddrList[minUtilityNbr],
				      GOSSIP_PORT,
				      INTENT_TO_CANCEL_NBR,
				      msg);
	vrtPtr->UpdateOnNbrTemp(nbrAddrList[minUtilityNbr]);
	
      }
    }
  }

  delete [] nbrAddrList;
  delete [] IUseAsNextHopCount;
  delete [] IAmNextHopCount;
}

void PerformanceAgent::RecvJoinRequest(int fromAddr,
				       int fromPort,
				       JoinRequestMsg *msg){
  int numRoutes;
  ProbeResponseRecord *probeResponseRecordPtr=
    vrtPtr->CreateProbeResponseRecord(&numRoutes);
  
  JoinResponseMsg *responseMsg=new 
    JoinResponseMsg(msg->GetSeqNum(),
		    msg->GetTime(),
		    GetCurrTime(),
		    DetermineSaturationLevel(),
		    numRoutes,
		    probeResponseRecordPtr);
  
  gossipAgentPtr->SendToNetwork(fromAddr,
				GOSSIP_PORT,
				JOIN_RESPONSE,
				responseMsg);
}
  
void PerformanceAgent::RecvProbeRequest(int fromAddr,
					int fromPort,
					ProbeRequestMsg *msg){
  
  /*******  
  A probe response sends all records, including 
  those considered dead 
  ********/

  int numRoutes;
  ProbeResponseRecord *probeResponseRecordPtr=
    vrtPtr->CreateProbeResponseRecord(&numRoutes);

  ProbeResponseMsg *responseMsg=new 
    ProbeResponseMsg(msg->GetSeqNum(),
		     msg->GetTime(),
		     GetCurrTime(),
		     DetermineSaturationLevel(),
		     numRoutes,
		     probeResponseRecordPtr);
  gossipAgentPtr->SendToNetwork(fromAddr,fromPort,PROBE_RESPONSE,responseMsg);
}

void PerformanceAgent::RecvProbeResponse(int fromAddr,
					 int fromPort,
					 ProbeResponseMsg *msgPtr){
  

  if(fromPort != GOSSIP_PORT){
    return;
  }
  
  // get the route 
  VRTEntryType *route=vrtPtr->GetRoute(fromAddr);
  if(route == NULL){
    /** Don't crash to protect against bogus packets!! **/
    MyWarning("PerformanceAgent: Probe Response from node %s/%d for which no entry exists", GetNameByAddr(fromAddr), fromPort);
    return;
  }
  
  //Estimate delay
  int roundTripDelay=GetCurrTime() - msgPtr->GetRemoteTimeRequestSent();
  if(roundTripDelay > PERFORMANCE_PROBE_RESPONSE_TIMER){
    return; 
  }

  if(fromAddr == timeSyncServerAddr){
    int offset= (msgPtr->GetRemoteTimeRequestSent() + roundTripDelay/2)
      -  (msgPtr->GetLocalTimeRequestRcvd());
    if(verbosity > 0){
      cout << "\n" << GetCurrTime() << ":"
	   << "Offset from " << GetNameByAddr(timeSyncServerAddr)
	   << "(" << GetNameByAddr(fromAddr) << ")"
	   << " is " << offset; 
    }
  }

  
  route->IntegrateDelayResult(roundTripDelay);
  route->UpdateOnRTProbeRcvd(msgPtr);
  
  if (nbrTabPtr->IsNbr(fromAddr)){
    return;
  }
}


/****************************
 Algorithm:

 -Go through list of nodes that is not dead, neighbor or myself.
 -Based on cached Probe Responses determine if:
     (i) link should be added
    (ii) link should be added if bw is good (but bw is unknown now).

 -If there are any links that belong to category (i),
  pick one of those links and send ADD_NBR_REQUEST.
  The link to pick is based on the following algorithm:
     -Pick any link with low delay, high delay penalty
     -If no such links exist, pick one with greatest utility

 -If there are no links that belong to category (i), but links
  belonging to category (ii), pick one/subset of links and
  do any of the following algorithms:

  (1) Naive: Pick 1 link at Random, add link.
  (2) Delay/GNP: Pick link with lowest delay, add link.
  (3) Delay + RTT probe: Pick 3 links with lowest delay
                         Do an RTT probe. Pick based on this.
  (4) Delay + TFRC: Pick 3 links with lowest delay.
                    Do a TFRC test.
********************************/



void PerformanceAgent::ShouldAddNewNbr(){
  
  /***********
   Apply application level filters to identify the good hosts
  ************/
  if (verbosity > 0) {
      vrtPtr->Print();
  }
  
  int numRoutes=vrtPtr->GetNumRoutes();

  SelectionTabType *selectionTab=
    new SelectionTabType[numRoutes];
  
  int numCandidates=0;

  for(VRTEntryType *route = vrtPtr->GetNextRouteInit(); 
      route != NULL; 
      route=vrtPtr->GetNextRoute()){

    assert(numCandidates < numRoutes);

    selectionTab[numCandidates].addr=route->GetAddr();
    selectionTab[numCandidates].nbrFlag=nbrTabPtr->IsNbr(route->GetAddr());
    selectionTab[numCandidates].linkAdd=0;
    selectionTab[numCandidates].linkAddIfBWGood=0;
    selectionTab[numCandidates].lowDelayHighPenaltyNbr=0;
    selectionTab[numCandidates].utilityGoodFlag=0;
    selectionTab[numCandidates].utility=0;

    assert(!(route->IsDead()));
    if (route->GetAddr() == GetMyAddr()) continue;
    if (
	(route->GetTimeSinceLastRTProbeRcvd() > PERIODIC_PERFORMANCE_TIMER) ||
	(route->GetTimeSinceLastRTProbeRcvd() < 0) 
	){
      continue;
    }
    
    if(route->GetLastRTProbe() == NULL){
      continue;
    }

    if(selectionTab[numCandidates].nbrFlag){
      ShouldScheduleBWTestToNbr(selectionTab,
				numCandidates);
    }
    else{
      EvaluateProspectiveNbr(
			     route->GetAddr(),
			     route->GetLastRTProbe(),
			     selectionTab,
			     numCandidates
			     );
    }
    
    numCandidates++;
  }
  
  int memberAddr = SelectMemberWithKnownBW(numCandidates,selectionTab);
  
  if(memberAddr != -1){
    SendAddNbrRequest(memberAddr);
  }
  else{ 
    if(
       (protocolMetric == BW_ONLY) ||
       (protocolMetric == BW_LATENCY)
       )/*BW_ONLY, BW_LATENCY*/{

      int shortListArr[NUM_MEMBERS_TO_SHORTLIST];
      ShortListMethod shortListMethod;
      
      
      switch(protocolMetric){
      case RANDOM:
      case RANDOM_10K:
      case RANDOM_TFRC:

	shortListMethod=RANDOM_SHORTLIST;
	break;

      case GNP:
      case GNP_10K:
      case GNP_TFRC:
      case GNP_10K_TFRC:
      case RTT:
      case RTT_10K:
      case RTT_TFRC:
      case RTT_10K_TFRC:
      case HISTORY:

	
	shortListMethod=DELAY_SHORTLIST;
	break;
	
      default:
	MyError("Unsupported probe method !!!");
      }
      
      
      int numShortListed=ShortListMembersWithUnknownBW(numCandidates,
						       selectionTab,
						       shortListMethod,
						       shortListArr);

      if(verbosity > 0){
	PrintSelectionTab(numCandidates, 
			  selectionTab, 
			  -1);

	PrintShortList(numShortListed,shortListArr);
      }

      if(numShortListed <= 0){
	return;
      }
      
      switch(probeMethod){
      case RANDOM:
      case GNP:
      case RTT:
	DoOptimisticPolicy(shortListArr[0]);
	break;
	
      case RANDOM_10K:
      case GNP_10K:
      case RTT_10K:
      case GNP_10K_TFRC:
      case RTT_10K_TFRC:
	{
	  connectivityAgentPtr->ScheduleSmallProbeTest(shortListArr,
						       numShortListed,
						       this);
	}
	
	break;
	
      case RANDOM_TFRC:
      case GNP_TFRC:
      case RTT_TFRC:
      case HISTORY:
      {
	int memberAddr=shortListArr[0];
	/** Pessimistic policy ***/
	pokeMgrPtr->ScheduleBwTest(memberAddr,GOSSIP_PORT);
      }
      break;
      
      default:
	MyError("Invalid probe method!!!");
      }
    }
  }
  delete [] selectionTab;
}


void PerformanceAgent::DoOptimisticPolicy(int memberAddr){
  if(nbrTabPtr->IsNbr(memberAddr)){
    VRTEntryType *route=vrtPtr->GetRoute(memberAddr);
    if(route == NULL){
      MyError("No route to neighbor!!!");
    }
      
    if(verbosity > 0){
      printf("\n OPTIMISTIC report: %s currBW=%d BRP=%d BLP=%d",
	     GetNameByAddr(memberAddr),
	     route->GetPhysicalBW(),
	     route->GetNumBWProbes(),
	     route->GetNumBWResultProbes()
	     );
    }

    route->IntegrateBWResult(MAX_SOURCE_RATE,BW_RESULT_LB);
  }
  else{
    SendAddNbrRequest(memberAddr);
  }
}

void PerformanceAgent::ReportFromSmallProbe(int addr){

  if(verbosity > 0){
    printf("\nSmall Probe Chooses: %s",GetNameByAddr(addr));
  }

  if(addr == -1){
    return;
  }

  switch(probeMethod){
  case RANDOM_10K:
  case GNP_10K:
  case RTT_10K:
    DoOptimisticPolicy(addr);
    break;
    
  case GNP_10K_TFRC:
  case RTT_10K_TFRC:  
    pokeMgrPtr->ScheduleBwTest(addr,GOSSIP_PORT);
    break;

  default:
    MyError("Non 10K probe method: should not come here !!!");
  }
}

  
void PerformanceAgent::SendAddNbrRequest(int toAddr){
  
  connectivityAgentPtr->SendAddNbrRequest(toAddr,
					  performanceAgentSeqNum,
					  PERFORMANCE);
  
  expectedResponseSeqNum=performanceAgentSeqNum;
  expectedResponseFromAddr=toAddr;
  
  performanceAgentSeqNum++;
  
}
				    

void PerformanceAgent::RecvAddNbrResponse(int fromAddr,
					  int fromPort,
					  AddNbrResponseMsg *msgPtr){
  
  if(
     (msgPtr->GetType() != PERFORMANCE) ||
     (expectedResponseFromAddr != fromAddr) ||
     (expectedResponseSeqNum != msgPtr->GetRemoteSeqNumOfPrevMsg())
     ){
    // Message can be ignored.
    if (verbosity > 0){
      cout << " REJECTED ";
    }
    return;
  }
  
  
  if(verbosity > 0){
    cout << "\n" << GetCurrTime() << ": ADD-NBR-PERFORMANCE " 
	 << GetNameByAddr(gossipAgentPtr->GetMyAddr()) 
	 << " " << GetNameByAddr(fromAddr) << " ";
  }
  
  connectivityAgentPtr->ProcessAddNbrResponse(fromAddr,
					      fromPort,
					      msgPtr);
  
}

void PerformanceAgent::RecvProbeMsg(int fromAddr, 
				    int fromPort, 
				    GossipPayLoad *msg){
  pokeMgrPtr->RecvPokeMsg(fromAddr, fromPort, msg);
}

int PerformanceAgent::GetMyAddr(){
  return(gossipAgentPtr->GetMyAddr());
}

void PerformanceAgent::UpdateBandwidth(int fromAddr, int rate) {
  VRTEntryType *route=vrtPtr->GetRoute(fromAddr);
  if(route == NULL) MyError("No Route to a probed node");

  if(rate < 0){
    // bw test fails
    route->SetBWTestFailure();
    return;
  }
  
  route->IntegrateBWResult(rate,BW_RESULT);

  if(!(nbrTabPtr->IsNbr(fromAddr))){

    ProbeRequestMsg *msg=new ProbeRequestMsg(performanceAgentSeqNum,
					     GetCurrTime()
					     );

    gossipAgentPtr->SendToNetwork(fromAddr,GOSSIP_PORT,PROBE_REQUEST,msg);
    performanceAgentSeqNum++;
  }
}

int PerformanceAgent::EvaluateProspectiveNbr(int fromAddr, 
					     ProbeResponseMsg *msgPtr,
					     SelectionTabType *selectionTab,
					     int candidateID) {
  
  int shouldAddNbr=0;
  int shouldAddNbrIfGoodBW=0;

  if(!(nbrTabPtr->IsNbr(fromAddr))){
    shouldAddNbr=ShouldAddNbr(fromAddr,
			      msgPtr,
			      selectionTab,
			      candidateID);
  }
  
  if(!shouldAddNbr){
    shouldAddNbrIfGoodBW=ShouldAddNbrIfGoodBW(fromAddr,
					      msgPtr,
					      selectionTab,
					      candidateID);
    
  }

  selectionTab[candidateID].linkAdd=shouldAddNbr;
  selectionTab[candidateID].linkAddIfBWGood=shouldAddNbrIfGoodBW;
  
  assert( 
	 (selectionTab[candidateID].linkAdd == 0) ||
	 (selectionTab[candidateID].linkAdd == 1)
	 );

  assert( 
	 (selectionTab[candidateID].linkAddIfBWGood == 0) ||
	 (selectionTab[candidateID].linkAddIfBWGood == 1)
	 );
  

  return(shouldAddNbr);
}

int PerformanceAgent::ShouldAddNbrUtility(int fromAddr,
                                          ProbeResponseMsg *msgPtr,
					  RoutingMetric *linkMetric,
					  SelectionTabType *selectionTab,
					  int candidateID
					  ){
  
  float utility=EvaluateUtility(fromAddr,msgPtr,linkMetric);
  int isUtilityGood=IsUtilityGood(utility, 
				  msgPtr->GetSaturationCode(),
				  nbrTabPtr->IsNbr(fromAddr));
  
  if(nbrTabPtr->IsNbr(fromAddr)){
    if(verbosity > 0){
      cout << "\n" << GetNameByAddr(fromAddr) <<
	" nbr utility = " << utility << " " << isUtilityGood; 
      msgPtr->Print();
    }
  }
  
  if(selectionTab != NULL){
    selectionTab[candidateID].utilityGoodFlag = isUtilityGood;
    selectionTab[candidateID].utility=utility;
  }
  
  return(isUtilityGood);
}



/*************************************
 EvaluateUtility: Evaluates the utility gain of adding a link to a member with 
                  address nodeProbed. probeResponseMsgPtr is the Probe Response
		  got when the member has been probed, and linkMetricToProbedNode
		  gives the routing cost of the link between the member and probed
		  node

 ***** Pseudocode for the algorithm is presented below: *******
  1. Go through each record in probe response.
  2. Check if node contained in my VRT
  I No,absent:
       utility += 0 : continue; 
       // Questionable decision, might want to reinvestigate.
  II Yes, present:
	utility += metric specific utility gain of adding new 
	           neighbor for this address

 Note: If my routing metric, or the the probed guy's routing metric to
       remote node is not well defined (NO_ROUTE/TR) do not consider in 
       utility. [Questionable, particularly for case when old bw/delay is not
       well defined]
		       
**************************************************************/

float PerformanceAgent::EvaluateUtility(int nodeProbed,
					ProbeResponseMsg *probeResponseMsgPtr,
                                        RoutingMetric *linkMetricToProbedNode
					){
  
  
  float utility=0;
  int numRecords=probeResponseMsgPtr->GetNumRecords();
  
  int i;
  for(i=0; i < numRecords; i++){
    ProbeResponseRecord *probeRecord=probeResponseMsgPtr->GetRecord(i);
    int addr,lastControlSeqNumOfAddrRcvdByProbedNode;
    RoutingMetric *routingMetricFromProbedNodeToAddr;
    
    addr=probeRecord->GetAddr();
    lastControlSeqNumOfAddrRcvdByProbedNode
      =probeRecord->GetLastControlSeqNum();
    routingMetricFromProbedNodeToAddr=probeRecord->GetRoutingMetric();
    
    VRTEntryType *route=vrtPtr->GetRoute(addr);
    
    if(route == NULL) continue; //Questionable
    if(route->IsDead()) continue;
    
    if(route != NULL){
      
      if(route->IsNextHop(nodeProbed)){
	continue;
      }
      
      // Similarly, don't consider in utility if the probed
      // member has NO_ROUTE or TR. This is definitely ok.
      if(
	 !(routingMetricFromProbedNodeToAddr->IsValidRoute())  
	 ){
	continue; 
      }

      RoutingMetric *currentRoutingMetric = 
	route->GetRoutingMetric();
      

      // If NO_ROUTE or TR, don't consider in utility. 
      // Sanjay: Changed this to reflect lack of partitionAgent:Sig2002
      
      if(!(currentRoutingMetric->IsValidRoute())){
	utility += 1;
      }

      else{
	utility += 
	  currentRoutingMetric->ComputeUtilityGain(linkMetricToProbedNode,
						   routingMetricFromProbedNodeToAddr);
      }
      delete currentRoutingMetric;
    }
  }
  return(utility);
}


int PerformanceAgent::IsUtilityGood(float utility, 
				    SaturationCodeType remoteSaturationCode,
				    int nbrFlag){
  
  SaturationCodeType mySaturationCode=
    DetermineSaturationLevel();
  float acceptThreshHigh=DetermineThreshHigh();
  float acceptThreshLow=DetermineThreshLow();
  

  if (verbosity > 0){

    cout << "\n" << "utility = " << utility 
	 << " threshHigh = " << acceptThreshHigh 
	 <<" threshLow = " << acceptThreshLow 
	 << " remoteSaturationCode = " << (int) remoteSaturationCode
	 <<" mySaturationCode = " << (int) mySaturationCode;
  }

  if(nbrFlag){
    if(utility > acceptThreshLow){
      return(1);
    }
    else{
      return(0);
    }
  }
  

  if(
     (
      (utility > acceptThreshHigh) && 
      (remoteSaturationCode != HIGH) && 
      (mySaturationCode != HIGH)
      ) 
     
     ||
     
     (
      (utility > acceptThreshLow)  && 
      (remoteSaturationCode == LOW) && 
      (mySaturationCode == LOW))
     ) {
    return(1); 
  }
  else{
    return(0);
  }
}



SaturationCodeType PerformanceAgent::DetermineSaturationLevel() {
  return connectivityAgentPtr->DetermineSaturationLevel();
}


float PerformanceAgent::DetermineThreshHigh(){
  /************ 
	Original algorithm: Estimate of group size / my_UpperDegreeBound
         New algorithm: Just Estimate of group size/ constant.
	                Same goes for other thresholds as well.
  *************/
  return(vrtPtr->GetNumRoutes()/ADD_HIGH_THRESH);
}

float PerformanceAgent::DetermineThreshLow(){
  return(vrtPtr->GetNumRoutes()/ADD_LOW_THRESH);
}

float PerformanceAgent::CalculateDroppingThresh(){
  return(vrtPtr->GetNumRoutes()/DROP_THRESH);
}

