#include <strings.h>
#include <iostream.h>
#include <assert.h>
#include <string.h>

#include <Util/inputSerializableStream.h>
#include <Util/outputSerializableStream.h>

#include "global.h"
#include "gossipPayLoad.h"

#include "METRICS/routingMetric.h"
#include "path.h"


char *PACKET_TYPE_STR[PACKET_TYPE_SIZE] = 
    {"DATAPACKET", "CONTROL", "PROBE", "POKE", "MEASUREMENT"};

GossipPayLoad::GossipPayLoad(GossipMsgType codeIn,
			     SerializableObject *gossipMsgPtrIn){
  hdr.code=codeIn;
  gossipMsgPtr=gossipMsgPtrIn;
  hdr.size=sizeof(GossipMsgHdr)+gossipMsgPtr->Size();
}

PacketType GossipPayLoad::GetPacketType() {
  PacketType packetType;

  switch (GetCode()) {
  case DATA:
    packetType=DATAPACKET;
    break;
  case ADD_NBR_REQUEST:
  case ADD_NBR_RESPONSE:
  case NBR_CONFIRM_REQUEST:
  case NBR_CONFIRM_RESPONSE:
  case PROBE_REQUEST:
  case PROBE_RESPONSE:
  case INTENT_TO_CANCEL_NBR:
  case CANCEL_NBR:
  case ESTIMATE_DELAY_REQUEST:
  case ESTIMATE_DELAY_RESPONSE:
  case POKE_REQUEST:
  case SMALL_PROBE_REQUEST:
  case JOIN_REQUEST:
  case JOIN_RESPONSE: 
    packetType=PROBE;
    break;
  case CONTROL_UPDATE:
    packetType=CONTROL;
    break;
  case POKE_DATA:
  case SMALL_PROBE_DATA:
    packetType=POKE;
    break;
  case RTT_ESTIMATOR:
    packetType=MEASUREMENT;
    break;
  default:
    MyError("Invalid Packet Type!");
    packetType = CONTROL;  /* eliminate compiler warning */
    break;
  }
  return packetType;
}

/******************
char *GossipPayLoad::GetPacketTypeString(PacketType pktType){
  switch(pktType){
  case DATAPACKET:
    return("DATAPACKET");
    break;
  case CONTROL:
    return("CONTROL");
    break;
  case PROBE:
    return("PROBE");
    break;
  case POKE:
    return("POKE");
    break;
  case MEASUREMENT:
    return("MEASUREMENT");
    break;
  }
}
*******************/

GossipPayLoad::~GossipPayLoad(){
  delete gossipMsgPtr;
}

GossipMsgType GossipPayLoad::GetCode(){
  return(hdr.code);
}

SerializableObject *GossipPayLoad::GetMsg(){
  return(gossipMsgPtr);
}

GossipPayLoad *GossipPayLoad::Duplicate(){
  GossipPayLoad *duplicate = new GossipPayLoad(hdr.code,
					       gossipMsgPtr->Duplicate());
  return(duplicate);
}

int GossipPayLoad::Size(){
  assert(hdr.size == ((int)sizeof(GossipMsgHdr))+gossipMsgPtr->Size());
  return(hdr.size);
}

void GossipPayLoad::Print(){
  cout << "\n GOSSIP PayLoad BEGIN";
  cout << "\n Code : " << (int) hdr.code;
  cout << "\n size: " << hdr.size;
  gossipMsgPtr->Print();
  cout << "\n GOSSIP PayLoad END";
}

char *GossipPayLoad::GetMsgCodeString(){
  switch(hdr.code){
  case ADD_NBR_REQUEST:
    return("ADD_NBR_REQUEST");
    break;
  case ADD_NBR_RESPONSE:
    return("ADD_NBR_RESPONSE");
    break;
  case NBR_CONFIRM_REQUEST:
    return("NBR_CONFIRM_REQUEST");
    break;
  case NBR_CONFIRM_RESPONSE:
    return("NBR_CONFIRM_RESPONSE");
    break;
  case CONTROL_UPDATE:
    return("CONTROL_UPDATE");
    break;
  case PROBE_REQUEST:
    return("PROBE_REQUEST");
    break;
  case PROBE_RESPONSE:
    return("PROBE_RESPONSE");
    break;
  case JOIN_REQUEST:
    return("JOIN_REQUEST");
    break;
  case JOIN_RESPONSE:
    return("JOIN_RESPONSE");
    break;
  case INTENT_TO_CANCEL_NBR:
    return("INTENT_TO_CANCEL_NBR");
    break;
  case CANCEL_NBR:
    return("CANCEL_NBR");
    break;
  case ESTIMATE_DELAY_REQUEST:
    return("ESTIMATE_DELAY_REQUEST");
    break;
  case ESTIMATE_DELAY_RESPONSE:
    return("ESTIMATE_DELAY_RESPONSE");
    break;
  case DATA:
    return("DATA");
    break;
  case SMALL_PROBE_REQUEST:
    return("SMALL_PROBE_REQUEST");
    break;
  case SMALL_PROBE_DATA:
    return("SMALL_PROBE_DATA");
    break;
  case POKE_REQUEST:
    return("POKE_REQUEST");
    break;
  case POKE_DATA:
    return("POKE_DATA");
    break;
  case RTT_ESTIMATOR:
    return("RTT_ESTIMATOR");
    break;
  default:
    MyError("Invalid Packet");
  }

  /* to get rid of compiler warning -- yhchu */
  MyError("should not be here");
  return "ERROR";
}


GossipPayLoad::GossipPayLoad() {
} 

void GossipPayLoad::ReadFromStream(InputSerializableStream *isPtr) {
  hdr.code=(GossipMsgType) isPtr->ReadInt();
  hdr.size=isPtr->ReadInt();
  
  switch(hdr.code){
  case ADD_NBR_REQUEST:
    gossipMsgPtr=new AddNbrRequestMsg();
    break;
  case ADD_NBR_RESPONSE:
    gossipMsgPtr = new AddNbrResponseMsg();
    break;
  case NBR_CONFIRM_REQUEST:
    gossipMsgPtr = new NbrConfirmRequestMsg();
    break;
  case NBR_CONFIRM_RESPONSE:
    gossipMsgPtr = new NbrConfirmResponseMsg();
    break;
  case CONTROL_UPDATE:
    gossipMsgPtr = new ControlUpdateMsg();
    break;
  case PROBE_REQUEST:
    gossipMsgPtr = new ProbeRequestMsg();
    break;
  case PROBE_RESPONSE:
    gossipMsgPtr = new ProbeResponseMsg();
    break;
  case JOIN_REQUEST:
    gossipMsgPtr = new JoinRequestMsg();
    break;
  case JOIN_RESPONSE:
    gossipMsgPtr = new JoinResponseMsg();
    break;
  case INTENT_TO_CANCEL_NBR:
    gossipMsgPtr = new IntentToCancelNbrMsg();
    break;
  case CANCEL_NBR:
    gossipMsgPtr = new CancelNbrMsg();
    break;
  case ESTIMATE_DELAY_REQUEST:
    gossipMsgPtr = new EstimateDelayRequestMsg();
    break;
  case ESTIMATE_DELAY_RESPONSE:
    gossipMsgPtr = new EstimateDelayResponseMsg();
    break;
  case DATA:
    gossipMsgPtr = new DataMsg();
    break;
  case POKE_REQUEST:
    gossipMsgPtr=new PokeRequestMsg();
    break;
  case POKE_DATA:
    gossipMsgPtr = new PokeDataMsg();
    break;
  case SMALL_PROBE_REQUEST:
    gossipMsgPtr=new SmallProbeRequestMsg();
    break;
  case SMALL_PROBE_DATA:
    gossipMsgPtr = new SmallProbeDataMsg();
    break;
  case RTT_ESTIMATOR:
    gossipMsgPtr = new RTTEstimationMsg();
    break;
  default:
    cerr << "Packet code = " << (int) hdr.code;
    MyError("unknown packet code");
  }

  gossipMsgPtr->ReadFromStream(isPtr);
}

void GossipPayLoad::WriteToStream(OutputSerializableStream *osPtr) {
  osPtr->WriteInt(hdr.code);
  osPtr->WriteInt(hdr.size);
  gossipMsgPtr->WriteToStream(osPtr);
}


// *********************   AddNbrResponseMsg ****************** //

UpdateRecord::UpdateRecord(){
  routingMetricPtr=NULL;
  path=NULL;
}

UpdateRecord::~UpdateRecord(){
  delete routingMetricPtr;
  delete path;
}

void UpdateRecord::SetAddr(int addrIn){
  addr=addrIn;
}

void UpdateRecord::SetLastControlSeqNum(int lastControlSeqNumIn){
  lastControlSeqNum=lastControlSeqNumIn;
}

void UpdateRecord::SetAliveFlag(int aliveFlagIn){
  aliveFlag=aliveFlagIn;
}

void UpdateRecord::SetRoutingMetric(RoutingMetric *routingMetricPtrIn){
  routingMetricPtr=routingMetricPtrIn;
}

void UpdateRecord::SetPath(MyPath *pathIn){
  path=pathIn;
}
  
int UpdateRecord::GetAddr(){
  return(addr);
}

int UpdateRecord::GetLastControlSeqNum(){
  return(lastControlSeqNum);
}

int UpdateRecord::GetAliveFlag(){
  return(aliveFlag);
}

RoutingMetric *UpdateRecord::GetRoutingMetric(){
  return(routingMetricPtr);
}

MyPath *UpdateRecord::GetPath(){
  return(path);
}

void UpdateRecord::Print() {
  cout << "\n <Addr,LCSN,AliveFlag> " 
       << GetNameByAddr(addr) << " " 
       << lastControlSeqNum << " "
       << aliveFlag << " \n";
  
  routingMetricPtr->Print();
  path->Print();
}

UpdateRecord *UpdateRecord::Duplicate(){
  UpdateRecord *duplicatePtr=new UpdateRecord();

  duplicatePtr->SetAddr(addr);
  duplicatePtr->SetLastControlSeqNum(lastControlSeqNum);
  duplicatePtr->SetAliveFlag(aliveFlag);

  RoutingMetric *dupRoutingMetricPtr=
    routingMetricPtr->Duplicate();
  duplicatePtr->SetRoutingMetric(dupRoutingMetricPtr);
  MyPath *dupPath=path->Duplicate();
  duplicatePtr->SetPath(dupPath);

  return(duplicatePtr);
}
  
void UpdateRecord::ReadFromStream(InputSerializableStream *isPtr){
  addr=isPtr->ReadInt();
  lastControlSeqNum=isPtr->ReadInt();
  aliveFlag=isPtr->ReadInt();
  routingMetricPtr=CreateNewRoutingMetric();
  routingMetricPtr->ReadFromStream(isPtr);
  path = new MyPath();
  path->ReadFromStream(isPtr);
}

void UpdateRecord::WriteToStream(OutputSerializableStream *osPtr)  {
  osPtr->WriteInt(addr);
  osPtr->WriteInt(lastControlSeqNum);
  osPtr->WriteInt(aliveFlag);
  routingMetricPtr->WriteToStream(osPtr);
  path->WriteToStream(osPtr);
}

int UpdateRecord::Size(){
  return(sizeof(addr)+ sizeof(lastControlSeqNum)+
	 sizeof(aliveFlag) +
	 routingMetricPtr->Size() + 
	 path->Size()
	 );
}

//****************************** AddNbrResponseMsg **************************//
AddNbrResponseMsg::AddNbrResponseMsg(long remoteTimeOfPrevMsgIn,  
				     long localTimeOfThisMsgIn,
				     int remoteSeqNumOfPrevMsgIn,
				     int localSeqNumOfThisMsgIn,
				     AddNbrRequestType typeIn,
				     int numRecordsIn,
				     UpdateRecord *recordPtrIn){
  
  remoteTimeOfPrevMsg=remoteTimeOfPrevMsgIn;
  localTimeOfThisMsg=localTimeOfThisMsgIn;
  remoteSeqNumOfPrevMsg=remoteSeqNumOfPrevMsgIn;
  localSeqNumOfThisMsg=localSeqNumOfThisMsgIn;
  type=typeIn;
  numRecords=numRecordsIn;
  recordPtr=recordPtrIn;
}

AddNbrResponseMsg::~AddNbrResponseMsg(){
  delete [] recordPtr;
}

UpdateRecord *AddNbrResponseMsg::GetRecordArray(){
  return(recordPtr);
}

UpdateRecord *AddNbrResponseMsg::GetRecord(int recordNum){
  return(&recordPtr[recordNum]);
}

long AddNbrResponseMsg::GetRemoteTimeOfPrevMsg(){
  return(remoteTimeOfPrevMsg);
}

long AddNbrResponseMsg::GetLocalTimeOfThisMsg(){
  return(localTimeOfThisMsg);
}

int AddNbrResponseMsg::GetRemoteSeqNumOfPrevMsg(){
  return(remoteSeqNumOfPrevMsg);
}

int AddNbrResponseMsg::GetLocalSeqNumOfThisMsg(){
  return(localSeqNumOfThisMsg);
}

int AddNbrResponseMsg::GetNumRecords(){
  return(numRecords);
}

AddNbrRequestType AddNbrResponseMsg::GetType(){
  return(type);
}


int AddNbrResponseMsg::Size(){
  int sumRecordSize = 0;

  for (int i=0; i<numRecords; i++) {
    sumRecordSize += recordPtr[i].Size();
  }

  return(sizeof(remoteTimeOfPrevMsg)+
	 sizeof(localTimeOfThisMsg)+
	 sizeof(remoteSeqNumOfPrevMsg)+
	 sizeof(localSeqNumOfThisMsg)+
	 sizeof(type) +
	 sizeof(numRecords)+ 
	 sumRecordSize);
}

AddNbrResponseMsg *AddNbrResponseMsg::Duplicate(){
  UpdateRecord *duplicateRecordPtr=new UpdateRecord[numRecords];

  for(int i=0;i < numRecords;i++){
    int addr,lastControlSeqNum,aliveFlag;
    RoutingMetric *routingMetricPtr;
    MyPath *pathPtr;

    addr=recordPtr[i].GetAddr(); 
    lastControlSeqNum=recordPtr[i].GetLastControlSeqNum();
    aliveFlag=recordPtr[i].GetAliveFlag();
    routingMetricPtr= (recordPtr[i].GetRoutingMetric())->Duplicate();
    pathPtr=(recordPtr[i].GetPath())->Duplicate();
    
    duplicateRecordPtr[i].SetAddr(addr);
    duplicateRecordPtr[i].SetLastControlSeqNum(lastControlSeqNum);
    duplicateRecordPtr[i].SetAliveFlag(aliveFlag);
    duplicateRecordPtr[i].SetRoutingMetric(routingMetricPtr);
    duplicateRecordPtr[i].SetPath(pathPtr);
  } 

  AddNbrResponseMsg *msg= new AddNbrResponseMsg(remoteTimeOfPrevMsg,
						localTimeOfThisMsg,
						remoteSeqNumOfPrevMsg,
						localSeqNumOfThisMsg,
						type,
						numRecords,
						duplicateRecordPtr);
  return(msg);
}

void AddNbrResponseMsg::Print(){
  cout << "\n AddNbrResponseMsg: BEGIN";
  cout << "\n <RTime,LTime,RSeq,LSeq,type,numRecords> = " 
       << remoteTimeOfPrevMsg << ", " << localTimeOfThisMsg << ", " 
       << remoteSeqNumOfPrevMsg << ", " << localSeqNumOfThisMsg << ", "
       << (int) type << ", " << numRecords; 
  for(int i=0;i<numRecords;i++)
    recordPtr[i].Print();
  cout << "\n AddNbrResponseMsg: END";
}

AddNbrResponseMsg::AddNbrResponseMsg(){
} 

void AddNbrResponseMsg::ReadFromStream(InputSerializableStream *isPtr){
  remoteTimeOfPrevMsg=isPtr->ReadInt();
  localTimeOfThisMsg=isPtr->ReadInt();
  remoteSeqNumOfPrevMsg=isPtr->ReadInt();
  localSeqNumOfThisMsg=isPtr->ReadInt();
  numRecords=isPtr->ReadInt();

  recordPtr = new UpdateRecord[numRecords];
  for (int i=0; i<numRecords; i++) {
    recordPtr[i].ReadFromStream(isPtr);
  }
  type=(AddNbrRequestType) isPtr->ReadInt();
}

void AddNbrResponseMsg::WriteToStream(OutputSerializableStream *osPtr) {
  osPtr->WriteInt(remoteTimeOfPrevMsg);
  osPtr->WriteInt(localTimeOfThisMsg);
  osPtr->WriteInt(remoteSeqNumOfPrevMsg);
  osPtr->WriteInt(localSeqNumOfThisMsg);
  osPtr->WriteInt(numRecords);
  for (int i =0; i<numRecords; i++) {
    recordPtr[i].WriteToStream(osPtr);
  }
  osPtr->WriteInt(type);
}


//****************************** ProbeRequestMsg **************************//
ProbeRequestMsg::ProbeRequestMsg(int seqNumIn,long timeIn){
  time=timeIn;
  seqNum=seqNumIn;
}

long ProbeRequestMsg::GetTime(){
  return(time);
}

int ProbeRequestMsg::GetSeqNum(){
  return(seqNum);
}

ProbeRequestMsg *ProbeRequestMsg::Duplicate(){
  ProbeRequestMsg *duplicate=new ProbeRequestMsg(seqNum,time);
  return(duplicate);
}

int ProbeRequestMsg::Size(){
  return(sizeof(time)+sizeof(seqNum));
}

void ProbeRequestMsg::Print(){
  cout << "\n ProbeRequestMsg: BEGIN";
  cout << "\n <Time,SeqNum> = " << time << seqNum;
  cout << "\n ProbeRequestMsg: END";
}


ProbeRequestMsg::ProbeRequestMsg() {
} 

void ProbeRequestMsg::ReadFromStream(InputSerializableStream *isPtr) {
  time=isPtr->ReadInt();
  seqNum=isPtr->ReadInt();
}

void ProbeRequestMsg::WriteToStream(OutputSerializableStream *osPtr) {
  osPtr->WriteInt(time);
  osPtr->WriteInt(seqNum);

}


//**************************** ProbeResponseRecord **************************//

ProbeResponseRecord::ProbeResponseRecord(){
  routingMetricPtr=NULL;
}

ProbeResponseRecord::~ProbeResponseRecord(){
  delete routingMetricPtr;
}

void ProbeResponseRecord::SetAddr(int addrIn){
  addr=addrIn;
}

void ProbeResponseRecord::SetLastControlSeqNum(int lastControlSeqNumIn){
  lastControlSeqNum=lastControlSeqNumIn;
}

void ProbeResponseRecord::SetRoutingMetric(RoutingMetric *routingMetricPtrIn){
  routingMetricPtr=routingMetricPtrIn;
}

int ProbeResponseRecord::GetAddr(){
  return(addr);
}

int ProbeResponseRecord::GetLastControlSeqNum(){
  return(lastControlSeqNum);
}

RoutingMetric *ProbeResponseRecord::GetRoutingMetric(){
  return(routingMetricPtr);
}

void ProbeResponseRecord::Print() {
  cout << "\n <Addr,LCSN,routingMetric> = " 
       << GetNameByAddr(addr) << " " 
       << lastControlSeqNum << " ";
  
  routingMetricPtr->Print();

}


void ProbeResponseRecord::ReadFromStream(InputSerializableStream *isPtr){
  
  addr=isPtr->ReadInt();
  lastControlSeqNum=isPtr->ReadInt();
  routingMetricPtr=CreateNewRoutingMetric();
  routingMetricPtr->ReadFromStream(isPtr);
}

void ProbeResponseRecord::WriteToStream(OutputSerializableStream *osPtr)  {
  osPtr->WriteInt(addr);
  osPtr->WriteInt(lastControlSeqNum);
  routingMetricPtr->WriteToStream(osPtr);
}

ProbeResponseRecord *ProbeResponseRecord::Duplicate(){

  ProbeResponseRecord *duplicatePtr=new ProbeResponseRecord();
  duplicatePtr->SetAddr(addr);
  duplicatePtr->SetLastControlSeqNum(lastControlSeqNum);

  RoutingMetric *dupRoutingMetricPtr=
    routingMetricPtr->Duplicate();
  duplicatePtr->SetRoutingMetric(dupRoutingMetricPtr); 

  return(duplicatePtr);
}

int ProbeResponseRecord::Size(){

  return(
	 sizeof(addr)+
	 sizeof(lastControlSeqNum)+
	 routingMetricPtr->Size() 
	 );
}

//***************************** ProbeResponseMsg ****************************//
ProbeResponseMsg::ProbeResponseMsg(int seqNumIn,
				   long remoteTimeRequestSentIn,
				   long localTimeRequestRcvdIn,
				   SaturationCodeType saturationCodeIn, 
				   int numRecordsIn,
				   ProbeResponseRecord *recordPtrIn){
  seqNum=seqNumIn;
  localTimeRequestRcvd=localTimeRequestRcvdIn;
  remoteTimeRequestSent=remoteTimeRequestSentIn;
  saturationCode=saturationCodeIn;
  numRecords=numRecordsIn;
  recordPtr=recordPtrIn;
}

ProbeResponseMsg::~ProbeResponseMsg(){
  delete [] recordPtr;
}

int ProbeResponseMsg::GetNumRecords(){
  return(numRecords);
}

int ProbeResponseMsg::GetSeqNum(){
  return(seqNum);
}

int ProbeResponseMsg::GetRemoteTimeRequestSent(){
  return(remoteTimeRequestSent);
}

int ProbeResponseMsg::GetLocalTimeRequestRcvd(){
  return(localTimeRequestRcvd);
}


void ProbeResponseMsg::Print(){
  cout << "\n ProbeResponseMsg: BEGIN";
  cout << "\n <SeqNum,time,numRecords,saturationCode> = " 
       << seqNum << " "
       << remoteTimeRequestSent << " "
       << localTimeRequestRcvd << " "
       << numRecords << " "
       << (int) saturationCode;
  for(int i=0; i < numRecords; i++)
    recordPtr[i].Print();
  cout << "\n ProbeResponseMsg: End";
}

SaturationCodeType ProbeResponseMsg::GetSaturationCode(){
  return(saturationCode);
}

ProbeResponseRecord *ProbeResponseMsg::GetRecord(int recordNum){
  return(&recordPtr[recordNum]);
}

ProbeResponseMsg *ProbeResponseMsg::Duplicate(){
  ProbeResponseRecord *duplicateRecordPtr=new ProbeResponseRecord[numRecords];

  for(int i=0;i < numRecords;i++){
    int addr,lastControlSeqNum;
    RoutingMetric *routingMetricPtr;

    addr=recordPtr[i].GetAddr(); 
    lastControlSeqNum=recordPtr[i].GetLastControlSeqNum();
    routingMetricPtr= (recordPtr[i].GetRoutingMetric())->Duplicate();
    
    duplicateRecordPtr[i].SetAddr(addr);
    duplicateRecordPtr[i].SetLastControlSeqNum(lastControlSeqNum);
    duplicateRecordPtr[i].SetRoutingMetric(routingMetricPtr);
  }

  ProbeResponseMsg *msg= new ProbeResponseMsg(seqNum,
					      remoteTimeRequestSent,
					      localTimeRequestRcvd,
					      saturationCode,
					      numRecords,
					      duplicateRecordPtr);
  return(msg);
}

int ProbeResponseMsg::Size(){
  int sumRecordSize = 0;

  for (int i=0; i<numRecords; i++) {
    sumRecordSize += recordPtr[i].Size();
  }

  int size =
    sizeof(seqNum)+
    sizeof(remoteTimeRequestSent)+
    sizeof(localTimeRequestRcvd)+
    sizeof(numRecords)+
    sizeof(saturationCode)+
    sumRecordSize;

  return(size);
}


ProbeResponseMsg::ProbeResponseMsg() {
} 

void ProbeResponseMsg::ReadFromStream(InputSerializableStream *isPtr) {
  seqNum=isPtr->ReadInt();
  remoteTimeRequestSent=isPtr->ReadInt();
  localTimeRequestRcvd=isPtr->ReadInt();
  numRecords=isPtr->ReadInt();
  saturationCode=(SaturationCodeType) isPtr->ReadInt();

  recordPtr = new ProbeResponseRecord[numRecords];
  for (int i=0; i<numRecords; i++) {
    recordPtr[i].ReadFromStream(isPtr);
  }
}

void ProbeResponseMsg::WriteToStream(OutputSerializableStream *osPtr) {
  osPtr->WriteInt(seqNum);
  osPtr->WriteInt(remoteTimeRequestSent);
  osPtr->WriteInt(localTimeRequestRcvd);
  osPtr->WriteInt(numRecords);
  osPtr->WriteInt(saturationCode);

  for (int i =0; i<numRecords; i++) {
    recordPtr[i].WriteToStream(osPtr);
  }
}


//************************ ControlUpdateMsg ********************** //

ControlUpdateMsg::ControlUpdateMsg(int seqNumIn,
				   int numRecordsIn,
				   UpdateRecord *recordPtrIn){
  seqNum=seqNumIn;
  numRecords=numRecordsIn;
  recordPtr=recordPtrIn;
}

ControlUpdateMsg::~ControlUpdateMsg(){
  delete [] recordPtr;
}

int ControlUpdateMsg::GetNumRecords(){
  return(numRecords);
}

int ControlUpdateMsg::GetSeqNum(){
  return(seqNum);
}

UpdateRecord *ControlUpdateMsg::GetRecord(int recordNum){
  return(&recordPtr[recordNum]);
}

UpdateRecord *ControlUpdateMsg::GetRecordArray(){
  return(recordPtr);
}

ControlUpdateMsg *ControlUpdateMsg::Duplicate(){
  UpdateRecord *duplicateRecordPtr=new UpdateRecord[numRecords];

  for(int i=0;i < numRecords;i++){
    int addr,lastControlSeqNum,aliveFlag;
    RoutingMetric *routingMetricPtr;
    MyPath *pathPtr;

    addr=recordPtr[i].GetAddr(); 
    lastControlSeqNum=recordPtr[i].GetLastControlSeqNum();
    aliveFlag=recordPtr[i].GetAliveFlag();
    routingMetricPtr= (recordPtr[i].GetRoutingMetric())->Duplicate();
    pathPtr=(recordPtr[i].GetPath())->Duplicate();
    
    duplicateRecordPtr[i].SetAddr(addr);
    duplicateRecordPtr[i].SetLastControlSeqNum(lastControlSeqNum);
    duplicateRecordPtr[i].SetAliveFlag(aliveFlag);
    duplicateRecordPtr[i].SetRoutingMetric(routingMetricPtr);
    duplicateRecordPtr[i].SetPath(pathPtr);
    
  }

  ControlUpdateMsg *msg= new ControlUpdateMsg(seqNum,numRecords,duplicateRecordPtr);
  return(msg);
}

int ControlUpdateMsg::Size(){
  int sumRecordSize = 0;
  
  for (int i=0; i<numRecords; i++) {
    sumRecordSize += recordPtr[i].Size();
  }

  return(sizeof(seqNum)+sizeof(numRecords)+sumRecordSize);
}

void ControlUpdateMsg::Print(){
  cout << "\n ControlUpdateMsg: BEGIN";
  cout << "\n <SeqNum,numRecords> = " << seqNum << ", " << numRecords;
  for(int i=0; i < numRecords; i++)
    recordPtr[i].Print();
  cout << "\n ControlUpdateMsg: END";
}

ControlUpdateMsg::ControlUpdateMsg() {
} 

void ControlUpdateMsg::ReadFromStream(InputSerializableStream *isPtr) {
  seqNum=isPtr->ReadInt();
  numRecords=isPtr->ReadInt();

  recordPtr = new UpdateRecord[numRecords];
  for (int i=0; i<numRecords; i++) {
    recordPtr[i].ReadFromStream(isPtr);
  }
}

void ControlUpdateMsg::WriteToStream(OutputSerializableStream *osPtr) {
  osPtr->WriteInt(seqNum);
  osPtr->WriteInt(numRecords);

  for (int i =0; i<numRecords; i++) {
    recordPtr[i].WriteToStream(osPtr);
  }
}

//******************** EstimateDelayRequestMsg **********************//
EstimateDelayRequestMsg::EstimateDelayRequestMsg(long timeIn, int lastControlSeqNumIn){
  time=timeIn;
  lastControlSeqNum=lastControlSeqNumIn;
}

EstimateDelayRequestMsg::~EstimateDelayRequestMsg(){
}

long EstimateDelayRequestMsg::GetTime(){
  return(time);
}

int EstimateDelayRequestMsg::GetLastControlSeqNum(){
  return(lastControlSeqNum);
}

EstimateDelayRequestMsg *EstimateDelayRequestMsg::Duplicate(){
  EstimateDelayRequestMsg *msg= new EstimateDelayRequestMsg(time,lastControlSeqNum);
  return(msg);
}

int EstimateDelayRequestMsg::Size(){
  return(sizeof(time)+sizeof(lastControlSeqNum));
}

void EstimateDelayRequestMsg::Print(){
  cout << "\n EstimateDelayRequestMsg: BEGIN";
  cout << "\n Time :" << time;
  cout << "\n LastControlSeqNum :" << lastControlSeqNum;
  cout << "\n EstimateDelayRequestMsg: END";
}

EstimateDelayRequestMsg::EstimateDelayRequestMsg() {
} 

void EstimateDelayRequestMsg::ReadFromStream(InputSerializableStream *isPtr) {
  time=isPtr->ReadInt();
  lastControlSeqNum=isPtr->ReadInt();

}

void EstimateDelayRequestMsg::WriteToStream(OutputSerializableStream *osPtr) {
  osPtr->WriteInt(time);
  osPtr->WriteInt(lastControlSeqNum);
}

//*********************** EstimateDelayResponseMsg ******************************//
EstimateDelayResponseMsg::EstimateDelayResponseMsg(long timeIn, 
						   int lastControlSeqNumIn){
  time=timeIn;
  lastControlSeqNum=lastControlSeqNumIn;
}

EstimateDelayResponseMsg::~EstimateDelayResponseMsg(){
}

long EstimateDelayResponseMsg::GetTime(){
  return(time);
}

int EstimateDelayResponseMsg::GetLastControlSeqNum(){
  return(lastControlSeqNum);
}

EstimateDelayResponseMsg *EstimateDelayResponseMsg::Duplicate(){
  EstimateDelayResponseMsg *msg = new EstimateDelayResponseMsg(time,lastControlSeqNum);
  return(msg);
}

int EstimateDelayResponseMsg::Size(){
  return(sizeof(time)+sizeof(lastControlSeqNum));
}

void EstimateDelayResponseMsg::Print(){
  cout << "\n EstimateDelayResponseMsg : BEGIN";
  cout << "\n time = " << time;
  cout << "\n LCSN = " << lastControlSeqNum;
  cout << "\n EstimateDelayResponseMsg: END";
}

EstimateDelayResponseMsg::EstimateDelayResponseMsg() {
} 

void EstimateDelayResponseMsg::ReadFromStream(InputSerializableStream *isPtr) {
  time=isPtr->ReadInt();
  lastControlSeqNum=isPtr->ReadInt();
}

void EstimateDelayResponseMsg::WriteToStream(OutputSerializableStream *osPtr) {
  osPtr->WriteInt(time);
  osPtr->WriteInt(lastControlSeqNum);
}

//********************* DataMsg ********************************//
DataMsg::DataMsg(int dataSizeIn, 
		 int sourceAddrIn, 
		 int seqNumIn, 
		 long timeIn, 
		 long lastHopSendRateIn,
		 int priorityIn,
		 char *dataPtrIn){
  sourceAddr=sourceAddrIn;
  dataSize=dataSizeIn;
  dataPtr=dataPtrIn;
  time=timeIn;
  lastHopSendRate=lastHopSendRateIn;
  seqNum=seqNumIn;
  priority = priorityIn;
}

DataMsg::~DataMsg(){
  delete [] dataPtr;
}

int DataMsg::Size(){
  return(sizeof(dataSize)+
	 sizeof(sourceAddr)+
	 sizeof(seqNum)+
	 sizeof(time)+
	 sizeof(lastHopSendRate)+
	 sizeof(priority)+
	 dataSize);
}

int DataMsg::GetSourceAddr(){
  return(sourceAddr);
}

int DataMsg::GetDataSize(){
  return(dataSize);
}

int DataMsg::GetSeqNum(){
  return(seqNum);
}

char *DataMsg::GetData() {
  return dataPtr;
}

long DataMsg::GetLastHopSendRate(){
  return(lastHopSendRate);
}

void DataMsg::ResetLastHopSendRate(long lastHopSendRateIn){
  lastHopSendRate=lastHopSendRateIn;
}
  
long DataMsg::GetTime(){
  return(time);
}

int DataMsg::GetPriority() {
  return priority;
}

DataMsg *DataMsg::Duplicate(){
  char *duplicateDataPtr;
  
  duplicateDataPtr=new char[dataSize];
  memcpy(duplicateDataPtr,dataPtr,dataSize);
  DataMsg *msg = new DataMsg(dataSize,
			     sourceAddr,
			     seqNum,
			     time,
			     lastHopSendRate,
			     priority,
			     duplicateDataPtr);
  return(msg);
}
  
void DataMsg::Print(){
  cout << "\n DATAMSG: BEGIN";
  cout << "\n sourceAddr seqNum dataSize " 
       << sourceAddr      << " "
       << seqNum          << " "
       << dataSize        << " "
       << lastHopSendRate << " "
       << time            << " "
       << priority        << " "; 
  cout << "\n DATAMSG: END";
}

DataMsg::DataMsg() { 
} 

void DataMsg::ReadFromStream(InputSerializableStream *isPtr) { 
  sourceAddr=isPtr->ReadInt();
  seqNum=isPtr->ReadInt();
  dataSize=isPtr->ReadInt();
  time=isPtr->ReadInt();
  lastHopSendRate=isPtr->ReadInt();
  priority = isPtr->ReadInt();

  dataPtr=new char[dataSize];
  isPtr->ReadCharArray(dataPtr,dataSize);

}

void DataMsg::WriteToStream(OutputSerializableStream *osPtr) {
  osPtr->WriteInt(sourceAddr);
  osPtr->WriteInt(seqNum);
  osPtr->WriteInt(dataSize);
  osPtr->WriteInt(time);
  osPtr->WriteInt(lastHopSendRate);
  osPtr->WriteInt(priority);

  osPtr->WriteCharArray(dataPtr, dataSize);
}


//********************* IntentToCancelNbrMsg ********************************//
int IntentToCancelNbrMsg::Size(){
  return(sizeof(leaveFlag));
}

void IntentToCancelNbrMsg::Print(){
  cout << "IntentToCancelNbrMsg: "
       << "LeaveFlag: " << leaveFlag;
}

IntentToCancelNbrMsg *IntentToCancelNbrMsg::Duplicate(){
  IntentToCancelNbrMsg *msg = new IntentToCancelNbrMsg(leaveFlag);
  return(msg);
}

IntentToCancelNbrMsg::IntentToCancelNbrMsg(){
}

IntentToCancelNbrMsg::IntentToCancelNbrMsg(int leaveFlagIn) {
  leaveFlag=leaveFlagIn;
}

long IntentToCancelNbrMsg::GetLeaveFlag(){
  return(leaveFlag);
}

void IntentToCancelNbrMsg::ReadFromStream(InputSerializableStream *isPtr) {
  leaveFlag=isPtr->ReadInt();
}

void IntentToCancelNbrMsg::WriteToStream(OutputSerializableStream *osPtr) {
  osPtr->WriteInt(leaveFlag);
}


//********************* CancelNbrMsg ********************************//
int CancelNbrMsg::Size(){
  return(sizeof(leaveFlag));
}

void CancelNbrMsg::Print(){
  cout << "CancelNbrMsg: "
       << "LeaveFlag: " << leaveFlag;
}

CancelNbrMsg *CancelNbrMsg::Duplicate(){
  CancelNbrMsg *msg = new CancelNbrMsg();
  return(msg);
}

CancelNbrMsg::CancelNbrMsg() {
  return;
}

CancelNbrMsg::CancelNbrMsg(int leaveFlagIn) {
  leaveFlag=leaveFlagIn;
}

long CancelNbrMsg::GetLeaveFlag(){
  return(leaveFlag);
}

void CancelNbrMsg::ReadFromStream(InputSerializableStream *isPtr) {
  leaveFlag=isPtr->ReadInt();
}

void CancelNbrMsg::WriteToStream(OutputSerializableStream *osPtr) {
  osPtr->WriteInt(leaveFlag);
}



//********************* PokeRequestMsg ********************************//
int PokeRequestMsg::Size(){
  return(sizeof(nonce));
}

void PokeRequestMsg::Print(){
  cout << "\n PokeRequestMsg: nonce %d" << nonce;
}

PokeRequestMsg *PokeRequestMsg::Duplicate(){
  PokeRequestMsg *msg = new PokeRequestMsg(nonce);
  return(msg);
}

PokeRequestMsg::PokeRequestMsg(int nonce) {
  this->nonce = nonce;
  return;
}

PokeRequestMsg::PokeRequestMsg() {
} 

void PokeRequestMsg::ReadFromStream(InputSerializableStream *isPtr) {
  nonce=isPtr->ReadInt();
}

int PokeRequestMsg::GetNonce() {
  return nonce;
}

void PokeRequestMsg::WriteToStream(OutputSerializableStream *osPtr) {
  osPtr->WriteInt(nonce);
}


//********************* PokeDataMsg ********************************//
PokeDataMsg::PokeDataMsg(int dataSizeIn, int sessionNonce, int seqNumIn, 
			 long timeIn, long lastHopSendRateIn, char *dataPtrIn) 
  : DataMsg(dataSizeIn, sessionNonce, seqNumIn, timeIn, lastHopSendRateIn,
	    PRIORITY_POKE, dataPtrIn) {}


PokeDataMsg::PokeDataMsg(){
} 

int PokeDataMsg::GetNonce() {
  return GetSourceAddr();
}

//********************* SmallProbeDataMsg ********************************//
SmallProbeDataMsg::SmallProbeDataMsg(int dataSizeIn, int sessionNonce, 
				     int seqNumIn, long timeIn, 
				     long lastHopSendRateIn, char *dataPtrIn) 
  : DataMsg(dataSizeIn, sessionNonce, seqNumIn, timeIn, lastHopSendRateIn,
	    PRIORITY_SMALL_PROBE, dataPtrIn) {}

SmallProbeDataMsg::SmallProbeDataMsg(){
} 

//********************* SmallProbeRequestMsg ******************************//
SmallProbeRequestMsg::SmallProbeRequestMsg(int nonce) 
  : PokeRequestMsg(nonce){}

SmallProbeRequestMsg::SmallProbeRequestMsg(){
} 

//*********************** RTTEstimationMsg ************************//
int RTTEstimationMsg::GetSourceAddr(){
  return(path[0]);
}

RTTMsgDirnType RTTEstimationMsg::GetDirn(){
  return(dirnFlg);
}

int RTTEstimationMsg::GetPathLen(){
  return(pathLen);
}

void RTTEstimationMsg::GetPath(int *pathOutPtr){
  for(int i=0; i < MyPath::MAX_PATH_LEN; i++){
    pathOutPtr[i]=path[i];
  }
}

void RTTEstimationMsg::GetForwardTime(long *forwardTimeOutPtr){
  for(int i=0; i < MyPath::MAX_PATH_LEN; i++){
    forwardTimeOutPtr[i]=forwardTime[i];
  }
}

void RTTEstimationMsg::GetReverseTime(long *reverseTimeOutPtr){
  for(int i=0; i < MyPath::MAX_PATH_LEN; i++){
    reverseTimeOutPtr[i]=reverseTime[i];
  }
}

int RTTEstimationMsg::GetPredecessor(int addr){
  int i;

  for(i=0; i < pathLen; i++){
    if(path[i] == addr){
      break;
    }
  }
  
  if(i >= MyPath::MAX_PATH_LEN){
    return(-1);
  }
  
  assert(i != 0);
  return(path[i-1]);
}

void RTTEstimationMsg::SetDirn(RTTMsgDirnType dirn){
  dirnFlg=dirn;
}

int RTTEstimationMsg::AppendForwardPath(int addr,
					long forwardTimeOfAddr){
  for(int i=0; i < pathLen; i++){
    if(addr == path[i]){
      return(0);
    }
  }
  

  if(pathLen >= MyPath::MAX_PATH_LEN){
    MyError("MyPath length exceeded");
    return(0);
  }
  
  path[pathLen]=addr;
  forwardTime[pathLen]=forwardTimeOfAddr;
  pathLen++;
  
  return(1);

}

int RTTEstimationMsg::AppendReversePath(int addr,
					int downstreamAddr,
					long time){
  int i;

  for(i=0; i < pathLen; i++){
    if(addr == path[i]){
      break;
    }
  }
  
  if(i >= pathLen){
    // I am not in forward path!!
    return(0);
  }
  
  if(i == pathLen-1){
    assert(downstreamAddr == -1);
  }
  
  if(downstreamAddr != path[i+1]){
    // did not get the packet from who I had sent to on the forward path
    return(0);
  }
  
  reverseTime[i]=time;
  return(1);
}

int RTTEstimationMsg::Size(){
  return(sizeof(dirnFlg) + 
	 sizeof(pathLen) + 
	 sizeof(path) + 
	 sizeof(forwardTime) + 
	 sizeof(reverseTime)
	 );
}


RTTEstimationMsg *RTTEstimationMsg::Duplicate(){
  
  RTTEstimationMsg *duplicatePtr =  new RTTEstimationMsg(dirnFlg,
							 pathLen,
							 path,
							 forwardTime,
							 reverseTime);
  return(duplicatePtr);
}



void RTTEstimationMsg::Print(){
  cout << "\n RTTEstimationMSG: BEGIN " << GetCurrTime();
  cout << "\nsourceAddr dirnFlg pathLen " 
       << GetNameByAddr(path[0]) << " "
       << (int) dirnFlg    << " "
       << pathLen    << " ";
  
  for(int i=0; i < pathLen; i++){
    cout << "\n" << GetNameByAddr(path[i]) << " " 
	 <<    forwardTime[i] << " "
	 <<    reverseTime[i] << " ";
  }
  
  cout << "\n RTTEstimationMSG: END";
}

RTTEstimationMsg::RTTEstimationMsg(RTTMsgDirnType dirnFlgIn,
				   int pathLenIn,
				   int *pathIn,
				   long *forwardTimeIn,
				   long *reverseTimeIn){

  dirnFlg=dirnFlgIn;
  pathLen=pathLenIn;
  
 for(int i=0; i < MyPath::MAX_PATH_LEN; i++){
    path[i]=pathIn[i];
    forwardTime[i]=forwardTimeIn[i];
    reverseTime[i]=reverseTimeIn[i];
  }
}

RTTEstimationMsg::RTTEstimationMsg(int sourceAddrIn, 
				   int sourceSendTime){
  dirnFlg=FORWARD;
  pathLen=1;
  
  for(int i=0; i < MyPath::MAX_PATH_LEN; i++){
    path[i]=RTT_ESTIMATION_INIT;
    forwardTime[i]=RTT_ESTIMATION_INIT;
    reverseTime[i]=RTT_ESTIMATION_INIT;
  }
  
  
  path[0]=sourceAddrIn;
  forwardTime[0]=sourceSendTime;
}

RTTEstimationMsg::RTTEstimationMsg(){
} 

void RTTEstimationMsg::ReadFromStream(InputSerializableStream *isPtr){
  dirnFlg=(RTTMsgDirnType) isPtr->ReadInt();
  pathLen=isPtr->ReadInt();

  for(int i=0; i < MyPath::MAX_PATH_LEN; i++){
    path[i]=isPtr->ReadInt();
  }
  
  for(int i=0; i < MyPath::MAX_PATH_LEN; i++){
    forwardTime[i]=isPtr->ReadInt();
  }
  
  for(int i=0; i < MyPath::MAX_PATH_LEN; i++){
    reverseTime[i]=isPtr->ReadInt();
  }
}

void RTTEstimationMsg::WriteToStream(OutputSerializableStream *osPtr){

  osPtr->WriteInt(dirnFlg);
  osPtr->WriteInt(pathLen);

  for(int i=0; i < MyPath::MAX_PATH_LEN; i++){
    osPtr->WriteInt(path[i]);
  }
  
  for(int i=0; i < MyPath::MAX_PATH_LEN; i++){
    osPtr->WriteInt(forwardTime[i]);
  }
  
  for(int i=0; i < MyPath::MAX_PATH_LEN; i++){
    osPtr->WriteInt(reverseTime[i]);
  }
}


/******************* ControlUpdateDemandMsg ******************************
int ControlUpdateDemandMsg::Size(){
  return(0);
}

void ControlUpdateDemandMsg::Print(){
}

ControlUpdateDemandMsg *ControlUpdateDemandMsg::Duplicate(){
  ControlUpdateDemandMsg *msg = new ControlUpdateDemandMsg();
  return(msg);
}

ControlUpdateDemandMsg::ControlUpdateDemandMsg() {
  return;
}

void ControlUpdateDemandMsg::ReadFromStream(InputSerializableStream *isPtr) {
  return;
}

void ControlUpdateDemandMsg::WriteToStream(OutputSerializableStream *osPtr) {
  return;
}
*************************************************************/
